
      module plevs
!---------------------------------------------------------------------
! 	... hybrid level definitions: p = a*p0 + b*ps
!           interfaces   p(k) = hyai(k)*ps0 + hybi(k)*ps
!           midpoints    p(k) = hyam(k)*ps0 + hybm(k)*ps
!---------------------------------------------------------------------

      use mo_grid, only : plev, plevp

      implicit none

      save

      real :: &
        ps0        ! base state sfc pressure for level definitions
      real, dimension(plevp) :: &
        hyai, &    ! ps0 component of hybrid coordinate - interfaces
        hybi       ! ps component of hybrid coordinate - interfaces
      real, dimension(plev) :: &
        hyam, &    ! ps0 component of hybrid coordinate - midpoints
        hybm, &    ! ps component of hybrid coordinate - midpoints
        dhya, &    ! dhya(k) = hyai(k+1) - hyai(k)
        dhyb       ! dhyb(k) = hybi(k+1) - hybi(k)

      contains

      subroutine iniplevs( xhyai, xhyam, xhybi, xhybm )
!-----------------------------------------------------------------------
! 	... initialize module plevs
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
! 	... dummy args
!-----------------------------------------------------------------------
      real, dimension(plevp), intent(in) :: &
        xhyai, &       ! p0 component of hybrid coordinate - interfaces
        xhybi          ! ps component of hybrid coordinate - interfaces
      real, dimension(plev), intent(in) :: &
        xhyam, &       ! p0 component of hybrid coordinate - midpoints
        xhybm          ! ps component of hybrid coordinate - midpoints

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      ps0 = 1.e5                ! base state surface pressure (pascals)

      hyai(:plev) = xhyai(:plev)
      hybi(:plev) = xhybi(:plev)
      hyam(:plev) = xhyam(:plev)
      hybm(:plev) = xhybm(:plev)
      hyai(plevp) = xhyai(plevp)
      hybi(plevp) = xhybi(plevp)

      dhya(:plev) = hyai(2:plevp) - hyai(:plev)
      dhyb(:plev) = hybi(2:plevp) - hybi(:plev)

      end subroutine iniplevs

      subroutine plevs0( ps, pint, pmid, pdel, plonl )
!-----------------------------------------------------------------------
! 	... define the pressures of the interfaces and midpoints from the
!           coordinate definitions and the surface pressure.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      real, intent(in)  :: &
           ps(plonl)          ! surface pressure (pascals)
      real, intent(out) :: &
           pint(plonl,plevp)  ! pressure at model interfaces
      real, dimension(plonl,plev), intent(out) :: &
           pmid, &            ! pressure at model levels
           pdel               ! layer thickness (pint(k+1) - pint(k))

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: &
           k               ! level indices

!-----------------------------------------------------------------------
! 	... set interface pressures
!-----------------------------------------------------------------------
      do k = 1,plevp
         pint(:,k) = hyai(k)*ps0 + hybi(k)*ps(:)
      end do

!-----------------------------------------------------------------------
! 	... set midpoint pressures and layer thicknesses
!-----------------------------------------------------------------------
      do k = 1,plev
         pmid(:,k) = hyam(k)*ps0 + hybm(k)*ps(:)
         pdel(:,k) = pint(:,k+1) - pint(:,k)
      end do

      end subroutine plevs0

      subroutine pdelab( ps, pdela, pdelb, plonl )
!-----------------------------------------------------------------------
! 	... contribution to pressure differences across layers
!           from the a and b components of the hybrid coordinate.
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	   plonl
      real, intent(in)  :: &
        ps(plonl)          ! surface pressure (pascals)
      real, dimension(plonl,plev), intent(out) :: &
        pdela, &            ! pressure diff due to a component
        pdelb               ! pressure diff due to b component

!-----------------------------------------------------------------------
! 	... local varibles
!-----------------------------------------------------------------------
      integer :: &
           k             ! level indices

      do k = 1,plev
         pdela(:,k) = dhya(k)*ps0
         pdelb(:,k) = dhyb(k)*ps(:)
      end do

      end subroutine pdelab

      end module plevs
