
      module mo_bvoc
!---------------------------------------------------------------------
!       ... Interactive calculation of BVOC emissions following Alex
!           Guenther's algorithms
!
! Jean-Francois Lamarque, March 2004
!
! Gabriele Pfister, October 2004
! Gabriele Pfister: updated to Guenther et al., ACPD 2006 (August 2007)
! April 2009 (GGP, LKE): update to MEGAN v2.1 emission maps
!                        incorporate mo_mterps
!---------------------------------------------------------------------

      use mo_grid, only : plon

      implicit none


      private
      public :: interp_map_inti
      public :: bvoc_inti
      public :: interp_map_iso
      public :: interp_map_mterps
      public :: megan_iso_emis
      public :: megan_mterps_emis
      public :: bvoc_rdrst
      public :: bvoc_wrrst
      public :: megan_cnt, megan_species

      save
      integer, parameter :: veg_types = 5
      integer, parameter :: veg_ext   = 20
      character(len=3)   :: varname(veg_types) = (/ 'ntr', 'btr', 'shr', 'grs', 'crp' /)
!-------------------------------------------------------------------------------------
!	... Tdaily_clim as recommended by Alex (personal communication)
!-------------------------------------------------------------------------------------
      real, parameter    :: Tdaily_clim = 297.   

      type area_map
         integer :: count
         integer, pointer :: src_xndx(:)
         integer, pointer :: src_yndx(:)
         real             :: total_src_area
         real, pointer    :: src_area(:)
      end type area_map

#ifdef SW_DEBUG
      integer, parameter   :: d_lon = 46
      integer, parameter   :: d_lat = 48
#endif
      integer              :: megan_cnt
      integer, allocatable :: mapping_ext(:)
      real, allocatable    :: maps_megan(:,:,:,:)
      real, allocatable    :: maps_clm(:,:,:,:)
      character(len=8), allocatable :: megan_species(:)

      type(area_map), allocatable  :: model2veg_map(:,:) 

      contains

      subroutine bvoc_inti( plonl, platl, pplon, nlon_veg, nlat_veg, &
                            npft_veg, lon_veg, lon_veg_edge, lat_veg, lat_veg_edge, &
                            vegetation_map, landmask, monthly_lai_clm, ts_avg, fsds_avg, ncdate )
!-------------------------------------------------------------------------------------
! 	... intialize interactive bvoc emissions
!-------------------------------------------------------------------------------------

      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_double_precision, mpi_comm_comp
#endif
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : emis_flsp
      use chem_mods,     only : srfems_lst, srfems_cnt, megan_map
      use netcdf

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: nlon_veg
      integer, intent(in) :: nlat_veg
      integer, intent(in) :: npft_veg
      integer, intent(in) :: ncdate
      real, intent(in)    :: lon_veg(nlon_veg)
      real, intent(in)    :: lon_veg_edge(nlon_veg+1)
      real, intent(in)    :: lat_veg(nlat_veg)
      real, intent(in)    :: lat_veg_edge(nlat_veg+1)
      real, intent(in)    :: vegetation_map(nlon_veg,nlat_veg,npft_veg)
      real, intent(in)    :: landmask(nlon_veg,nlat_veg)
      real, intent(in)    :: monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12)
      real, intent(in)    :: ts_avg(plonl,platl,pplon)
      real, intent(in)    :: fsds_avg(plonl,platl,pplon)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer :: i, j, ii, jj
      integer :: m, n, l, id
      integer :: astat, iret
      integer :: i1, j1
      integer :: ncid
      integer :: dimid, vid
      integer :: nlon_data
      integer :: nlat_data
      integer :: month
      integer :: day

      character(len=80) :: filename
      character(len=80) :: err_msg

!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
      megan_cnt = count( megan_map(:) )
      if( megan_cnt == 0 ) then
         return
      end if
      allocate( megan_species(megan_cnt),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'bvoc_inti: failed to allocate megan_species; error = ',astat
         call endrun
      end if
      allocate( maps_megan(plonl,platl,pplon,megan_cnt),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'bvoc_inti: failed to allocate maps_megan; error = ',astat
         call endrun
      end if
!-------------------------------------------------------------------------------------
! 	... allocate and read isoprene potential maps from Alex Guenther
!-------------------------------------------------------------------------------------
      allocate( maps_clm(nlon_veg,nlat_veg,veg_types,megan_cnt),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'bvoc_inti: failed to allocate maps_clm; error = ',astat
         call endrun
      end if

      megan_species(:) = pack( srfems_lst, megan_map )

master_only : &
      if( masternode ) then
species_loop : &
         do m = 1,megan_cnt
!-------------------------------------------------------------------------------------
! 	... open bvoc netcdf file
!-------------------------------------------------------------------------------------
            filename = 'megan.' // trim(megan_species(m)) // '.nc'
            ncid = open_netcdf_file( filename, trim(emis_flsp%local_path), &
                                        trim(emis_flsp%remote_path), masteronly=.true. )
!---------------------------------------------------------------------------
! 	... get and check the dimensions
!---------------------------------------------------------------------------
            call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid ), 'bvoc_inti: dimension nlon not found' )
            call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlon_data ), 'bvoc_inti: failed to read nlon' )
            call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), 'bvoc_inti: dimension nlat not found' )
            call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat_data ), 'bvoc_inti: failed to read nlat' )
            if( nlon_data /= nlon_veg .or. nlat_data /= nlat_veg ) then
               write(*,*) 'bvoc_inti: ',trim(filename),' dimensions :'
               write(*,*) '           (',nlon_data,' x ',nlat_data,')'
               write(*,*) '           must match vegetation dataset dimensions'
               write(*,*) '           (',nlon_veg,' x ',nlat_veg,')'
               call endrun
            end if
!-------------------------------------------------------------------------------------
! 	... read data
!-------------------------------------------------------------------------------------
            do n = 1,veg_types
               err_msg = 'bvoc_inti: failed to find ' // varname(n) // ' id'
               call handle_ncerr( nf_inq_varid( ncid, varname(n), vid ), trim(err_msg) )
               err_msg = 'bvoc_inti: failed to read ' // varname(n)
               call handle_ncerr( nf_get_var_double( ncid, vid, maps_clm(:,:,n,m) ), trim(err_msg) )
            end do
!---------------------------------------------------------------------------
! 	... close netcdf file
!---------------------------------------------------------------------------
            err_msg = 'bvoc_inti: error closing ' // trim(filename)
            call handle_ncerr( nf_close( ncid ), trim(err_msg) )
         end do species_loop
      end if master_only

#ifdef USE_MPI
!------------------------------------------------------------------------------
!       ... bcast arrays
!------------------------------------------------------------------------------
      call mpi_bcast( maps_clm, nlon_veg*nlat_veg*veg_types*megan_cnt, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'bvoc_inti: failed to bcast maps_clm; error = ',iret
         call endrun
      end if
#endif

!---------------------------------------------------------------------------
! 	... regrid to model grid
!---------------------------------------------------------------------------
      month = mod( ncdate,10000 )/100
      day   = mod( ncdate,100 )
      write(*,*) ' '
      write(*,*) '------------------------------'
      write(*,*) 'bvoc_inti: month,day = ',month,day
      write(*,*) '------------------------------'
      write(*,*) ' '

species_loop_1 : &
      do m = 1,megan_cnt
         select case( megan_species(m) )
         case( 'ISOP' )
            call interp_map_iso( nlon_veg, nlat_veg, npft_veg, landmask, vegetation_map, &
                                 monthly_lai_clm, ts_avg, fsds_avg, month, plonl, &
                                 platl, pplon, m )
         case( 'C10H16' )
            call interp_map_mterps( nlon_veg, nlat_veg, npft_veg, landmask, vegetation_map, &
                                    monthly_lai_clm, month, plonl, platl, pplon, m )
         end select
      end do species_loop_1

      end subroutine bvoc_inti

      subroutine interp_map_inti( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                                  lon_veg, lon_veg_edge, plonl, platl, pplon )
!---------------------------------------------------------------------------
! 	... initialize interpolation
!---------------------------------------------------------------------------

      use mo_grid, only      : plon, plat
      use mo_constants, only : r2d, phi, lam
      use m_adv, only        : has_npole
      use mo_mpi, only       : base_lat

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)      :: plonl
      integer, intent(in)      :: platl
      integer, intent(in)      :: pplon
      integer, intent(in)      :: nlon_veg, nlat_veg, npft_veg
      real, intent(in)         :: lon_veg(nlon_veg)
      real, intent(in)         :: lon_veg_edge(nlon_veg+1)
      real, intent(in)         :: lat_veg(nlat_veg)
      real, intent(in)         :: lat_veg_edge(nlat_veg+1)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer                  :: i, j, ii, jj
      integer                  :: astat
      integer                  :: cnt
      integer, allocatable     :: ind_lon(:)
      integer, allocatable     :: ind_lat(:)
      real                     :: wrk_total_area
      real                     :: wrk_area
      real                     :: lon_edge(plon+1)
      real                     :: lat_edge(platl+1)
      real                     :: lat1, lat2
      real                     :: y1, y2
      real                     :: lon1, lon2
      real                     :: x1, x2
      real                     :: dx, dy
      real                     :: hdx, hdy
      real                     :: lon_veg_edge_ext(-veg_ext:nlon_veg+veg_ext)
      logical                  :: found

!--------------------------------------------------------------
! 	... allocate local and global arrays
!--------------------------------------------------------------
      allocate( ind_lon(plon+1), ind_lat(platl+1), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'interp_map_inti: failed to allocate ind_lon, ind_lat arrays; error = ',astat
	 call endrun
      end if
      allocate( mapping_ext(-veg_ext:nlon_veg+veg_ext), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'bvoc_inti: failed to allocate mapping_ext array; error = ',astat
	 call endrun
      end if
      allocate( model2veg_map(plon,platl), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'bvoc_inti: failed to allocate vegatation to model map; error = ',astat
	 call endrun
      end if
!-------------------------------------------------------------
! 	... set model horizontal grid edges
!-------------------------------------------------------------
      dx  = (lam(2) - lam(1))*r2d
      hdx = .5 * dx
      do i = 1,plon
         lon_edge(i) = lam(i)*r2d - hdx
      end do
      lon_edge(plon+1) = lon_edge(plon) + dx
      dy  = (phi(2) - phi(1))*r2d
      hdy = .5 * dy
      if( .not. has_npole ) then
        do j = 1,platl+1
           lat_edge(j) = phi(j+base_lat) * r2d - hdy
        end do
      else
        do j = 1,platl
           lat_edge(j) = phi(j+base_lat) * r2d - hdy
        end do
        lat_edge(platl+1) = lat_edge(platl) + dy
      end if
      do j = 1,platl+1
         lat_edge(j) = min( lat_edge(j), 90. )
         lat_edge(j) = max( lat_edge(j),-90. )
      end do

!-------------------------------------------------------------
! 	... wrap around the longitudes for vegetation grid
!-------------------------------------------------------------
      do i = -veg_ext,0
        lon_veg_edge_ext(i) = lon_veg_edge(nlon_veg+i) - 360.
        mapping_ext(i)      =  nlon_veg + i
      end do
      do i = 1,nlon_veg
        lon_veg_edge_ext(i) = lon_veg_edge(i)
        mapping_ext(i)      = i
      end do
      do i = nlon_veg+1,nlon_veg+veg_ext
        lon_veg_edge_ext(i) = lon_veg_edge(i-nlon_veg) + 360.
        mapping_ext(i)      = i - nlon_veg
      end do
#ifdef DEBUG
      write(*,*) 'interp_map_inti : lon_edge ',lon_edge
      write(*,*) 'interp_map_inti : lat_edge ',lat_edge
      write(*,*) 'interp_map_inti : mapping_ext ',mapping_ext
#endif
      do i = 1,plon+1
        lon1  = lon_edge(i) 
        found = .false.
        do ii = -veg_ext,nlon_veg+veg_ext-1
          dx = lon_veg_edge_ext(ii) - lon1
          dy = lon_veg_edge_ext(ii+1) - lon1
          if( dx*dy <= 0. ) then
            ind_lon(i) = ii
            found = .true.
            exit
          end if
        end do
        if( .not. found ) then
           write(*,*) ' '
           write(*,*) 'interp_map_inti: failed to find interval for j,lon1 = ',i,lon1
           write(*,*) 'interp_map_inti: lon_veg_edge_ext(',-veg_ext,':',-veg_ext+1,') = ', &
                       lon_veg_edge_ext(-veg_ext:-veg_ext+1)
           write(*,*) 'interp_map_inti: lon_veg_edge_ext(', &
                       nlon_veg+veg_ext-1,':',nlon_veg+veg_ext,') = ', &
                       lon_veg_edge_ext(nlon_veg+veg_ext-1:nlon_veg+veg_ext)
           write(*,*) ' '
           call endrun
        end if
      end do

      do j = 1,platl+1
        lat1  = lat_edge(j)
        found = .false.
        do jj = 1,nlat_veg
          dx = lat_veg_edge(jj) - lat1
          dy = lat_veg_edge(jj+1) - lat1
          if( dx*dy <= 0. ) then
            ind_lat(j) = jj
            found = .true.
            exit
          end if
        end do
        if( .not. found ) then
           write(*,*) ' '
           write(*,*) 'interp_map_inti: failed to find interval for j,lat1 = ',j,lat1
           write(*,*) 'interp_map_inti: lat_veg_edge(1:2) = ',lat_veg_edge(1:2)
           write(*,*) 'interp_map_inti: lat_veg_edge(',nlat_veg,':',nlat_veg+1,') = ',lat_veg_edge(nlat_veg:nlat_veg+1)
           write(*,*) ' '
           call endrun
        end if
      end do

#ifdef DEBUG
      write(*,*) 'interp_map_inti : ind_lon ',ind_lon
      write(*,*) 'interp_map_inti : ind_lat ',ind_lat
#endif

!------------------------------------------------------------------
! 	... allocate model2veg_map components
!------------------------------------------------------------------
      do j = 1,platl
        do i = 1,plon
           cnt = (ind_lat(j+1) - ind_lat(j) + 1)*(ind_lon(i+1) - ind_lon(i) + 1)
           allocate( model2veg_map(i,j)%src_area(cnt), &
                     model2veg_map(i,j)%src_xndx(cnt), &
                     model2veg_map(i,j)%src_yndx(cnt), stat=astat )
           if( astat /= 0 ) then
              write(*,*) 'interp_map_inti: failed to allocate model2veg components; error = ',astat
              call endrun
           end if 
           model2veg_map(i,j)%count = cnt
        end do
     end do

lat_loop : &
      do j = 1,platl
lon_loop : &
        do i = 1,plon
          wrk_total_area  = 0.
          cnt             = 0
          do jj = ind_lat(j),ind_lat(j+1)
            y1 = max( lat_edge(j),lat_veg_edge(jj) )
            y2 = min( lat_edge(j+1),lat_veg_edge(jj+1) ) 
            dy = (y2 - y1)/(lat_veg_edge(jj+1) - lat_veg_edge(jj))
            do ii = ind_lon(i),ind_lon(i+1)
              cnt = cnt + 1
              x1 = max( lon_edge(i),lon_veg_edge_ext(ii) )
              x2 = min( lon_edge(i+1),lon_veg_edge_ext(ii+1) ) 
              dx = (x2 - x1)/(lon_veg_edge_ext(ii+1) - lon_veg_edge_ext(ii))
              wrk_area = dx * dy
              model2veg_map(i,j)%src_area(cnt) = wrk_area
              model2veg_map(i,j)%src_xndx(cnt) = ii
              model2veg_map(i,j)%src_yndx(cnt) = jj
              wrk_total_area = wrk_total_area + wrk_area
            end do
          end do
          model2veg_map(i,j)%total_src_area = wrk_total_area
        end do lon_loop
      end do lat_loop

      deallocate( ind_lon, ind_lat )

      end subroutine interp_map_inti

      subroutine interp_map_iso( nlon_veg, nlat_veg, npft_veg, landmask, vegetation_map, &
                                 monthly_lai_clm, ts_avg, fsds_avg, month, plonl, &
                                 platl, pplon, ndx )
!------------------------------------------------------------------
!	... interpolate isoprene emissions from the "land" grid
!           to the model grid
!------------------------------------------------------------------

      use mo_grid, only      : plon, plat
      use mo_constants, only : r2d, phi, lam
      use m_adv, only        : has_npole
      use mo_mpi, only       : base_lat
      use mo_control, only   : dyn_has_ts_avg

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)      :: plonl
      integer, intent(in)      :: platl
      integer, intent(in)      :: pplon
      integer, intent(in)      :: nlon_veg, nlat_veg, npft_veg
      integer, intent(in)      :: month
      integer, intent(in)      :: ndx
      real, intent(in)         :: ts_avg(plonl,platl,pplon)
      real, intent(in)         :: fsds_avg(plonl,platl,pplon)
      real, intent(in)         :: landmask(nlon_veg,nlat_veg)
      real, intent(in)         :: vegetation_map (nlon_veg,nlat_veg,npft_veg)
      real, intent(in)         :: monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer                  :: i, j, ii, jj, i_ndx, n
      integer                  :: ip, i1
      integer                  :: nl, nu
      integer                  :: cell
      integer                  :: pft_li(veg_types)
      integer                  :: pft_lu(veg_types)
      real                     :: wrk_area
      real                     :: total_iso
      real                     :: ts_wrk
      real                     :: fsds_wrk
      real                     :: work_iso(veg_types)
      real                     :: wrk_clm(veg_types)
      real                     :: lai_fac(npft_veg)
      real                     :: veg_wrk(npft_veg)
      real                     :: lai_wrk(npft_veg,12)
      logical                  :: age_wrk(npft_veg)

!-----------------------------------------------------------------
!       ... initializations
!-----------------------------------------------------------------
      age_wrk(:) = .true.
      age_wrk((/2,3,5,6,10/)) = .false.
      pft_li(:veg_types)   = (/ 2, 5, 10, 13, 16 /)
      pft_lu(:veg_types-1) = (/ 4, 9, 12, 15 /)
      pft_lu(veg_types)    = npft_veg

lat_loop : &
      do j = 1,platl
lon_loop : &
        do i = 1,plon
          total_iso = 0.
          ip = (i - 1)/plonl + 1
          i1 = mod( i - 1,plonl ) + 1
          if( dyn_has_ts_avg ) then
             ts_wrk   = ts_avg(i1,j,ip)
             fsds_wrk = fsds_avg(i1,j,ip)
          end if
cell_loop : &
          do cell = 1,model2veg_map(i,j)%count
              ii = model2veg_map(i,j)%src_xndx(cell)
              jj = model2veg_map(i,j)%src_yndx(cell)
              wrk_area = model2veg_map(i,j)%src_area(cell)
              i_ndx    = mapping_ext(ii)
!-----------------------------------------------------------------
!       ... no emissions for ocean grid point 
!-----------------------------------------------------------------
              if( nint(landmask(i_ndx,jj)) == 0 ) then
                 cycle cell_loop
              end if
              wrk_clm(:)   = maps_clm(i_ndx,jj,:,ndx)
              veg_wrk(:)   = vegetation_map(i_ndx,jj,:)
              lai_wrk(:,:) = monthly_lai_clm(i_ndx,jj,:,:)
              lai_fac(:)   = fac_lai( npft_veg, lai_wrk, ts_wrk, fsds_wrk, month, age_wrk )
!------------------------------------------------------------------------------------
!       ... emissions from :
!           (1) fine leaf trees (vegtypes = 'ntr')
!           (2) broadleaf (vegtypes = 'btr')
!           (3) shrubs (vegtypes = 'shr')
!           (4) grass (vegtypes = 'grs')
!           (5) crops (vegtypes = 'crp')
!-------------------------------------------------------------------------------------
              do n = 1,veg_types
                 nl = pft_li(n)
                 nu = pft_lu(n)
                 work_iso(n)  = dot_product( lai_fac(nl:nu),veg_wrk(nl:nu) ) * wrk_clm(n)
              end do
              total_iso  = total_iso + sum( work_iso ) * wrk_area
          end do cell_loop
!-------------------------------------------------------------------------------------
! 	... divide by total grid area
!-------------------------------------------------------------------------------------
          maps_megan(i1,j,ip,ndx) = total_iso/(model2veg_map(i,j)%total_src_area + 1.e-30)
#ifdef SW_DEBUG
          if( (base_lat+j) == d_lat .and. i == d_lon ) then
             write(*,*) ' '
             write(*,*) 'interp_map_iso: interpolated iso = ',total_iso
             write(*,*) ' '
          end if
#endif
        end do lon_loop
      end do lat_loop

      end subroutine interp_map_iso

      function fac_lai( npft, clai, ts_avg, fsds_avg, month, doage )
!-------------------------------------------------------------------------------------
!       Monthly exchange Ratio according to MEGAN (Version 1.0)
!	input LAI expected to cover one year, monthly mean
!-------------------------------------------------------------------------------------

      use mo_control, only   : dyn_has_ts_avg

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: npft
      integer, intent(in) :: month
      real, intent(in)    :: ts_avg
      real, intent(in)    :: fsds_avg
      real, intent(in)    :: clai(:,:)
      logical, intent(in) :: doage(:)
!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
! ggp: equations 18 and 19 from Guenther et al. include a dependence of ti and tm on temperature
! of preceding timestep (i.e. month here); not considered here. 
! ggp/lamar: instead of using a constant ti and tm, it can be calculated based on information 
! of monthly average temperature
!-------------------------------------------------------------------------------------
      integer, parameter :: t(12) = (/ 31,31,28,31,30,31,30,31,31,30,31,30 /)

      integer :: i, n
      integer :: mnthm1
      real    :: tg
      real    :: x
      real    :: wrk
      real    :: gamma
      real    :: ti, tm
      real    :: Fnew
      real    :: Fgro
      real    :: Fmat
      real    :: Fsen
      real    :: lai_n, lai_p

!-------------------------------------------------------------------------------------
! 	... function declarations
!-------------------------------------------------------------------------------------
      real    :: fac_lai(npft)

      if( month > 1 ) then
         mnthm1 = month - 1
      else
         mnthm1 = 12
      end if

!-------------------------------------------------------------------------------------
! 	... calculations following equations 17&18 in Guenther et al. [2006]
!-------------------------------------------------------------------------------------
      if( dyn_has_ts_avg ) then
         if( ts_avg <= 303. ) then
            ti = 5. + 0.7*(300. - ts_avg)
         else
	    ti = 2.9
         end if
      else
         ti = 5. + 0.7*(300. - Tdaily_clim)
      end if
      tm = 2.3*ti

      fac_lai(1) = 0.
plant_type_loop : &
      do n = 2,npft
         if( doage(n) ) then 
            Fnew = 0.
            Fgro = 0.
            Fmat = 0.
            Fsen = 0.
            lai_n = clai(n,month)
            lai_p = clai(n,mnthm1)
            if( lai_n == lai_p ) then
               Fmat = 0.8
               Fsen = 0.1
               Fgro = 0.1
            else if( lai_p > lai_n ) then 
               Fsen = (lai_p - lai_n) / lai_p
               Fmat = 1. - Fsen
            else if( lai_p < lai_n ) then 
               Fsen = 0.
               x    = lai_p/lai_n
               wrk  = 1. - x
               if( t(month) <= tm ) then 
                  Fmat =  x
               else
                  Fmat = x + (((t(month) - tm)/t(month) ) * wrk)
               end if
               if( t(month) <= ti ) then 
                  Fnew = wrk
                  Fgro = 1. - (Fnew + Fmat)
               else 
                  Fnew = (ti/t(month)) * wrk
                  Fgro = 1. - (Fnew + Fmat)
               end if
            end if
!-------------------------------------------------------------------------------------
!	... equations 15 and 16 in Guenther et al. [2006]
!-------------------------------------------------------------------------------------
            gamma   = .05*Fnew + .6*Fgro + 1.125*Fmat + Fsen
         else
            gamma   = 1.
         end if
         fac_lai(n) = gamma * .49 * clai(n,month) / sqrt( 1. + 0.2 * clai(n,month)*clai(n,month) )
      end do plant_type_loop

#ifdef FAC_LAI_DIAGS
      write(*,*) 'ggp ti, tm, fac_lai ', ti, tm, fac_lai
#endif

      end function fac_lai

      subroutine megan_iso_emis( lat, ip, calday, ts, fsds, &
                                 sflx, ts_avg, fsds_avg, plonl, zen_angle )
!-------------------------------------------------------------------------------------
!	... biogenic voc isoprene emissions
!-------------------------------------------------------------------------------------

      use mo_grid, only      : pcnst
      use chem_mods, only    : adv_mass
      use mo_chem_utls, only : get_spc_ndx
      use mo_control, only   : dyn_has_ts_avg
      use mo_constants, only : twopi
      use mo_constants, only : rad_to_degree => r2d
      use mo_constants, only : degree_to_rad => d2r
      use mo_mpi, only       : base_lat

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: lat              ! latitude index
      integer, intent(in) :: ip               ! longitude tile index
      integer, intent(in) :: plonl
      real, intent(in)    :: calday           ! julian day + fraction (greenwich)
      real, intent(in)    :: ts(plonl)        ! surface temperature
      real, intent(in)    :: fsds(plonl)      ! surface direct radiation (w/m^2)
      real, intent(in)    :: ts_avg(plonl)    ! averaged surface temperature
      real, intent(in)    :: fsds_avg(plonl)  ! averaged surface direct radiation (w/m^2)
      real, intent(in)    :: zen_angle(plonl) ! solar zenith angle in radians
      real, intent(inout) :: sflx(plonl)      ! surface flux for advected species

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer, parameter :: days_in_month(12) = (/ 31,28,31,30,31,30,31,31,30,31,30,31 /)
      real, parameter :: ctm1   =  80.
      real, parameter :: ctm2   = 200.
      real, parameter :: const0 = 4.76 *.5            ! what is this?
      real, parameter :: amufac = 1.65979e-23         ! 1.e4* kg / amu

      integer :: month, total_days
      integer :: i, n, ndx, file
      integer :: isop_ndx
      real    :: ppfd, x, Eopt, sol_angle_deg, sol_angle_rad, Ptoa, phi
      real    :: Topt, zen_angle_deg
      real    :: total_area, total_emis, fac_par, fac_tmp
      real    :: Tdaily, Pdaily	
      real    :: t_diff
      real    :: biso_flx(plonl)

      do ndx = 1,megan_cnt
         if( megan_species(ndx) == 'ISOP' ) then 
            isop_ndx = get_spc_ndx( 'ISOP' )
            exit
         end if
      end do
      if( ndx > megan_cnt ) then
         write(*,*) 'megan_iso_emis: ISOP is not in megan species list'
         call endrun
      end if

!-------------------------------------------------- -----------------------------------
! 	... find month from calendar day
!-------------------------------------------------------------------------------------
      month = 0
      if( calday >= 365. ) then
        month = 12
      else
        do n = 1,12
          total_days = sum( days_in_month(1:n) )
          if( calday <= total_days ) then
            month = n
            exit
          end if
        end do
      end if
      if( month == 0 ) then
        write(*,*) 'megan_iso_emis: cannot find month for calday = ',calday
        call endrun
      end if

long_loop : &
      do i = 1,plonl
!-------------------------------------------------------------------------------------
! 	... PAR correction
!           Alex suggested to use a typical daily PPFD for tropical regions
!          ggp/lamar: Pdaily would be (24-h) daily average above canopy PPFD 
!-------------------------------------------------------------------------------------
#ifdef SW_DEBUG
       if( (base_lat+lat) == d_lat .and. (ip-1)*plonl+i == d_lon ) then
          write(*,*) ' '
          write(*,*) 'megan_iso_emis: iso emis = ',sflx(i)
       end if
#endif
         if( dyn_has_ts_avg ) then
            Pdaily =  fsds_avg(i) * const0
            Tdaily  = ts_avg(i)
         else
            Pdaily =  420. * const0
            Tdaily  = Tdaily_clim
         end if
         ppfd   = fsds(i) * const0
	 zen_angle_deg = zen_angle(i) * rad_to_degree
         sol_angle_deg = 90. - zen_angle_deg
         sol_angle_rad = sol_angle_deg * degree_to_rad
	 Ptoa = 3000. + 99.* cos( twopi*(calday - 10.)/365. )
	 phi  = ppfd / (sin( sol_angle_rad )*Ptoa)
!-------------------------------------------------------------------------------------
!  phi can get > 1 and then fac_par gets negative with the above equation
!  set phi=1 if phi> 1 as recommended by Alex
!-------------------------------------------------------------------------------------
         phi = min( phi,1. )
	 if( sol_angle_deg >= 90.0 .or. sol_angle_deg <= 0. ) then 
	    fac_par = 0.
	 else 
	    fac_par = sin(sol_angle_rad)*(2.49 * (1. + .0005 *(Pdaily - 400.))*phi - .9*phi*phi)
	 end if

!        write(*,*) 'megan_iso_emis ', zen_angle_deg,sol_angle_deg, Ptoa, phi, ppfd, fac_par, Pdaily
!-------------------------------------------------------------------------------------
! 	... temperature correction  equation 14
!           Topt from equation 8
!-------------------------------------------------------------------------------------
         t_diff  = Tdaily - Tdaily_clim
	 Topt    = 313. + 0.6*t_diff
         x       = (ts(i) - Topt)/(ts(i)*Topt*.00831)
	 Eopt    = 1.75 * exp( .08*t_diff )
         fac_tmp = Eopt * (ctm2 * exp( ctm1*x ))/(ctm2 - ctm1*(1. - exp( ctm2*x )))

!	write(*,*) 'megan_iso_emis; fac_tmp: ',Ptoa, phi, fac_tmp, ppfd, Tdaily
!       write(*,*) 'megan_iso_emis: ', Pdaily, Tdaily, ts_avg(i), fsds_avg(i)

!-------------------------------------------------------------------------------------
! 	... regridded potential emissions, including LAI correction
!-------------------------------------------------------------------------------------
         total_emis = maps_megan(i,lat,ip,ndx)
!-------------------------------------------------------------------------------------
! 	... change units from microg/m2/h to mol/cm2/s
!           2.46e8 = (6E23 molec/mole) / (68E6 migrog/mole) * (1E-4 m2/cm2)/(3600 sec/hr)
!-------------------------------------------------------------------------------------
         biso_flx(i) = total_emis * fac_par * fac_tmp * 2.46e8
#ifdef SW_DEBUG
       if( (base_lat+lat) == d_lat .and. (ip-1)*plonl+i == d_lon ) then
          write(*,*) 'megan_iso_emis: iso emis = ',biso_flx(i)*amufac*adv_mass(isop_ndx)
          write(*,*) ' '
       end if
#endif
      end do long_loop

!-------------------------------------------------------------------------------------
! 	... change to surface flux units for MOZART
!-------------------------------------------------------------------------------------
      sflx(:) = sflx(:) + biso_flx(:) * amufac * adv_mass(isop_ndx)

      end subroutine megan_iso_emis

      subroutine interp_map_mterps( nlon_veg, nlat_veg, npft_veg, landmask, vegetation_map, &
                                    monthly_lai_clm, month, plonl, platl, pplon, ndx )
!-------------------------------------------------------------------------------------
!	... interpolate monoterpine surface emission from "land" grid
!           to model grid
!-------------------------------------------------------------------------------------

      use mo_grid, only      : plon, plat
      use mo_constants, only : r2d, phi, lam
      use m_adv, only        : has_npole
      use mo_mpi, only       : base_lat

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in)      :: plonl
      integer, intent(in)      :: platl
      integer, intent(in)      :: pplon
      integer, intent(in)      :: nlon_veg, nlat_veg, npft_veg
      integer, intent(in)      :: month
      integer, intent(in)      :: ndx
      real, intent(in)         :: landmask       (nlon_veg,nlat_veg)
      real, intent(in)         :: vegetation_map (nlon_veg,nlat_veg,npft_veg)
      real, intent(in)         :: monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer                  :: i, j, ii, jj, i_ndx, n
      integer                  :: ip, i1
      integer                  :: nl, nu
      integer                  :: cell
      integer                  :: pft_li(veg_types)
      integer                  :: pft_lu(veg_types)
      real                     :: wrk_area
      real                     :: total_mterp
      real                     :: work_mterp(veg_types)
      real                     :: lai_fac(npft_veg)
      real                     :: veg_wrk(npft_veg)
      real                     :: work_iso(veg_types)
      real                     :: wrk_clm(veg_types)

!-----------------------------------------------------------------
!       ... initializations
!-----------------------------------------------------------------
      pft_li(:veg_types)   = (/ 2, 5, 10, 13, 16 /)
      pft_lu(:veg_types-1) = (/ 4, 9, 12, 15 /)
      pft_lu(veg_types)    = npft_veg

lat_loop : &
      do j = 1,platl
lon_loop : &
        do i = 1,plon
          total_mterp  = 0.
          ip = (i - 1)/plonl + 1
          i1 = mod( i - 1,plonl ) + 1
cell_loop : &
          do cell = 1,model2veg_map(i,j)%count
              ii = model2veg_map(i,j)%src_xndx(cell)
              jj = model2veg_map(i,j)%src_yndx(cell)
              wrk_area = model2veg_map(i,j)%src_area(cell)
              i_ndx    = mapping_ext(ii)
!-----------------------------------------------------------------
! 	... no emissions for ocean grid point 
!-----------------------------------------------------------------
              if( nint(landmask(i_ndx,jj)) == 0 ) then
                 cycle cell_loop
              end if
              wrk_clm(:) = maps_clm(i_ndx,jj,:,ndx)
              veg_wrk(:) = vegetation_map(i_ndx,jj,:)
              lai_fac(:) = monthly_lai_clm(i_ndx,jj,:,month)
!------------------------------------------------------------------------------------
!       ... emissions from :
!           (1) fine leaf trees (vegtypes = 'ntr')
!           (2) broadleaf (vegtypes = 'btr')
!           (3) shrubs (vegtypes = 'shr')
!           (4) grass (vegtypes = 'grs')
!           (5) crops (vegtypes = 'crp')
!-------------------------------------------------------------------------------------
              do n = 1,veg_types
                 nl = pft_li(n)
                 nu = pft_lu(n)
                 work_mterp(n)  = dot_product( lai_fac(nl:nu),veg_wrk(nl:nu) ) * wrk_clm(n)
              end do
              total_mterp = total_mterp + sum( work_mterp ) * wrk_area
          end do cell_loop
!-------------------------------------------------------------------------------------
! 	... divide by total grid area
!-------------------------------------------------------------------------------------
          maps_megan(i1,j,ip,ndx) = .2*total_mterp/(model2veg_map(i,j)%total_src_area + 1.e-30)
#ifdef SW_DEBUG
          if( (base_lat+j) == d_lat .and. i == d_lon ) then
             write(*,*) ' '
             write(*,*) 'interp_map_mterps: interpolated mterps = ',.2*total_mterp
             write(*,*) ' '
          end if
#endif
        end do lon_loop
      end do lat_loop

      end subroutine interp_map_mterps

      subroutine megan_mterps_emis( lat, ip, calday, ts, sflx, plonl )

      use mo_grid,   only    : pcnst
      use chem_mods, only    : adv_mass
      use mo_chem_utls, only : get_spc_ndx
      use mo_mpi, only       : base_lat

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: lat             ! latitude index
      integer, intent(in) :: ip              ! longitude tile index
      integer, intent(in) :: plonl
      real, intent(in)    :: calday          ! julian day + fraction (greenwich)
      real, intent(in)    :: ts(plonl)       ! surface temperature
      real, intent(inout) :: sflx(plonl)     ! surface flux for advected species

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      real, parameter :: tstd   = 303.15
      real, parameter :: amufac = 1.65979e-23         ! 1.e4* kg / amu

      integer :: i,n
      integer :: ndx, mterps_ndx
      real    :: ppfd, x
      real    :: total_area, total_emis, fac_par, fac_tmp
      real    :: bmterp_flx(plonl)

      do ndx = 1,megan_cnt
         if( megan_species(ndx) == 'C10H16' ) then 
            mterps_ndx = get_spc_ndx( 'C10H16' )
            exit
         end if
      end do
      if( ndx > megan_cnt ) then
         write(*,*) 'megan_mterps_emis: C10H16 is not in megan species list'
         call endrun
      end if

      do i = 1,plonl
#ifdef SW_DEBUG
       if( (base_lat+lat) == d_lat .and. (ip-1)*plonl+i == d_lon ) then
          write(*,*) ' '
          write(*,*) 'megan_mterps_emis: mterps emis = ',sflx(i)
       end if
#endif
!-------------------------------------------------------------------------------------
! 	... temperature correction
!-------------------------------------------------------------------------------------
         x = ts(i) - tstd
         fac_tmp = exp( .09*x )
         fac_par = 1.
!-------------------------------------------------------------------------------------
! 	... regridded potential emissions, including LAI correction
!-------------------------------------------------------------------------------------
        total_emis = maps_megan(i,lat,ip,ndx)
!-------------------------------------------------------------------------------------
! 	... change units from microg/m2/h to mol/cm2/s
!           1.229e8 = (6E23 molec/mole) / (136E6 migrog/mole) * (1E-4 m2/cm2)/(3600 sec/hr)
!-------------------------------------------------------------------------------------
        bmterp_flx(i) = total_emis * fac_par * fac_tmp * 1.229e8
#ifdef SW_DEBUG
       if( (base_lat+lat) == d_lat .and. (ip-1)*plonl+i == d_lon ) then
          write(*,*) 'megan_mterps_emis: mterps emis = ',bmterp_flx(i)*amufac*adv_mass(mterps_ndx)
          write(*,*) ' '
!         call endrun
       end if
#endif
      end do
!-------------------------------------------------------------------------------------
! 	... change to surface flux units for mozart
!-------------------------------------------------------------------------------------
      sflx(:) = sflx(:) + bmterp_flx(:) * amufac * adv_mass(mterps_ndx)

      end subroutine megan_mterps_emis

      subroutine bvoc_rdrst( plonl, platl, pplon, ncid )
!---------------------------------------------------------------------- 
!	... read soil no restart variables
!---------------------------------------------------------------------- 
      
      use netcdf
      use mo_mpi,  only : base_lat
      use mo_grid, only : plon

      implicit none

!---------------------------------------------------------------------- 
!	... dummy arguments
!---------------------------------------------------------------------- 
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncid                          ! netcdf dataset id

!---------------------------------------------------------------------- 
!	... local variables
!---------------------------------------------------------------------- 
      integer              :: j, m
      integer              :: var_id
      integer              :: start(2)
      integer              :: cnt(2)
      real                 :: wrk2d(plon,platl)
      character(len=16)    :: varname

      start(:) = (/ 1, base_lat+1 /)
      cnt(:)   = (/ plon, platl /)
!-----------------------------------------------------------------------
!     	... read megan maps
!-----------------------------------------------------------------------
      do m = 1,megan_cnt
         varname = 'megan_map_' // trim( megan_species(m) )
         call handle_ncerr( nf_inq_varid( ncid, trim(varname), var_id ), &
                            'bvoc_rdrst: Failed to get ' // trim(varname) // ' variable id' )
         call handle_ncerr( nf_get_vara_double( ncid, var_id, start, cnt, wrk2d ), &
                            'bvoc_rdrst: failed to read ' // trim(varname) // ' variable' )
         do j = 1,platl
	    maps_megan(:,j,:,m) = reshape( wrk2d(:,j), (/ plonl, pplon /) )
         end do
         write(*,*) 'bvoc_rdrst: read ' // trim(varname)
      end do

      end subroutine bvoc_rdrst

      subroutine bvoc_wrrst( plonl, platl, pplon, ncid )
!---------------------------------------------------------------------- 
!	... write soil no restart variables
!---------------------------------------------------------------------- 
      
      use netcdf
      use mo_mpi
      use mo_grid, only : plon, plat

      implicit none

!---------------------------------------------------------------------- 
!	... dummy arguments
!---------------------------------------------------------------------- 
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncid                          ! netcdf dataset id

!---------------------------------------------------------------------- 
!	... local variables
!---------------------------------------------------------------------- 
      integer              :: astat
      integer              :: istat
      integer              :: j, m
      integer              :: var_id
      integer              :: count
      integer              :: node
      integer              :: nodes
      integer              :: offset
      real                 :: wrk2d(plon,plat)
      real   , allocatable :: gather_2d(:,:,:,:)
      character(len=16)    :: varname

#ifdef USE_MPI
      if( ded_io_node ) then
	 nodes = maxnodes + 1
      else
	 nodes = maxnodes
      end if
      count = plon*platl
!-----------------------------------------------------------------------
!     	... gather megan maps
!-----------------------------------------------------------------------
      if( io_node ) then
         allocate( gather_2d(plonl,platl,pplon,nodes),stat=astat )
      else
         allocate( gather_2d(1,1,1,1),stat=astat )
      end if
      if( astat /= 0 ) then
	 write(*,*) 'bvoc_wrrst: failed to allocate gather_2d; error = ',astat
	 call endrun
      end if
#endif
      do m = 1,megan_cnt
         varname = 'megan_map_' // trim( megan_species(m) )
#ifdef USE_MPI
         call mpi_gather( maps_megan(:,:,:,m), count, mpi_double_precision, &
	                  gather_2d, count, mpi_double_precision, &
		          gather_node, mpi_comm_world, istat )
         if( istat /= MPI_SUCCESS ) then
            write(*,*) 'bvoc_wrrst: mpi_gather failed for ' // trim(varname) // '; error code = ',istat
	    call endrun
         end if
#endif
!-----------------------------------------------------------------------
!     	... write megan maps
!-----------------------------------------------------------------------
         if( io_node ) then
            call handle_ncerr( nf_inq_varid( ncid, trim(varname), var_id ), &
                               'bvoc_wrrst: failed to get ' // trim(varname) // ' variable id' )
#ifdef USE_MPI
	    do node = 1,maxnodes
	       offset = (node-1)*platl
	       do j = 1,platl
	          wrk2d(:,j+offset) = reshape( gather_2d(:,j,:,node), (/ plon /) )
	       end do
	    end do
#else
	    do j = 1,platl
	       wrk2d(:,j) = reshape( maps_megan(:,j,:,m), (/ plon /) )
	    end do
#endif
            call handle_ncerr( nf_put_var_double( ncid, var_id, wrk2d ), &
                               'bvoc_wrrst: failed to write ' // trim(varname) // ' variable' )
            write(*,*) 'bvoc_wrrst: ' // trim(varname) // ' written'
         end if
      end do

      if( allocated( gather_2d ) ) then
         deallocate( gather_2d )
      end if

      end subroutine bvoc_wrrst

      end module mo_bvoc
