
      module mo_chem_utls

      private
      public :: adjh2o, inti_mr_xform, mmr2vmr, vmr2mmr, negtrc, &
                get_spc_ndx, get_het_ndx, get_extfrc_ndx, &
                has_drydep, has_srfems, has_fixed_lbc, get_rxt_ndx, get_grp_ndx, &
                get_grp_mem_ndx, get_inv_ndx, chem_utls_inti, iniele, com_mass
      public :: has_megan_srfems, has_xactive_srfems
      public :: has_fixed_ubc

      save

      integer :: ox_ndx, o3_ndx, o1d_ndx, o_ndx
      logical :: do_ox

      type ELEMENT
	 character(len=2) :: sym
	 real             :: wght
      end type ELEMENT

      integer :: tab_max = 100
      integer :: id_cnt = 1
      character(len=39) :: id
      type( ELEMENT    ):: e_table(100)

      contains

      subroutine chem_utls_inti
!-----------------------------------------------------------------------
!     ... Initialize the chem utils module
!-----------------------------------------------------------------------

      implicit none

      ox_ndx = get_spc_ndx( 'OX' )
      if( ox_ndx > 0 ) then
         o3_ndx  = get_grp_mem_ndx( 'O3' )
         o1d_ndx = get_grp_mem_ndx( 'O1D' )
         o_ndx   = get_grp_mem_ndx( 'O' )
         do_ox   = o3_ndx > 0 .and. o1d_ndx > 0 .and. o_ndx > 0
      else
         o3_ndx  = 1
         o1d_ndx = 1
         o_ndx   = 1
         do_ox = .false.
      end if

      end subroutine chem_utls_inti

      subroutine adjh2o( h2o, sh, mbar, vmr, plonl )
!-----------------------------------------------------------------------
!     ... transform water vapor from mass to volumetric mixing ratio
!-----------------------------------------------------------------------

      use mo_grid,    only : plev, pcnstm1

      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      real, intent(in)    :: vmr(plonl,plev,pcnstm1)                    ! xported species (mol/mol)
      real, intent(in)    :: sh(plonl,plev)                             ! specific humidity (kg/kg)
      real, intent(in)    :: mbar(plonl,plev)                           ! atmos mean mass
      real, intent(inout) :: h2o(plonl,plev)                            ! water vapor (mol/mol)

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      real, parameter :: mh2o = 1. /18.01528

      integer ::   k, ndx_ch4
      real    ::   t_value(plonl)

!-----------------------------------------------------------------------
!	... limit dyn files water vapor
!-----------------------------------------------------------------------
!     ndx_ch4 = get_spc_ndx( 'CH4' )
      ndx_ch4 = -1
      if( ndx_ch4 > 0 ) then
         do k = 1,plev
            h2o(:,k)   = mbar(:,k) * sh(:,k) * mh2o
            t_value(:) = 6.e-6 - 2.*vmr(:,k,ndx_ch4)
            where( t_value(:) > h2o(:,k) )
               h2o(:,k) = t_value(:)
            endwhere
         end do
      else
         do k = 1,plev
            h2o(:,k)   = mbar(:,k) * sh(:,k) * mh2o
         end do
      end if

      end subroutine adjh2o      

      subroutine inti_mr_xform( sh, mbar, plonl )
!-----------------------------------------------------------------
!	... initialize mean atmospheric "wet" mass
!-----------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

!-----------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------
      integer, intent(in) :: plonl
      real, intent(in)    :: sh(plonl,plev)     ! specific humidity (kg/kg)
      real, intent(out)   :: mbar(plonl,plev)   ! mean wet atm mass ( amu )

!-----------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------
      real, parameter :: dry_mass = 28.966    ! amu
      real, parameter :: mfac = 1. / .622

      integer :: k

      do k = 1,plev
         mbar(:,k) = dry_mass
      end do

      end subroutine inti_mr_xform

      subroutine mmr2vmr( vmr, mmr, mbar, plonl )
!-----------------------------------------------------------------
!	... xfrom from mass to volume mixing ratio
!-----------------------------------------------------------------

      use chem_mods, only : adv_mass
      use mo_grid,   only : plev, pcnstm1, pcnst

      implicit none

!-----------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------
      integer, intent(in) :: plonl
      real, intent(in)    :: mbar(plonl,plev)
      real, intent(in)    :: mmr(plonl,plev,pcnst)
      real, intent(out)   :: vmr(plonl,plev,pcnstm1)

!-----------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------
      integer :: k, m

      do m = 1,pcnstm1
         if( adv_mass(m) /= 0. ) then
            do k = 1,plev
               vmr(:,k,m) = mbar(:,k) * mmr(:,k,m) / adv_mass(m)
            end do
         end if
      end do

      end subroutine mmr2vmr

      subroutine vmr2mmr( vmr, mmr, nas, grp_ratios, mbar, plonl )
!-----------------------------------------------------------------
!	... xfrom from mass to volume mixing ratio
!-----------------------------------------------------------------

      use chem_mods, only : adv_mass, nadv_mass, grpcnt
      use mo_grid,   only : plev, pcnstm1, pcnst

      implicit none

!-----------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------
      integer, intent(in) :: plonl
      real, intent(in)    :: mbar(plonl,plev)
      real, intent(in)    :: vmr(plonl,plev,pcnstm1)
      real, intent(inout) :: mmr(plonl,plev,pcnst)
      real, intent(in)    :: grp_ratios(:,:,:)
      real, intent(out)   :: nas(:,:,:)

!-----------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------
      integer :: k, m
      real    :: grp_mass(plonl)            ! weighted group mass

!-----------------------------------------------------------------
!	... the non-group species
!-----------------------------------------------------------------
      do m = 1,pcnstm1
         if( adv_mass(m) /= 0. ) then
            do k = 1,plev
               mmr(:,k,m) = adv_mass(m) * vmr(:,k,m) / mbar(:,k)
            end do
         end if
      end do
!-----------------------------------------------------------------
!	... the "group" species
!-----------------------------------------------------------------
      if( do_ox ) then
         do k = 1,plev
            grp_mass(:)     = grp_ratios(:,k,o3_ndx) * nadv_mass(o3_ndx) &
                              + grp_ratios(:,k,o_ndx) * nadv_mass(o_ndx) &
                              + grp_ratios(:,k,o1d_ndx) * nadv_mass(o1d_ndx)      
            mmr(:,k,ox_ndx)  = grp_mass(:) * vmr(:,k,ox_ndx) / mbar(:,k)
            grp_mass(:)     = mmr(:,k,ox_ndx) / grp_mass(:)
            nas(:,k,o3_ndx)  = nadv_mass(o3_ndx) * grp_ratios(:,k,o3_ndx) * grp_mass(:)
            nas(:,k,o_ndx)   = nadv_mass(o_ndx) * grp_ratios(:,k,o_ndx) * grp_mass(:)
            nas(:,k,o1d_ndx) = nadv_mass(o1d_ndx) * grp_ratios(:,k,o1d_ndx) * grp_mass(:)
         end do
      end if

      end subroutine vmr2mmr

      subroutine negtrc( lat, header, fld, plonl )
!-----------------------------------------------------------------------
!  	... check for negative constituent values and
!	    replace with zero value
!-----------------------------------------------------------------------

      use mo_grid,    only : plev, pcnstm1
      use m_tracname, only : tracnam
      use mo_control, only : pdiags

      implicit none

!-----------------------------------------------------------------------
!  	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)          :: lat                      ! current latitude
      integer, intent(in)          :: plonl
      character(len=*), intent(in) :: header                   ! caller tag
      real, intent(inout)          :: fld(plonl,plev,pcnstm1)  ! field to check

!-----------------------------------------------------------------------
!  	... local variables
!-----------------------------------------------------------------------
      integer :: m
      integer :: nneg                       ! flag counter
      integer :: iw, kw
      integer :: windex(2)
      real    :: worst

      do m  = 1,pcnstm1
         nneg = count( fld(:,:,m) < 0. )
         if( nneg > 0 ) then
            where( fld(:,:,m) < 0. )
               fld(:,:,m) = 0.
            endwhere
            if( pdiags%negtrc ) then
               worst     = minval( fld(:,:,m) )
               windex(:) = minloc( fld(:,:,m) )
               iw        = windex(1)
               kw        = windex(2)
            end if
         end if
         if( pdiags%negtrc .and. nneg > 0 ) then
            write(*,*) header(:len(header)), tracnam(m), ' has ',nneg,' neg values'
            write(*,*) ' worst =',worst,' @ long = ',iw,' lat = ',lat,' eta = ',kw
         end if
      end do

      end subroutine negtrc

      integer function get_spc_ndx( spc_name )
!-----------------------------------------------------------------------
!     ... return overall species index associated with spc_name
!-----------------------------------------------------------------------

      use chem_mods,  only : pcnstm1
      use m_tracname, only : tracnam

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: spc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      get_spc_ndx = -1
      do m = 1,pcnstm1
         if( trim( spc_name ) == trim( tracnam(m) ) ) then
            get_spc_ndx = m
            exit
         end if
      end do

      end function get_spc_ndx

      integer function get_grp_ndx( grp_name )
!-----------------------------------------------------------------------
!     ... return group index associated with spc_name
!-----------------------------------------------------------------------

      use chem_mods,  only : ngrp, grp_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: grp_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      get_grp_ndx = -1
      do m = 1,ngrp
         if( trim( grp_name ) == trim( grp_lst(m) ) ) then
            get_grp_ndx = m
            exit
         end if
      end do

      end function get_grp_ndx

      integer function get_grp_mem_ndx( mem_name )
!-----------------------------------------------------------------------
!     ... return group member index associated with spc_name
!-----------------------------------------------------------------------

      use chem_mods,  only : grpcnt
      use m_tracname, only : natsnam

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: mem_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      get_grp_mem_ndx = -1
      if( grpcnt > 0 ) then
         do m = 1,max(1,grpcnt)
            if( trim( mem_name ) == trim( natsnam(m) ) ) then
               get_grp_mem_ndx = m
               exit
            end if
         end do
      end if

      end function get_grp_mem_ndx

      integer function get_inv_ndx( invariant )
!-----------------------------------------------------------------------
!     ... return overall external frcing index associated with spc_name
!-----------------------------------------------------------------------

      use chem_mods,  only : nfs, inv_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: invariant

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      get_inv_ndx = -1
      do m = 1,nfs
         if( trim( invariant ) == trim( inv_lst(m) ) ) then
            get_inv_ndx = m
            exit
         end if
      end do

      end function get_inv_ndx

      integer function get_het_ndx( het_name )
!-----------------------------------------------------------------------
!     ... return overall het process index associated with spc_name
!-----------------------------------------------------------------------

      use chem_mods,  only : hetcnt, het_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: het_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      get_het_ndx = -1
      do m = 1,max(1,hetcnt)
         if( trim( het_name ) == trim( het_lst(m) ) ) then
            get_het_ndx = m
            exit
         end if
      end do

      end function get_het_ndx

      integer function get_extfrc_ndx( frc_name )
!-----------------------------------------------------------------------
!     ... return overall external frcing index associated with spc_name
!-----------------------------------------------------------------------

      use chem_mods,  only : extcnt, extfrc_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: frc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      get_extfrc_ndx = -1
      if( extcnt > 0 ) then
         do m = 1,max(1,extcnt)
            if( trim( frc_name ) == trim( extfrc_lst(m) ) ) then
               get_extfrc_ndx = m
               exit
            end if
         end do
      end if

      end function get_extfrc_ndx

      integer function get_rxt_ndx( rxt_tag )
!-----------------------------------------------------------------------
!     ... return overall reaction index associated with rxt_tag
!-----------------------------------------------------------------------

      use chem_mods,  only : rxt_tag_cnt, rxt_tag_lst, rxt_tag_map

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: rxt_tag

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      get_rxt_ndx = -1
      do m = 1,rxt_tag_cnt
         if( trim( rxt_tag ) == trim( rxt_tag_lst(m) ) ) then
            get_rxt_ndx = rxt_tag_map(m)
            exit
         end if
      end do

      end function get_rxt_ndx

      logical function has_drydep( spc_name )
!-----------------------------------------------------------------------
!     ... return logical for species dry deposition
!-----------------------------------------------------------------------

      use chem_mods,  only : drydep_cnt, drydep_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: spc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      has_drydep = .false.
      do m = 1,drydep_cnt
         if( trim( spc_name ) == trim( drydep_lst(m) ) ) then
            has_drydep = .true.
            exit
         end if
      end do

      end function has_drydep

      logical function has_srfems( spc_name )
!-----------------------------------------------------------------------
!     ... return logical for species surface emission
!-----------------------------------------------------------------------

      use chem_mods,  only : srfems_cnt, srfems_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: spc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      has_srfems = .false.
      do m = 1,srfems_cnt
         if( trim( spc_name ) == trim( srfems_lst(m) ) ) then
            has_srfems = .true.
            exit
         end if
      end do

      end function has_srfems

      logical function has_megan_srfems( spc_name )
!-----------------------------------------------------------------------
!     ... return logical for species megan surface emission
!-----------------------------------------------------------------------

      use chem_mods,  only : srfems_cnt, srfems_lst, megan_map

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: spc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      has_megan_srfems = .false.
      do m = 1,srfems_cnt
         if( trim( spc_name ) == trim( srfems_lst(m) ) ) then
            has_megan_srfems = megan_map(m)
            exit
         end if
      end do

      end function has_megan_srfems

      logical function has_xactive_srfems( spc_name )
!-----------------------------------------------------------------------
!     ... return logical for species megan surface emission
!-----------------------------------------------------------------------

      use chem_mods,  only : srfems_cnt, srfems_lst, xactive_srf_flx_map

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: spc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      has_xactive_srfems = .false.
      do m = 1,srfems_cnt
         if( trim( spc_name ) == trim( srfems_lst(m) ) ) then
            has_xactive_srfems = xactive_srf_flx_map(m)
            exit
         end if
      end do

      end function has_xactive_srfems

      logical function has_fixed_lbc( spc_name )
!-----------------------------------------------------------------------
!     ... return logical for species fixed lb condition
!-----------------------------------------------------------------------

      use chem_mods,  only : fbc_cnt, flbc_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: spc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      has_fixed_lbc = .false.
      do m = 1,fbc_cnt(1)
         if( trim( spc_name ) == trim( flbc_lst(m) ) ) then
            has_fixed_lbc = .true.
            exit
         end if
      end do

      end function has_fixed_lbc

      logical function has_fixed_ubc( spc_name )
!-----------------------------------------------------------------------
!     ... return logical for species fixed lb condition
!-----------------------------------------------------------------------

      use chem_mods,  only : fbc_cnt, fubc_lst

      implicit none

!-----------------------------------------------------------------------
!     ... dummy arguments
!-----------------------------------------------------------------------
      character(len=*), intent(in) :: spc_name

!-----------------------------------------------------------------------
!     ... local variables
!-----------------------------------------------------------------------
      integer :: m

      has_fixed_ubc = .false.
      do m = 1,fbc_cnt(2)
         if( trim( spc_name ) == trim( fubc_lst(m) ) ) then
            has_fixed_ubc = .true.
            exit
         end if
      end do

      end function has_fixed_ubc

      subroutine iniele
!-----------------------------------------------------------
!	... Initialize the element mass table and mass computation
!-----------------------------------------------------------

      implicit none

!-----------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------
      integer :: i

      e_table(:)%sym = '  '
      e_table(1) =  ELEMENT( 'H ',1.0074 )
      e_table(2) =  ELEMENT( 'He',4.0020602 )
      e_table(3) =  ELEMENT( 'Li',6.941 )
      e_table(4) =  ELEMENT( 'Be',9.012182 )
      e_table(5) =  ELEMENT( 'B ',10.811 )
      e_table(6) =  ELEMENT( 'C ',12.011 )
      e_table(7) =  ELEMENT( 'N ',14.00674 )
      e_table(8) =  ELEMENT( 'O ',15.9994 )
      e_table(9) =  ELEMENT( 'F ',18.9984032 )
      e_table(10) = ELEMENT( 'Ne',20.1797 )
      e_table(11) = ELEMENT( 'Na',22.989768 )
      e_table(12) = ELEMENT( 'Mg',24.305 )
      e_table(13) = ELEMENT( 'Al',26.981539 )
      e_table(14) = ELEMENT( 'Si',28.0855 )
      e_table(15) = ELEMENT( 'P ',30.97362 )
      e_table(16) = ELEMENT( 'S ',32.066 )
      e_table(17) = ELEMENT( 'Cl',35.4527 )
      e_table(18) = ELEMENT( 'Ar',39.948 )
      e_table(19) = ELEMENT( 'K ',39.0983 )
      e_table(20) = ELEMENT( 'Ca',40.078 )
      e_table(21) = ELEMENT( 'Sc',44.95591 )
      e_table(22) = ELEMENT( 'Ti',47.867 )
      e_table(23) = ELEMENT( 'V ',50.9415 )
      e_table(24) = ELEMENT( 'Cr',51.9961 )
      e_table(25) = ELEMENT( 'Mn',54.93085 )
      e_table(26) = ELEMENT( 'Fe',55.845 )
      e_table(27) = ELEMENT( 'Co',58.9332 )
      e_table(28) = ELEMENT( 'Ni',58.6934 )
      e_table(29) = ELEMENT( 'Cu',63.546 )
      e_table(30) = ELEMENT( 'Zn',65.39 )
      e_table(31) = ELEMENT( 'Ga',69.723 )
      e_table(32) = ELEMENT( 'Ge',72.61 )
      e_table(33) = ELEMENT( 'As',74.92159 )
      e_table(34) = ELEMENT( 'Se',78.96 )
      e_table(35) = ELEMENT( 'Br',79.904 )
      e_table(36) = ELEMENT( 'Kr',83.8 )
      e_table(37) = ELEMENT( 'Rb',85.4678 )
      e_table(38) = ELEMENT( 'Sr',87.62 )
      e_table(39) = ELEMENT( 'Y ',88.90585 )
      e_table(40) = ELEMENT( 'Zr',91.224 )
      e_table(41) = ELEMENT( 'Nb',92.90638 )
      e_table(42) = ELEMENT( 'Mo',95.94 )
      e_table(43) = ELEMENT( 'Tc',98. )
      e_table(44) = ELEMENT( 'Ru',101.07 )
      e_table(45) = ELEMENT( 'Rh',102.9055 )
      e_table(46) = ELEMENT( 'Pd',106.42 )
      e_table(47) = ELEMENT( 'Ag',107.8682 )
      e_table(48) = ELEMENT( 'Cd',112.411 )
      e_table(49) = ELEMENT( 'In',114.818 )
      e_table(50) = ELEMENT( 'Sn',118.71 )
      e_table(51) = ELEMENT( 'Sb',121.76 )
      e_table(52) = ELEMENT( 'Te',127.6 )
      e_table(53) = ELEMENT( 'I ',126.90447 )
      e_table(54) = ELEMENT( 'Xe',131.29 )
      e_table(55) = ELEMENT( 'Cs',132.90543 )
      e_table(56) = ELEMENT( 'Ba',137.327 )
      e_table(57) = ELEMENT( 'La',138.9055 )
      e_table(58) = ELEMENT( 'Hf',178.49 )
      e_table(59) = ELEMENT( 'Ta',180.9479 )
      e_table(60) = ELEMENT( 'W ',183.84 )
      e_table(61) = ELEMENT( 'Re',186.207 )
      e_table(62) = ELEMENT( 'Os',190.23 )
      e_table(63) = ELEMENT( 'Ir',192.217 )
      e_table(64) = ELEMENT( 'Pt',195.08 )
      e_table(65) = ELEMENT( 'Au',196.96654 )
      e_table(66) = ELEMENT( 'Hg',200.59 )
      e_table(67) = ELEMENT( 'Tl',204.3833 )
      e_table(68) = ELEMENT( 'Pb',207.2 )
      e_table(69) = ELEMENT( 'Bi',208.98037 )
      e_table(70) = ELEMENT( 'Po',209. )
      e_table(71) = ELEMENT( 'At',210. )
      e_table(72) = ELEMENT( 'Rn',222. )
      e_table(73) = ELEMENT( 'Fr',223. )
      e_table(74) = ELEMENT( 'Ra',226.025 )
      e_table(75) = ELEMENT( 'Ac',227.028 )

      do i = 1,tab_max
	 if( e_table(i)%sym == '  ' ) then
	    exit
	 end if
      end do
      tab_max = i - 1

      id(:1) = e_table(1)%sym(:1)
      do i = 2,tab_max
	 if( scan( e_table(i)%sym(:1), id(:id_cnt) ) == 0 ) then
	    id_cnt = id_cnt + 1
	    id(id_cnt:id_cnt) = e_table(i)%sym(:1)
	 end if
      end do

      end subroutine iniele

      real function com_mass( compound )
!-----------------------------------------------------------
!	... Compute the mass of input compound
!-----------------------------------------------------------

      implicit none

!-----------------------------------------------------------
!	... Dummy args
!-----------------------------------------------------------
      character(len=*), intent(in) :: compound

!-----------------------------------------------------------
!	... Local variables
!-----------------------------------------------------------
      integer :: beg, end, pos, nump, index
      integer :: ios, el_cnt

      end = len_trim( compound )
      com_mass = 0.
table_search : &
      do
	 pos = scan( compound(:end), id(:id_cnt), back = .true. )
	 if( pos == 0 ) then
	    exit
	 end if
	 nump = scan( compound(pos+1:end), '0123456789' )
	 if( nump /= 0 ) then
	    nump = pos + nump
	    read(compound(nump:end),*,iostat=ios) el_cnt
	    if( ios /= 0 .or. el_cnt == 0 ) then
	       com_mass = 0.
	       exit
	    end if
	    end = nump - 1
	 else
	    el_cnt = 1
	 end if
	 do index = 1,tab_max
	    if( e_table(index)%sym == compound(pos:end) ) then
	       exit
	    end if
	 end do
	 com_mass = com_mass + e_table(index)%wght * real( el_cnt )
	 end = pos - 1
	 if( end <= 0 ) then
	    exit
	 end if
      end do table_search

      end function com_mass

      end module mo_chem_utls
