
      module mo_chemdr

      use mo_chem_utls,     only : inti_mr_xform, adjh2o, negtrc, mmr2vmr, vmr2mmr
      use mo_chem_utls,     only : get_spc_ndx, get_grp_ndx, get_grp_mem_ndx
      use mo_chem_utls,     only : get_het_ndx, get_extfrc_ndx
      use mo_dust,          only : nqdust

      implicit none

      private
      public :: chemdr
      public :: chemdr_inti

      save

      integer :: no_ndx, no2_ndx
      integer :: no3_ndx, hno3_ndx, n2o5_ndx, ho2no2_ndx
      integer :: pan_ndx, mpan_ndx, onit_ndx, onitr_ndx, isopno3_ndx
      integer :: ox_grp_ndx, ox_ndx, o3_ndx
      integer :: synoz_ndx
      real    :: esfact = 1.           ! earth sun distance factor
      logical :: has_nox
      logical :: has_noy
      character(len=32) :: dust_nm(nqdust)

      contains

      subroutine chemdr_inti
!-----------------------------------------------------------------------
!	... intialize chemistry driver
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer          :: m
      character(len=2) :: numc

!-----------------------------------------------------------------------
!	... check for ozone and related compounds
!-----------------------------------------------------------------------
      ox_grp_ndx = get_grp_ndx( 'OX' )
      ox_ndx     = get_spc_ndx( 'OX' )
      if( ox_ndx < 1 ) then
         ox_ndx  = get_spc_ndx( 'O3' )
      end if
      synoz_ndx  = get_extfrc_ndx( 'SYNOZ' )
!-----------------------------------------------------------------------
!	... check for nox
!-----------------------------------------------------------------------
      no_ndx  = get_spc_ndx( 'NO' )
      no2_ndx = get_spc_ndx( 'NO2' )
      has_nox = no_ndx > 0 .and. no2_ndx > 0
      if( has_nox ) then
         no3_ndx     = get_spc_ndx( 'NO3' )
         hno3_ndx    = get_spc_ndx( 'HNO3' )
         ho2no2_ndx  = get_spc_ndx( 'HO2NO2' )
         n2o5_ndx    = get_spc_ndx( 'N2O5' )
         pan_ndx     = get_spc_ndx( 'PAN' )
         onit_ndx    = get_spc_ndx( 'ONIT' )
         onitr_ndx   = get_spc_ndx( 'ONITR' )
         mpan_ndx    = get_spc_ndx( 'MPAN' )
         isopno3_ndx = get_spc_ndx( 'ISOPNO3' )
         has_noy = no3_ndx > 0 .and. hno3_ndx > 0 &
                               .and. ho2no2_ndx > 0 .and. n2o5_ndx > 0 &
                               .and. pan_ndx > 0 .and. onit_ndx > 0  &
                               .and. onitr_ndx > 0 .and. mpan_ndx > 0  &
                               .and. isopno3_ndx > 0
      end if

!-----------------------------------------------------------------------
!     	... set outfld names
!-----------------------------------------------------------------------
      do m = 1,nqdust
         write(numc,'(i2)') 10+m
         dust_nm(m) = 'DUST' // numc(2:2)
      end do

      end subroutine chemdr_inti

      subroutine chemdr( mmr, pint, nstep, calday, ncdate, &
                         ncsec, lat, ip, delt, ps, &
                         pmid, pdel, oro, tsurf, zma, &
                         zi, zia, phis, cldfr, cldtop, cmfdqr, &
                         nrain, nevapr, cwat, tfld, um1, &
                         vm1, sh, snow, zm, plonl )
!-----------------------------------------------------------------------
!     ... Chem_solver advances the volumetric mixing ratio
!         forward one time step via a combination of explicit,
!         ebi, hov, fully implicit, and/or rodas algorithms.
!-----------------------------------------------------------------------

      use mo_control,       only : xactive_prates, use_dust
      use chem_mods,        only : indexm, nadv_mass, phtcnt, gascnt, rxntot, clscnt1, clscnt4, clscnt5
      use chem_mods,        only : ncol_abs, grpcnt, nfs, extcnt, hetcnt
      use chem_mods,        only : rxt_rate_map
      use chem_mods,        only : fbc_cnt
      use mo_ub_vals,       only : set_ub_vals
      use mo_ub_vals,       only : set_ub_h2o
      use mo_flbc,          only : flbc_set
      use mo_histout,       only : outfld, hfile, moz_file_cnt, hst_file_max, sim_file_cnt
      use mo_grid,          only : plev, plevp, pcnstm1, pcnst, plnplv
      use mo_sulf,          only : sulf_interp
      use mo_strato_sad,    only : strato_sad_set
      use mo_dust,          only : dust_interp
      use mo_photo,         only : set_ub_col, setcol, table_photo, xactive_photo, diurnal_geom, sundis
      use mo_calendar,      only : caldayr
      use mo_exp_sol,       only : exp_sol
      use mo_imp_sol,       only : imp_sol
      use mo_rodas_sol,     only : rodas_sol
      use mo_usrrxt,        only : usrrxt
      use mo_setinv,        only : setinv
      use mo_setext,        only : setext
      use mo_sethet,        only : sethet
      use mo_srfalb,        only : srfalb
      use mo_setrxt,        only : setrxt
      use mo_setinv,        only : has_h2o
      use mo_adjrxt,        only : adjrxt
      use mo_phtadj,        only : phtadj
      use mo_rxt_mod,       only : rxt_mod
      use mo_grp_ratios,    only : set_grp_ratios
      use mo_make_grp_vmr,  only : mak_grp_vmr
      use mo_mass,          only : qmassa
      use eslookup,         only : aqsat
      use mo_setsox,        only : setsox, has_sox
      use mo_setsoa,        only : setsoa, has_soa
      use mo_seasalt,       only : set_seasalt, has_seasalt
      use mo_aerosols,      only : aerosols_formation, has_aerosols
      use mo_tropopause,    only : tropp_lev
      use mo_constants,     only : rgrav

      implicit none

!-----------------------------------------------------------------------
!        ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) ::  nstep                   ! time index
      integer, intent(in) ::  lat                     ! latitude index
      integer, intent(in) ::  ip                      ! longitude tile index
      integer, intent(in) ::  ncdate                  ! date at end present time step
      integer, intent(in) ::  ncsec                   ! seconds relative to ncdate
      integer, intent(in) ::  plonl                   ! long tile dimension
      real,    intent(in) ::  calday                  ! time of year in days for midpoint time
      real,    intent(in) ::  delt                    ! timestep in seconds
      real, intent(in)    ::  ps(plonl)               ! surface press (Pa)
      real, intent(in)    ::  oro(plonl)              ! surface orography flag
      real, intent(in)    ::  tsurf(plonl)            ! surface temperature (K)
      real, intent(in)    ::  phis(plonl)             ! surf geopot
      real, intent(in)    ::  cldtop(plonl)           ! cloud top level ( 1 ... plev )
      real, intent(in)    ::  snow(plonl)             ! snow height (m)
      real, intent(in)    ::  pmid(plonl,plev)        ! midpoint press (Pa)
      real, intent(in)    ::  pdel(plonl,plev)        ! delta press across midpoints (Pa)
      real, intent(in)    ::  zma(plonl,plev)         ! abs geopot height at midpoints (m)
      real, intent(in)    ::  zm(plonl,plev)          ! rel geopot height at midpoints (m)
      real, intent(in)    ::  cmfdqr(plonl,plev)      ! dq/dt for convective rainout (1/s)
      real, intent(in)    ::  nrain(plonl,plev)       ! release of strt precip (1/s)
      real, intent(in)    ::  nevapr(plonl,plev)      ! evap precip (1/s)
      real, intent(in)    ::  cwat(plonl,plev)        ! total cloud water (kg/kg)
      real, intent(in)    ::  tfld(plonl,plev)        ! midpoint temperature (K)
      real, intent(in)    ::  um1(plonl,plev)         ! midpoint wind (m/s)
      real, intent(in)    ::  vm1(plonl,plev)         ! midpoint wind (m/s)
      real, intent(in)    ::  sh(plonl,plev)          ! specific humidity (kg/kg)
      real, intent(in)    ::  zi(plonl,plevp)         ! rel geopot height at interfaces (m)
      real, intent(in)    ::  zia(plonl,plevp)        ! abs geopot height at interfaces (m)
      real, intent(in)    ::  cldfr(plonl,plevp)      ! cloud fraction
      real, intent(in)    ::  pint(plonl,plevp)       ! interface pressure (Pa)
      real, intent(inout) ::  mmr(plonl,plev,pcnst)   ! xported species ( mmr )

!-----------------------------------------------------------------------
!     	... local variables
!-----------------------------------------------------------------------
      integer, parameter :: inst = 1
      integer, parameter :: avrg = 2
      real, parameter    :: m2km = 1.e-3
      real, parameter    :: mw_dusti = 1./34.

      integer  ::  i, k, m, n, hndx, file
      integer  ::  spc_ndx
      integer  ::  ltrop(plonl)                                          ! tropopause vertical index
      real     ::  sunon
      real     ::  sunoff
      real     ::  caldayn                                               ! day of year at end of time step
      real     ::  invariants(plonl,plev,max(1,nfs))                     ! invariant concentations (/cm^3)
      real     ::  group_ratios(plonl,plev,max(1,grpcnt))
      real     ::  group_vmr(plonl,plev,max(1,grpcnt))
      real     ::  col_dens(plonl,plev,max(1,ncol_abs))                  ! column densities (molecules/cm^2)
      real     ::  col_delta(plonl,0:plev,max(1,ncol_abs))               ! layer column densities (molecules/cm^2)
      real     ::  het_rates(plonl,plev,max(1,hetcnt))                   ! washout removal rate constants (1/s)
      real     ::  extfrc(plonl,plev,max(1,extcnt))                      ! extraneous forcing (/cm^3/s)
      real     ::  vmr(plonl,plev,pcnstm1)                               ! xported species ( vmr )
      real     ::  reaction_rates(plonl,plev,rxntot)                     ! reaction rate constants
      real     ::  nas(plonl,plev,max(1,grpcnt))                         ! non-advected species( mmr )
      real     ::  h2ovmr(plonl,plev)                                    ! water vapor volume mixing ratio
      real     ::  mbar(plonl,plev)                                      ! mean wet atmospheric mass ( amu )
      real     ::  zmida(plonl,plev)                                     ! abs midpoint geopotential (km)
      real     ::  zmidr(plonl,plev)                                     ! rel midpoint geopotential (km)
      real     ::  sulfate(plonl,plev)                                   ! sulfate aerosols concentration
      real     ::  strato_sad(plonl,plev)                                ! stratospheric sad (1/cm)
      real     ::  fracday(plonl)                                        ! day inidicatior: 1 if point in day, 0 if point at night
      real     ::  sad(plonl,plev)                                       ! total trop. sad (cm^2/cm^3)
      real     ::  dust(plonl,plev,nqdust)                               ! dust aerosols concentration (mol/mol)
      real     ::  relhum(plonl,plev)                                    ! relative humidity
      real     ::  satv(plonl,plev)                                      ! wrk array for relative humidity
      real     ::  satq(plonl,plev)                                      ! wrk array for relative humidity
      real     ::  wrk(plonl,plev)                                       ! wrk array
      real     ::  zinta(plonl,plevp)                                    ! interface geopotential (km)
      real     ::  zintr(plonl,plevp)                                    ! interface geopotential (km)
      real     ::  zen_angle(plonl)
      real     ::  loc_angle(plonl)
      real     ::  albs(plonl)
      real     ::  wd_col(plonl,1)
      real     ::  dt_diag(plonl,7)
      logical  ::  polar_night
      logical  ::  polar_day
      logical  ::  group_write(moz_file_cnt)
      character(len=32) :: fldname
      integer, parameter :: wd_num = 14
      character(len=32)  :: wd_name
      character(len=8)   :: wd_list(wd_num) = (/ 'HNO3    ','HO2NO2  ','ONIT    ','ONITR   ' &
                                                ,'NH3     ','NH4     ','NH4NO3  ','SA1     ' &
                                                ,'SA2     ','SA3     ','SA4     ','SO4     ' &
                                                ,'SO2     ','SOA     ' /)
      integer            :: wd_index, length

!-----------------------------------------------------------------------      
!        ... xform geopotential height from m to km 
!-----------------------------------------------------------------------      
      do k = 1,plev
         zmida(:,k) = m2km * zma(:,k)
         zmidr(:,k) = m2km * zm(:,k)
         zinta(:,k) = m2km * zia(:,k)
         zintr(:,k) = m2km * zi(:,k)
      end do
      zinta(:,plevp) = m2km * zia(:,plevp)
      zintr(:,plevp) = m2km * zi(:,plevp)
      caldayn = caldayr( ncdate, ncsec )

      dt_diag(:,:) = 0.
      if( phtcnt /= 0 ) then
!-----------------------------------------------------------------------      
!        ... calculate parameters for diurnal geometry
!-----------------------------------------------------------------------      
         call diurnal_geom( ip, lat, caldayn, polar_night, polar_day, &
                            sunon, sunoff, loc_angle,  zen_angle, plonl )
      end if
!-----------------------------------------------------------------------      
!        ... initialize xform between mass and volume mixing ratios
!-----------------------------------------------------------------------      
      call inti_mr_xform( sh, mbar, plonl )
!-----------------------------------------------------------------------      
!        ... xform from mmr to vmr
!-----------------------------------------------------------------------      
      call mmr2vmr( vmr, mmr, mbar, plonl )
!-----------------------------------------------------------------------      
!   	... find the tropopause location
!-----------------------------------------------------------------------      
      call tropp_lev( lat, ip, zmida, pmid, tfld, &
                      caldayn, ltrop, plonl, hstout=.true. )
      if( has_h2o ) then
!-----------------------------------------------------------------------      
!        ... xform water vapor from mmr to vmr and set upper bndy values
!-----------------------------------------------------------------------      
         call adjh2o( h2ovmr, sh, mbar, vmr, plonl )
         call set_ub_h2o( lat, ip, h2ovmr, pmid, zmida, &
                          tfld, ltrop, caldayn, plonl )
      else
         h2ovmr(:,:) = 0.
      end if
!-----------------------------------------------------------------
!	... compute the relative humidity
!-----------------------------------------------------------------
      call aqsat( tfld, pmid, satv, satq, plonl, &
                  plonl, plev, 1, plev )
      do k = 1,plev
         relhum(:,k) = .622 * h2ovmr(:,k) / satq(:,k)
         relhum(:,k) = max( 0.,min( 1.,relhum(:,k) ) )
      end do

      if( nfs > 0 ) then
!-----------------------------------------------------------------------      
!        ... set the "invariants"
!-----------------------------------------------------------------------      
         call setinv( invariants, tfld, h2ovmr, pmid, plonl, &
                      lat, ip )
      end if
has_col_density : &
      if( ncol_abs > 0 .and. phtcnt > 0 ) then
!-----------------------------------------------------------------------      
!        ... xform family ox assuming that all ox is o3
!-----------------------------------------------------------------------      
          if( ox_grp_ndx > 0 ) then
             o3_ndx = get_grp_mem_ndx( 'O3' )
             if( o3_ndx > 0 ) then
               vmr(:,:,ox_ndx) = mbar(:,:) * mmr(:,:,ox_ndx) / nadv_mass(o3_ndx)
             end if
          end if
!-----------------------------------------------------------------------      
!        ... set the column densities at the upper boundary
!-----------------------------------------------------------------------      
         call set_ub_col( col_delta, vmr, invariants, pint(:,1), pdel, &
                          plonl, lat )
      end if has_col_density
!-----------------------------------------------------------------------      
!        ... set dust concentrations
!-----------------------------------------------------------------------      
      if( use_dust ) then
         call dust_interp( lat, ip, pmid, caldayn, dust, plonl )
         do k = 1,plev
            wrk(:,k) = mbar(:,k) * mw_dusti
         end do
!-----------------------------------------------------------------------      
!        ... convert from mass to volume mixing ratio
!-----------------------------------------------------------------------      
         do m = 1,nqdust
            do k = 1,plev
               dust(:,k,m) = dust(:,k,m) * wrk(:,k)
            end do
            do file = 1,moz_file_cnt
               call outfld( trim(dust_nm(m)), dust(1,1,m), plonl, ip, lat, file )
            end do
         end do
      end if


      reaction_rates(:,:,:) = 0.
has_gas_phase_rxts : &
      if( gascnt > 0 ) then
!-----------------------------------------------------------------------      
!       ...  Set rates for "tabular" and user specified reactions
!-----------------------------------------------------------------------      
         call setrxt( reaction_rates, tfld, invariants(1,1,indexm), plonl )
         call sulf_interp( lat, ip, pmid, caldayn, sulfate, plonl )
         call strato_sad_set( pmid, lat, ip, ncdate, ncsec, strato_sad, plonl )
         call usrrxt( reaction_rates, tfld, invariants, h2ovmr, ps, &
                      pmid, invariants(1,1,indexm), sulfate, vmr, mmr, &
                      relhum, strato_sad, ltrop, lat, ip, plonl, sad )

!-----------------------------------------------------------------------      
!       ...  History output for instantaneous reaction rate constants
!-----------------------------------------------------------------------      
         do file = 1,moz_file_cnt
            if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(9,inst) > 0 ) then
               do m = 1,hfile(file)%histout_cnt(9,inst)
	          fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(9,inst)+m-1)
	          hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(9,inst)+m-1)
                  call outfld( fldname, reaction_rates(1,1,hndx+phtcnt), plonl, ip, lat, file )
               end do
            end if
!-----------------------------------------------------------------------      
!       ...  History output for time averaged reaction rate constants
!-----------------------------------------------------------------------      
            do m = 1,hfile(file)%histout_cnt(9,avrg)
	       fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(9,avrg)+m-1)
	       hndx = hfile(file)%timav_map(hfile(file)%histout_ind(9,avrg)+m-1)
               call outfld( fldname, reaction_rates(1,1,hndx+phtcnt), plonl, ip, lat, file )
            end do
         end do
         call adjrxt( reaction_rates, invariants, invariants(1,1,indexm), plnplv )
      end if has_gas_phase_rxts
has_photolysis : &      
      if( phtcnt > 0 ) then
!-----------------------------------------------------------------------
!        ... Compute the photolysis rates at time = t(n+1)
!-----------------------------------------------------------------------      
!-----------------------------------------------------------------------      
!     	... Calculate the surface albedo
!-----------------------------------------------------------------------      
	 call srfalb( lat, ip, albs, caldayn, tsurf, &
                      oro, snow,  plonl )
         if( polar_night ) then
            reaction_rates(:,:,1:phtcnt) = 0.
            fracday(:)                   = 0.
         else
	    esfact = sundis( ncdate )
            if( ncol_abs > 0 ) then
!-----------------------------------------------------------------------      
!     	... Set the column densities
!-----------------------------------------------------------------------      
               call setcol( col_delta, col_dens, vmr, pdel, plonl )
            end if
!-----------------------------------------------------------------------      
!     	... Calculate the photodissociation rates
!-----------------------------------------------------------------------      
            if( .not. xactive_prates ) then
               call table_photo( reaction_rates, pmid, pdel, tfld, zmida, &
                                 col_dens, zen_angle, albs, cwat, cldfr, &
                                 esfact, vmr, invariants, plonl )
            else
                call xactive_photo( reaction_rates, vmr, tfld, cwat, cldfr, &
		                   pmid, zmidr, col_dens, zen_angle, albs, &
		                   invariants(1,1,indexm), ps, tsurf, lat, ip, &
		                   ncdate, ncsec, sunon, sunoff, esfact, &
                                   relhum, dust, dt_diag, fracday, plonl )
            end if
         end if
!-----------------------------------------------------------------------
! 	... optical depths diagnostics
!-----------------------------------------------------------------------
         do file = 1,sim_file_cnt
            call outfld( 'DTCBS', dt_diag, plonl, ip, lat, file )
            call outfld( 'DTOCS', dt_diag(:,2), plonl, ip, lat, file )
            call outfld( 'DTSO4', dt_diag(:,3), plonl, ip, lat, file )
            call outfld( 'DTANT', dt_diag(:,4), plonl, ip, lat, file )
            call outfld( 'DTSAL', dt_diag(:,5), plonl, ip, lat, file )
            call outfld( 'DTDUST', dt_diag(:,6), plonl, ip, lat, file )
            call outfld( 'DTTOTAL', dt_diag(:,7), plonl, ip, lat, file )
            call outfld( 'FRACDAY', fracday, plonl, ip, lat, file )
         end do
!-----------------------------------------------------------------------      
!       ...  History output for instantaneous photo rate constants
!-----------------------------------------------------------------------      
         do file = 1,moz_file_cnt
            if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(8,inst) > 0 ) then
               do m = 1,hfile(file)%histout_cnt(8,inst)
                  fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(8,inst)+m-1)
	          hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(8,inst)+m-1)
                  call outfld( fldname, reaction_rates(1,1,hndx), plonl, ip, lat, file )
               end do
            end if
!-----------------------------------------------------------------------      
!       ...  History output for time averaged photo rate constants
!-----------------------------------------------------------------------      
            do m = 1,hfile(file)%histout_cnt(8,avrg)
               fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(8,avrg)+m-1)
               hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(8,avrg)+m-1)
               call outfld( fldname, reaction_rates(1,1,hndx), plonl, ip, lat, file )
            end do
         end do
!-----------------------------------------------------------------------      
!     	... Adjust the photodissociation rates
!-----------------------------------------------------------------------      
         call phtadj( reaction_rates, invariants, invariants(1,1,indexm), plnplv )
      end if has_photolysis
has_wet_removal : &
      if( hetcnt > 0 ) then
!-----------------------------------------------------------------------
!        ... Compute the heterogeneous rates at time = t(n+1)
!-----------------------------------------------------------------------      
         call sethet( het_rates, pmid, lat, zmida, phis, &
                      tfld, cmfdqr, nrain, nevapr, delt, &
                      invariants(1,1,indexm), vmr, plonl )

!-----------------------------------------------------------------------      
!       ...  history output for instantaneous wet removal rates
!-----------------------------------------------------------------------      
         do file = 1,moz_file_cnt
            if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(10,inst) > 0 ) then
               do m = 1,hfile(file)%histout_cnt(10,inst)
                  fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(10,inst)+m-1)
                  hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(10,inst)+m-1)
                  call outfld( fldname, het_rates(1,1,hndx), plonl, ip, lat, file )
               end do
            end if
!-----------------------------------------------------------------------      
!       ...  history output for time averaged wet removal rates
!-----------------------------------------------------------------------      
            do m = 1,hfile(file)%histout_cnt(10,avrg)
               fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(10,avrg)+m-1)
               hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(10,avrg)+m-1)
               call outfld( fldname, het_rates(1,1,hndx), plonl, ip, lat, file )
            end do
         end do
!-----------------------------------------------------------------------      
!         ... output instantaneous integrated wet deposition
!-----------------------------------------------------------------------      
         do file = 1,moz_file_cnt
            if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(18,inst) > 0 ) then
               do m = 1,hfile(file)%histout_cnt(18,inst)
                  fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(18,inst)+m-1)
                  hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(18,inst)+m-1)
                  n       = mod( hndx,1000 )
                  wd_index = hndx/1000
                  wd_col(:,1) = 0.
                  do k = 1,plev
                     wd_col(:,1) = wd_col(:,1) &
                                 + (1. - exp( -het_rates(:,k,wd_index)*delt )) * mmr(:,k,n) * pdel(:,k)
                  end do
                  call outfld( fldname, wd_col(:,1)*rgrav, plonl, ip, lat, file )
               end do
            end if
!-----------------------------------------------------------------------      
!         ... output time averaged integrated wet deposition
!-----------------------------------------------------------------------      
            do m = 1,hfile(file)%histout_cnt(18,avrg)
               fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(18,avrg)+m-1)
               hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(18,avrg)+m-1)
               n       = mod( hndx,1000 )
               wd_index = hndx/1000
               wd_col(:,1) = 0.
               do k = 1,plev
                  wd_col(:,1) = wd_col(:,1) &
                              + (1. - exp( -het_rates(:,k,wd_index)*delt )) * mmr(:,k,n) * pdel(:,k)
               end do
               call outfld( fldname, wd_col(:,1)*rgrav, plonl, ip, lat, file )
            end do
         end do
      end if has_wet_removal
has_external_frcing : &
      if( extcnt > 0 ) then
!-----------------------------------------------------------------------
!        ... compute the extraneous frcing at time = t(n+1)
!-----------------------------------------------------------------------      
         call setext( extfrc, lat, ip, zintr, cldtop, &
                      ncdate, ncsec, plonl )
!-----------------------------------------------------------------------      
!       ...  History output for instantaneous external forcing rates
!-----------------------------------------------------------------------      
         do file = 1,moz_file_cnt
            if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(11,inst) > 0 ) then
               do m = 1,hfile(file)%histout_cnt(11,inst)
                  fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(11,inst)+m-1)
                  hndx = hfile(file)%inst_map(hfile(file)%histout_ind(11,inst)+m-1)
                  call outfld( fldname, extfrc(1,1,hndx), plonl, ip, lat, file )
               end do
            end if
!-----------------------------------------------------------------------      
!       ...  History output for time averaged external forcing rates
!-----------------------------------------------------------------------      
            do m = 1,hfile(file)%histout_cnt(11,avrg)
               fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(11,avrg)+m-1)
               hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(11,avrg)+m-1)
               call outfld( fldname, extfrc(1,1,hndx), plonl, ip, lat, file )
            end do
         end do
         do m = 1,max(1,extcnt)
            if( m /= synoz_ndx ) then
               do k = 1,plev
                  extfrc(:,k,m) = extfrc(:,k,m) / invariants(:,k,indexm)
               end do
            end if
         end do
      end if has_external_frcing
has_groups : &
      if( grpcnt > 0 ) then
!-----------------------------------------------------------------------
!        ... Set the group ratios
!-----------------------------------------------------------------------      
         call set_grp_ratios( group_ratios, reaction_rates, vmr, mmr, nas, &
                              mbar, invariants, plonl )
!-----------------------------------------------------------------------
!     	... Modify the reaction rate of any reaction
!           with group member or proportional reactant(s)
!-----------------------------------------------------------------------
         call rxt_mod( reaction_rates, het_rates, group_ratios, plnplv )
      end if has_groups

!=======================================================================
!        ... Call the class solution algorithms
!=======================================================================
      if( clscnt1 > 0 .and. rxntot > 0 ) then
!-----------------------------------------------------------------------
!	... Solve for "explicit" species
!-----------------------------------------------------------------------
         call exp_sol( vmr, reaction_rates, het_rates, extfrc, nstep, &
                       delt, invariants(1,1,indexm), pdel, lat, ip, &
                       plonl, plnplv )
      end if
      if( clscnt4 > 0 .and. rxntot > 0 ) then
!-----------------------------------------------------------------------
!	... Solve for "Implicit" species
!-----------------------------------------------------------------------
         call imp_sol( vmr, reaction_rates, het_rates, extfrc, nstep, &
                       delt, invariants(1,1,indexm), pdel, lat, ip, &
                       plonl, plnplv )
      end if
      if( clscnt5 > 0 .and. rxntot > 0 ) then
!-----------------------------------------------------------------------
!	... Solve for "Rodas" species
!-----------------------------------------------------------------------
         call rodas_sol( vmr, reaction_rates, het_rates, extfrc, nstep, &
                         delt, invariants(1,1,indexm), lat, ip, plonl, plnplv )
      end if
      if( has_sox ) then
         call setsox( pmid, lat, delt, ip, tfld, &
                      sh, nrain, nevapr, cmfdqr, cwat, &
                      invariants(1,1,indexm), vmr, plonl )
      end if
      if( has_seasalt ) then
         call set_seasalt( pmid, lat, delt, ip, tfld, &
                           um1, vm1, relhum, pdel, zma, &
                           invariants(1,1,indexm), vmr, oro, plonl )
      end if
      if( has_soa ) then
         call setsoa( delt, reaction_rates, tfld, vmr, invariants(1,1,indexm), &
                      lat, ip, plonl )
      end if
      if( has_aerosols ) then
         call aerosols_formation( lat, ip, pmid, tfld, relhum, &
                                  invariants(1,1,indexm), vmr, plonl )
      end if

!-----------------------------------------------------------------------      
!       ...  history output for instantaneous photo rates
!-----------------------------------------------------------------------      
         do file = 1,moz_file_cnt
            if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(21,inst) > 0 ) then
               do m = 1,hfile(file)%histout_cnt(21,inst)
                  fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(21,inst)+m-1)
	          hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(21,inst)+m-1)
	          do k = 1,plev
	             wrk(:,k) = reaction_rates(:,k,hndx)*invariants(:,k,indexm)
                     spc_ndx  = rxt_rate_map(hndx,1)
                     if( spc_ndx > 0 ) then
	                wrk(:,k) = wrk(:,k) * vmr(:,k,spc_ndx)
                     end if
	          end do
                  call outfld( fldname, wrk, plonl, ip, lat, file )
               end do
            end if
!-----------------------------------------------------------------------      
!       ...  history output for time averaged photo rate constants
!-----------------------------------------------------------------------      
            do m = 1,hfile(file)%histout_cnt(21,avrg)
               fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(21,avrg)+m-1)
               hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(21,avrg)+m-1)
	       do k = 1,plev
	          wrk(:,k) = reaction_rates(:,k,hndx)*invariants(:,k,indexm)
                  spc_ndx  = rxt_rate_map(hndx,1)
                  if( spc_ndx > 0 ) then
	             wrk(:,k) = wrk(:,k) * vmr(:,k,spc_ndx)
                  end if
	       end do
               call outfld( fldname, wrk, plonl, ip, lat, file )
            end do
         end do
!-----------------------------------------------------------------------      
!       ...  history output for instantaneous reaction rates
!-----------------------------------------------------------------------      
         do file = 1,moz_file_cnt
            if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(22,inst) > 0 ) then
               do m = 1,hfile(file)%histout_cnt(22,inst)
                  fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(22,inst)+m-1)
	          hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(22,inst)+m-1) + phtcnt
	          do k = 1,plev
	             wrk(:,k) = reaction_rates(:,k,hndx)*invariants(:,k,indexm)
                     do n = 1,2
                        spc_ndx = rxt_rate_map(hndx,n)
                        if( spc_ndx > 0 ) then
	                   wrk(:,k) = wrk(:,k) * vmr(:,k,spc_ndx)
                        else
                           exit
                        end if
	             end do
	          end do
                  call outfld( fldname, wrk, plonl, ip, lat, file )
               end do
            end if
!-----------------------------------------------------------------------      
!       ...  history output for time averaged reaction rates
!-----------------------------------------------------------------------      
            do m = 1,hfile(file)%histout_cnt(22,avrg)
               fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(22,avrg)+m-1)
               hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(22,avrg)+m-1) + phtcnt
	       do k = 1,plev
	          wrk(:,k) = reaction_rates(:,k,hndx)*invariants(:,k,indexm)
                  do n = 1,2
                     spc_ndx = rxt_rate_map(hndx,n)
                     if( spc_ndx > 0 ) then
	                wrk(:,k) = wrk(:,k) * vmr(:,k,spc_ndx)
                     else
                        exit
                     end if
	          end do
	       end do
               call outfld( fldname, wrk, plonl, ip, lat, file )
            end do
         end do
!-----------------------------------------------------------------------      
!         ... Check for negative values and reset to zero
!-----------------------------------------------------------------------      
      call negtrc( lat, 'After chemistry ', vmr, plonl )
!-----------------------------------------------------------------------      
!         ... Set values near upper boundary
!-----------------------------------------------------------------------      
      if( fbc_cnt(2) > 0 ) then
         call set_ub_vals( lat, ip, vmr, pmid, pint, &
                           zmida, tfld, ltrop, caldayn, plonl )
      end if
!-----------------------------------------------------------------------      
!         ... set specified lower bndy values
!-----------------------------------------------------------------------      
      call flbc_set( lat, ip, ncdate, ncsec, vmr, plonl )
!-----------------------------------------------------------------------      
!         ... Output instantaneous "wet" advected volume mixing
!-----------------------------------------------------------------------      
      do file = 1,moz_file_cnt
         if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(1,inst) > 0 ) then
            do m = 1,hfile(file)%histout_cnt(1,inst)
               fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(1,inst)+m-1)
               hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(1,inst)+m-1)
               call outfld( fldname, vmr(1,1,hndx), plonl, ip, lat, file )
            end do
         end if
!-----------------------------------------------------------------------      
!         ... Output time averaged "wet" advected volume mixing ratios
!-----------------------------------------------------------------------      
         do m = 1,hfile(file)%histout_cnt(1,avrg)
            fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(1,avrg)+m-1)
            hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(1,avrg)+m-1)
            call outfld( fldname, vmr(1,1,hndx), plonl, ip, lat, file )
         end do
      end do
      do file = 1,moz_file_cnt
!-----------------------------------------------------------------------      
!         ... output local time species
!-----------------------------------------------------------------------      
         do m = 1,hfile(file)%histout_cnt(20,avrg)
            fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(20,avrg)+m-1)
            if( index( fldname, '_emis' ) == 0 ) then
               hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(20,avrg)+m-1)
               call outfld( fldname, vmr(1,1,hndx), plonl, ip, lat, file )
            end if
         end do
      end do
!-----------------------------------------------------------------------      
!         ... Output instantaneous "wet" non-advected volume mixing
!-----------------------------------------------------------------------      
      group_write(:moz_file_cnt) = hfile(:moz_file_cnt)%wrhstts .and. &
                                   hfile(:moz_file_cnt)%histout_cnt(2,inst) > 0
      if( any( group_write(:moz_file_cnt) ) .or. &
          any( hfile(:moz_file_cnt)%histout_cnt(2,avrg) > 0 ) ) then
         call mak_grp_vmr( vmr, group_ratios(1,1,1), group_vmr(1,1,1), plonl )
      end if
      do file = 1,moz_file_cnt
         if( hfile(file)%wrhstts .and. hfile(file)%histout_cnt(2,inst) > 0 ) then
            do m = 1,hfile(file)%histout_cnt(2,inst)
               fldname = hfile(file)%hist_inst(hfile(file)%histout_ind(2,inst)+m-1)
               hndx    = hfile(file)%inst_map(hfile(file)%histout_ind(2,inst)+m-1)
               call outfld( fldname, group_vmr(1,1,hndx), plonl, ip, lat, file )
            end do
         end if
!-----------------------------------------------------------------------      
!         ... Output time averaged "wet" non-advected volume mixing ratios
!-----------------------------------------------------------------------      
         do m = 1,hfile(file)%histout_cnt(2,avrg)
            fldname = hfile(file)%hist_timav(hfile(file)%histout_ind(2,avrg)+m-1)
            hndx    = hfile(file)%timav_map(hfile(file)%histout_ind(2,avrg)+m-1)
            call outfld( fldname, group_vmr(1,1,hndx), plonl, ip, lat, file )
         end do
      end do
!-----------------------------------------------------------------------      
!         ... check for nox,noy output
!-----------------------------------------------------------------------      
      if( has_nox ) then
         do k = 1,plev
            wrk(:,k) = vmr(:,k,no_ndx) + vmr(:,k,no2_ndx)
         end do
         do file = 1,moz_file_cnt
            call outfld( 'NOX', wrk, plonl, ip, lat, file )
         end do
      end if
      if( has_noy ) then
         do k = 1,plev
            wrk(:,k) = vmr(:,k,no_ndx) + vmr(:,k,no2_ndx) + vmr(:,k,no3_ndx) &
                       + vmr(:,k,hno3_ndx) + vmr(:,k,ho2no2_ndx) + 2.*vmr(:,k,n2o5_ndx) &
                       + vmr(:,k,pan_ndx) + vmr(:,k,onit_ndx) + vmr(:,k,mpan_ndx) &
                       + vmr(:,k,onitr_ndx) + vmr(:,k,isopno3_ndx)
         end do
         do file = 1,moz_file_cnt
            call outfld( 'NOY', wrk, plonl, ip, lat, file )
         end do
      end if

      do file = 1,moz_file_cnt
       call outfld( 'RELHUM', relhum,  plonl,  ip, lat, file )
       call outfld( 'H2O', h2ovmr,  plonl,  ip, lat, file )
       call outfld( 'SAD', sad,  plonl,  ip, lat, file )
      enddo

!-----------------------------------------------------------------------      
!         ... Xform from vmr to mmr
!-----------------------------------------------------------------------      
      call vmr2mmr( vmr, mmr, nas, group_ratios, mbar, plonl )

      end subroutine chemdr

      end module mo_chemdr
