
      module mo_extfrc
!---------------------------------------------------------------
! 	... insitu forcing module
!---------------------------------------------------------------

      use mo_grid,   only : plon, plat, plev
      use m_types,   only : time_ramp
      use chem_mods, only : pcnstm1
      use mo_calendar, only : diffdat

      implicit none

      type :: forcing
         integer           :: frc_ndx
         integer           :: gndx
         integer           :: ntimes
         integer           :: tim_ndx(2)
         integer           :: jlim(2)
         integer           :: nlon
         integer           :: nlev
         integer, pointer  :: dates(:)
         integer, pointer  :: secs(:)
         real              :: mw
         character(len=80) :: filename
         real, pointer     :: times(:)
         real, pointer     :: levi(:)
         real, pointer     :: src(:,:,:,:,:)
         integer           :: xactive_type_cnt
         character(len=32) :: xactive_type(5)
         character(len=8)  :: species
         character(len=8)  :: units
      end type forcing

      private
      public  :: extfrc_inti
      public  :: extfrc_set
      public  :: extfrc_timestep_init

      save

      integer, parameter :: time_span = 12

      integer :: ntimes
      integer :: extfrc_cnt
      logical :: has_extfrc(pcnstm1) = .false.
      character(len=80) ::   filename
      character(len=80) ::   lpath
      character(len=80) ::   mspath
      character(len=30) ::   extfrc_hres                ! forcing file horiz resolution

      type(forcing), allocatable  :: forcings(:)
      type(time_ramp)             :: extfrc_timing

      contains

      subroutine extfrc_inti( plonl, platl, pplon, extfrc_timing_in, ncdate, ncsec )
!-----------------------------------------------------------------------
! 	... initialize the surface forcings
!-----------------------------------------------------------------------

      use m_tracname,    only : tracnam
      use mo_chem_utls,  only : get_extfrc_ndx
      use chem_mods,     only : frc_from_dataset
      use mo_constants,  only : d2r
      use mo_control,    only : frc_flsp
      use mo_file_utils, only : open_netcdf_file
      use mo_charutl,    only : upcase
      use netcdf

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate
      integer, intent(in) :: ncsec
      type(time_ramp), intent(in) :: extfrc_timing_in

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: astat
      integer :: j, l, m, n                           ! Indices
      integer :: t1, t2
      integer :: ncid
      integer :: dimid
      integer :: varid
      integer :: yr, wrk_date, wrk_sec
      real    :: seq
      real    :: wrk_time
      character(len=8)  :: species
      character(len=8)  :: spc_name
      character(len=8)  :: time_type

!-----------------------------------------------------------------------
! 	... check forcings timing
!-----------------------------------------------------------------------
      extfrc_timing = extfrc_timing_in
      call upcase( extfrc_timing%type, time_type )
      extfrc_timing%type = time_type
      if( time_type /= 'SERIAL' .and. time_type /= 'CYCLICAL' &
                                .and. time_type /= 'FIXED' ) then
         write(*,*) 'extfrc_inti: time type ',trim(time_type),' is not SERIAL,CYCLICAL, or FIXED'
         call endrun
      end if

      wrk_sec  = ncsec
      if( time_type == 'SERIAL' ) then
	 wrk_date = ncdate + extfrc_timing%yr_offset*10000
      else if( time_type == 'CYCLICAL' ) then
         wrk_date = (extfrc_timing%date/10000)*10000 + mod(ncdate,10000)
      else
         wrk_date = extfrc_timing%date
         wrk_sec  = 0
      end if
      wrk_time = diffdat( 0, 0, wrk_date, wrk_sec )
      write(*,*) 'extfrc_inti: wrk_date,wrk_sec,wrk_time = ',wrk_date,wrk_sec,wrk_time

!-----------------------------------------------------------------------
! 	... species has insitu forcing ?
!-----------------------------------------------------------------------
      do m = 1,pcnstm1
         spc_name = trim( tracnam(m) )
         n        = get_extfrc_ndx( spc_name )
         if( n > 0 ) then
            has_extfrc(m) = frc_from_dataset(n)
         end if
      end do

      write(*,*) ' '
      extfrc_cnt = count( has_extfrc(:) )
      if( extfrc_cnt > 0 ) then
         write(*,*) 'Species with insitu forcings'
         do m = 1,pcnstm1
            if( has_extfrc(m) ) then
               write(*,*) trim(tracnam(m))
            end if
         end do
      else
         write(*,*) 'There are no species with insitu forcings'
         return
      end if
      write(*,*) ' '

!-----------------------------------------------------------------------
! 	... allocate forcings type array
!-----------------------------------------------------------------------
      allocate( forcings(extfrc_cnt), stat=astat )
      if( astat/= 0 ) then
	 write(*,*) 'extfrc_inti: failed to allocate forcings array; error = ',astat
	 call endrun
      end if

!-----------------------------------------------------------------------
! 	... diagnostics setup
!-----------------------------------------------------------------------
      extfrc_hres = frc_flsp%hor_res
      if( extfrc_hres(1:1) /= '.' ) then
         extfrc_hres = '.' // extfrc_hres
      end if
      lpath  = frc_flsp%local_path
      mspath = frc_flsp%remote_path

!-----------------------------------------------------------------------
! 	... setup the forcing type array
!-----------------------------------------------------------------------
      n = 0
species_loop : &
      do m = 1,pcnstm1
has_forcing : &
         if( has_extfrc(m) ) then
            spc_name = trim( tracnam(m) )
            n        = n + 1
!-----------------------------------------------------------------------
! 	... default settings
!-----------------------------------------------------------------------
            forcings(n)%frc_ndx          = get_extfrc_ndx( spc_name )
            forcings(n)%xactive_type_cnt = 0
            forcings(n)%species          = spc_name
            forcings(n)%filename         = 'extfrc.' // trim(spc_name) // trim(extfrc_hres) // '.nc'
         end if has_forcing
      end do species_loop

!-----------------------------------------------------------------------
! 	... diagnostics
!-----------------------------------------------------------------------
      write(*,*) ' '
      write(*,*) 'extfrc_inti: diagnostics'
      write(*,*) ' '
      write(*,*) 'extfrc timing specs'
      write(*,*) 'type = ',extfrc_timing%type
      if( time_type == 'SERIAL' ) then
         write(*,*) 'year offset = ',extfrc_timing%yr_offset
      else if( time_type == 'CYCLICAL' ) then
         write(*,*) 'year = ',extfrc_timing%date/10000
      else
         write(*,*) 'date = ',extfrc_timing%date
      end if
      write(*,*) ' '
      write(*,*) 'there are ',extfrc_cnt,' species with external forcing files'
      do m = 1,extfrc_cnt
         write(*,*) ' '
         write(*,*) 'forcing type ',m
         write(*,*) 'species = ',trim(forcings(m)%species)
         write(*,*) 'frc ndx = ',forcings(m)%frc_ndx
         write(*,*) 'filename= ',trim(forcings(m)%filename)
      end do
      write(*,*) ' '

!-----------------------------------------------------------------------
! 	... get timing information, allocate arrays, and read in dates
!-----------------------------------------------------------------------
frcing_loop : &
      do m = 1,extfrc_cnt
         ncid = open_netcdf_file( forcings(m)%filename, lpath, mspath )
         call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid ), &
                            'extfrc_inti: Failed to find time dimension for species ' // trim(forcings(m)%species) )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, forcings(m)%ntimes ), &
                            'extfrc_inti: Failed to get length of dimension date for species ' // &
                            trim(forcings(m)%species) )
         allocate( forcings(m)%dates(forcings(m)%ntimes), &
                   forcings(m)%secs(forcings(m)%ntimes),stat=astat )
         if( astat/= 0 ) then
	    write(*,*) 'extfrc_inti: failed to allocate dates array for species ',trim(forcings(m)%species),'; error = ',astat
            call endrun
         end if
         allocate( forcings(m)%times(forcings(m)%ntimes),stat=astat )
         if( astat/= 0 ) then
            write(*,*) 'extfrc_inti: failed to allocate times array for species ',trim(forcings(m)%species),'; error = ',astat
	    call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'date', varid ), &
                            'extfrc_inti: Failed to find dat lat for species ' // trim(forcings(m)%species) )
         call handle_ncerr( nf_get_var_int( ncid, varid, forcings(m)%dates ), &
                            'extfrc_inti: Failed to read variable dates for species ' // trim(forcings(m)%species) )
         astat = nf_inq_varid( ncid, 'datesec', varid )
         if( astat == nf_noerr ) then
           call handle_ncerr( nf_get_var_int( ncid, varid, forcings(m)%secs ), &
                              'extfrc_inti: Failed to read variable secs for species ' // trim(forcings(m)%species) )
         else
           forcings(m)%secs(:) = 0
         endif
         do n = 1,forcings(m)%ntimes
            forcings(m)%times(n) = diffdat( 0, 0, forcings(m)%dates(n), forcings(m)%secs(n) )
         end do
	 if( time_type /= 'CYCLICAL' ) then
            if( wrk_time < forcings(m)%times(1) .or. wrk_time > forcings(m)%times(forcings(m)%ntimes) ) then
               write(*,*) 'extfrc_inti: extfrc time out of bounds for dataset = ',trim(forcings(m)%filename)
               call endrun
            end if
            do n = 2,forcings(m)%ntimes
               if( wrk_time <= forcings(m)%times(n) ) then
                  exit
               end if
            end do
            forcings(m)%tim_ndx(1) = n - 1
         else
            yr = extfrc_timing%date/10000
            do n = 1,forcings(m)%ntimes
               if( yr == forcings(m)%dates(n)/10000 ) then
                  exit
               end if
            end do
            if( n >= forcings(m)%ntimes ) then
               write(*,*) 'extfrc_inti: extfrc time out of bounds for dataset = ',trim(forcings(m)%filename)
               call endrun
	    end if
            forcings(m)%tim_ndx(1) = n
         end if
         select case( time_type )
            case( 'FIXED' )
               forcings(m)%tim_ndx(2) = n
            case( 'CYCLICAL' )
               do n = forcings(m)%tim_ndx(1),forcings(m)%ntimes
                  if( yr /= forcings(m)%dates(n)/10000 ) then
                     exit
                  end if
               end do
               forcings(m)%tim_ndx(2) = n - 1
	       if( (forcings(m)%tim_ndx(2) - forcings(m)%tim_ndx(1)) < 2 ) then
                  write(*,*) 'extfrc_inti: cyclical forcings require at least two time points'
                  call endrun
               end if
            case( 'SERIAL' )
               forcings(m)%tim_ndx(2) = min( forcings(m)%ntimes,forcings(m)%tim_ndx(1) + time_span )
         end select
         t1 = forcings(m)%tim_ndx(1)
         t2 = forcings(m)%tim_ndx(2)
         write(*,*) ' '
         write(*,*) 'frcing time cnt = ',forcings(m)%ntimes
         write(*,*) 'frcing times'
         write(*,'(10i10)') forcings(m)%dates(:)
         write(*,'(1p,5g15.7)') forcings(m)%times(:)
         write(*,*) 'frcing time indicies = ',forcings(m)%tim_ndx(:)
         write(*,'(10i10)') forcings(m)%dates(t1:t2)
         write(*,*) ' '
!-----------------------------------------------------------------------
! 	... readin the insitu forcings
!-----------------------------------------------------------------------
         call extfrc_get( plonl, platl, pplon, ncid, forcings(m), .true. )
!-----------------------------------------------------------------------
! 	... close the file
!-----------------------------------------------------------------------
         call handle_ncerr( nf_close( ncid ), &
                            'extfrc_inti: Failed to close NetCDF file =' // trim(forcings(m)%filename) )
      end do frcing_loop

      end subroutine extfrc_inti

      subroutine extfrc_timestep_init( plonl, platl, pplon, ncdate, ncsec )
!-----------------------------------------------------------------------
!       ... check serial case for time span
!-----------------------------------------------------------------------

      use netcdf
      use mo_file_utils, only : open_netcdf_file

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate              ! present date (yyyymmdd)
      integer, intent(in) :: ncsec               ! seconds in present date

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer                     :: m
      integer                     :: nlev
      integer                     :: t1, t2, tcnt
      integer                     :: astat
      integer                     :: ncid
      real                        :: wrk_time

      if( extfrc_cnt < 1 .or. extfrc_timing%type /= 'SERIAL' ) then
         return
      end if
      wrk_time = diffdat( 0, 0, ncdate + extfrc_timing%yr_offset*10000, ncsec )

src_loop : &
      do m = 1,extfrc_cnt
         if( wrk_time > forcings(m)%times(forcings(m)%tim_ndx(2)) ) then
            tcnt = forcings(m)%tim_ndx(2) - forcings(m)%tim_ndx(1)
            forcings(m)%tim_ndx(1) = forcings(m)%tim_ndx(2)
            forcings(m)%tim_ndx(2) = min( forcings(m)%ntimes,forcings(m)%tim_ndx(1) + time_span )
            t1 = forcings(m)%tim_ndx(1)
            t2 = forcings(m)%tim_ndx(2)
!-----------------------------------------------------------------------
! 	... allocate extfrc array
!-----------------------------------------------------------------------
            if( associated( forcings(m)%src ) ) then
               deallocate( forcings(m)%src,stat=astat )
               if( astat/= 0 ) then
                  write(*,*) 'extfrc_timestep_init: failed to deallocate forcings src; error = ',astat
                  call endrun
               end if
            end if
            nlev = forcings(m)%nlev
            allocate( forcings(m)%src(plonl,nlev,platl,pplon,t1:t2),stat=astat )
            if( astat/= 0 ) then
               write(*,*) 'extfrc_timestep_init: failed to allocate forcings src; error = ',astat
               call endrun
            end if

            ncid = open_netcdf_file( forcings(m)%filename, lpath, mspath )
!-----------------------------------------------------------------------
! 	... readin the surface forcings
!-----------------------------------------------------------------------
            call extfrc_get( plonl, platl, pplon, ncid, forcings(m), .false. )
!-----------------------------------------------------------------------
! 	... close the file
!-----------------------------------------------------------------------
            call handle_ncerr( nf_close( ncid ), &
                               'extfrc_timestep_init: Failed to close NetCDF file =' // trim(forcings(m)%filename) )
         end if
      end do src_loop

      end subroutine extfrc_timestep_init

      subroutine extfrc_get( plonl, platl, pplon, ncid, forcings, initial )
!-----------------------------------------------------------------------
!       ... read surface forcings from NetCDF file
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_constants,  only : phi, lam, d2r
      use mo_file_utils, only : open_netcdf_file
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)           :: plonl
      integer, intent(in)           :: platl
      integer, intent(in)           :: pplon
      integer, intent(in)           :: ncid
      logical, intent(in)           :: initial
      type(forcing), intent(inout)  :: forcings

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer                     :: j, jl, ju, k, m               ! Indices
      integer                     :: t1, t2, tcnt
      integer                     :: ierr
      integer                     :: nlat, nlon, nlev
      integer                     :: nvars, ndims
      integer                     :: vid, attlen
      integer                     :: dimid_lat, dimid_lon, dimid_time, dimid_lev
      integer                     :: gndx
      integer                     :: jlim(2)
      integer, allocatable        :: dimid(:)
      real                        :: total, total_wrk
      real, allocatable           :: lat(:)
      real, allocatable           :: lon(:)
      real, allocatable           :: lev(:)
      real, allocatable           :: wrk(:,:,:,:)
      real                        :: wrk2d(plon,platl)
      character(len=nf_max_name)  :: varname
      character(len=100)          :: var_longname
      character(len=8)            :: species

      species = forcings%species
!-----------------------------------------------------------------------
!       ... get grid dimensions from file
!-----------------------------------------------------------------------
!           latitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'extfrc_get: Failed to find dimension lat for species ' // trim(species) )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                         'extfrc_get: Failed to get length of dimension lat for species ' // &
                         trim(species) )
!-----------------------------------------------------------------------
!           longitudes
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                         'extfrc_get: Failed to find dimension lon for species ' // trim(species) )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                         'extfrc_get: Failed to find dimension lon for species ' // trim(species) )
!-----------------------------------------------------------------------
!	... levels
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'altitude', dimid_lev ), &
                         'extfrc_get: Failed to find dimension altitude for species ' // trim(species) )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lev, nlev ), &
                         'extfrc_get: Failed to find dimension lev for species ' // trim(species) )

initialization : &
      if( initial ) then
!-----------------------------------------------------------------------
!           latitudes
!-----------------------------------------------------------------------
         allocate( lat(nlat), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: lat allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                            'extfrc_get: Failed to find variable lat for species ' // trim(species) )
         call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                            'extfrc_get: Failed to read variable lat for species ' // trim(species) )
         lat(:nlat) = lat(:nlat) * d2r
 
!-----------------------------------------------------------------------
!           longitudes
!-----------------------------------------------------------------------
         allocate( lon(nlon), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: lon allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                            'extfrc_get: Failed to find variable lon for species ' // trim(species) )
         call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                            'extfrc_get: Failed to read variable lon for species ' // trim(species) )
         lon(:nlon) = lon(:nlon) * d2r
!-----------------------------------------------------------------------
!       ... set up horizontal regridding
!-----------------------------------------------------------------------
         gndx = regrid_inti( nlat, plat, &
                             nlon, plon, &
                             lon,  lam, &
                             lat,  phi, &
                             0, platl, &
                             do_lons=.true., do_lats=.true. )
         deallocate( lat, lon, stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: Failed to deallocate lat,lon; ierr = ',ierr
            call endrun
         end if
         if( gndx /= 0 ) then
            jlim = regrid_lat_limits( gndx )
         else
            jlim = (/ base_lat+1,base_lat+platl /)
         end if
         write(*,'(1x,''extfrc_get: gndx='',i2,'', grid limits = '',2i4,'', jl,ju='',2i4)') gndx,jlim,base_lat+1,base_lat+platl
!-----------------------------------------------------------------------
!	... levels
!-----------------------------------------------------------------------
         allocate( forcings%levi(nlev+1), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: levi allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'altitude_int', vid ), &
                            'extfrc_get: Failed to find variable altitude_int for species ' // trim(species) )
         call handle_ncerr( nf_get_var_double( ncid, vid, forcings%levi ), &
                            'extfrc_get: Failed to read variable lev for species ' // trim(species) )
         forcings%gndx    = gndx
         forcings%jlim(:) = jlim(:)
         forcings%nlon    = nlon
         forcings%nlev    = nlev
!-----------------------------------------------------------------------
! 	... allocate src array
!-----------------------------------------------------------------------
         t1 = forcings%tim_ndx(1)
         t2 = forcings%tim_ndx(2)
         allocate( forcings%src(plonl,nlev,platl,pplon,t1:t2),stat=ierr )
         if( ierr/= 0 ) then
            write(*,*) 'extfrc_inti: failed to allocate forcings src; error = ',ierr
            call endrun
         end if
      else initialization
         gndx    = forcings%gndx
         jlim(:) = forcings%jlim(:)
         nlon    = forcings%nlon
         nlev    = forcings%nlev
      end if initialization

!-----------------------------------------------------------------------
! 	... times
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid_time ), &
                         'extfrc_get: Failed to find dimension time for species ' // trim(species) )

      call handle_ncerr( nf_inq_nvars( ncid, nvars ), &
                         'extfrc_get: Failed to get number of variables for species ' // &
                         trim(species) )

!-----------------------------------------------------------------------
!       ... initialize total forcings to zero
!-----------------------------------------------------------------------
      forcings%src(:,:,:,:,:) = 0.
!-----------------------------------------------------------------------
!       ... loop over file variables
!-----------------------------------------------------------------------
      t1 = forcings%tim_ndx(1)
      t2 = forcings%tim_ndx(2)
types_loop : &
      do vid = 1,nvars
         ierr = nf_inq_varname( ncid, vid, varname )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: Failed to get name of variable # ',vid, &
                '; species=' // trim(species)
            call endrun
         end if
         call handle_ncerr( nf_inq_varndims( ncid, vid, ndims ), &
                            'extfrc_get: Failed to get number of dimensions for ' // &
                            'variable ' // trim(varname) // ', species=' // trim(species) )
         if( ndims < 4 ) then
            cycle types_loop
         else if( ndims > 4 ) then
            write(*,*) 'extfrc_get: Skipping variable ', trim(varname),', ndims = ',ndims, &
                       ' , species=',trim(species)
            cycle types_loop
         end if
!-----------------------------------------------------------------------
!       ... check order of dimensions; must be (lon, lat, date).
!-----------------------------------------------------------------------
         allocate( dimid(ndims),stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: dimid allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_vardimid( ncid, vid, dimid ), &
                            'extfrc_get: Failed to get dimension IDs for variable ' // &
                            trim(varname)  // ', species=' // trim(forcings%species) )
         if( dimid(1) /= dimid_lon .or. dimid(2) /= dimid_lat  .or. &
             dimid(3) /= dimid_lev .or. dimid(4) /= dimid_time ) then
            write(*,*) 'extfrc_get: Dimensions in wrong order for variable ',trim(varname)
            write(*,*) '...      Expecting (lon, lat, lev, date)'
            call endrun
         end if
         deallocate( dimid,stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: Failed to deallocate dimid, ierr = ',ierr
            call endrun
         end if
!-----------------------------------------------------------------------
!       ... check for long_name of variable
!-----------------------------------------------------------------------
         var_longname = ' '
         ierr = nf_get_att_text( ncid, vid, 'long_name', var_longname )
         if( ierr /= 0 ) then
            var_longname = trim( varname )
         end if
!-----------------------------------------------------------------------
!       ... read data from this src type
!-----------------------------------------------------------------------
         tcnt = t2 - t1 + 1
         allocate( wrk(nlon,jlim(1):jlim(2),nlev,tcnt), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: wrk allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                                (/ 1, jlim(1), 1, t1/), &                     ! start
                                                (/ nlon, jlim(2)-jlim(1)+1, nlev, tcnt /), &  ! count
                                                wrk ), &
                            'extfrc_get: Failed to read variable ' // trim( varname ) // &
                            ', species=' // trim(species) )
         jl = base_lat + 1
         ju = base_lat + platl
         do m = t1,t2
            do k = 1,nlev
               call regrid_2d( wrk(:,jlim(1):jlim(2),k,m-t1+1), wrk2d, gndx, jl, ju, &
                               do_poles=.true. )
               forcings%src(:,k,:,:,m) = forcings%src(:,k,:,:,m) &
                          + reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
            end do
         end do
         deallocate( wrk,stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'extfrc_get: Failed to deallocate wrk, ierr = ',ierr
            call endrun
         end if
      end do types_loop

      end subroutine extfrc_get

      subroutine extfrc_set( lat, ip, ncdate, ncsec, frcing, &
                             zint, plonl )
!--------------------------------------------------------
!	... form the external forcing
!--------------------------------------------------------

      use chem_mods, only : extcnt
      use mo_grid,   only : plevp

      implicit none

!--------------------------------------------------------
!	... dummy arguments
!--------------------------------------------------------
      integer, intent(in) ::   plonl
      integer, intent(in) ::   lat                                ! latitude index
      integer, intent(in) ::   ip                                 ! longitude tile index
      integer, intent(in) ::   ncdate                             ! present date (yyyymmdd)
      integer, intent(in) ::   ncsec                              ! seconds in present date
      real, intent(in)    ::   zint(plonl,plevp)                  ! interface geopot above surface (km)
      real, intent(inout) ::   frcing(plonl,plev,max(1,extcnt))   ! insitu forcings (molec/cm^3/s)

!--------------------------------------------------------
!	... local variables
!--------------------------------------------------------
      integer  ::  i, m, n
      integer  ::  last, next
      integer  ::  wrk_date, wrk_sec
      integer  ::  tcnt
      integer  ::  nlev
      integer  ::  astat
      real     ::  dels
      real     ::  data_sum, model_sum
      real     ::  model_dz(plev)
      real, allocatable ::  data_dz(:)
      real     ::  wrk_time
      real     ::  model_z(plevp)
      real     ::  frc_model(plev,2)
      real, allocatable ::  frc_data(:)

      if( extcnt < 1 .or. extfrc_cnt < 1 ) then
         return
      end if

!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
      wrk_sec  = ncsec
      select case( extfrc_timing%type )
         case( 'SERIAL' )
            wrk_date = ncdate + extfrc_timing%yr_offset*10000
         case( 'CYCLICAL' )
            wrk_date = (extfrc_timing%date/10000)*10000 + mod( ncdate,10000 )
         case( 'FIXED' )
            wrk_date = extfrc_timing%date
            wrk_sec  = 0
      end select
      wrk_time = diffdat( 0, 0, wrk_date, wrk_sec )

!--------------------------------------------------------
!	... set non-zero forcings
!--------------------------------------------------------
src_loop : &
      do m = 1,extfrc_cnt
!--------------------------------------------------------
!	... set time interpolation factor
!--------------------------------------------------------
         if( extfrc_timing%type /= 'CYCLICAL' ) then
            do n = forcings(m)%tim_ndx(1)+1,forcings(m)%tim_ndx(2)
               if( wrk_time <= forcings(m)%times(n) ) then
                  last = n - 1
                  next = n
                  exit
               end if
            end do
            if( n > forcings(m)%ntimes ) then
               write(*,*) 'extfrc_set: interp time is out of bounds'
               call endrun
            end if
            dels = (wrk_time - forcings(m)%times(last)) &
                   /(forcings(m)%times(next) - forcings(m)%times(last))
         else
            tcnt = forcings(m)%tim_ndx(2) - forcings(m)%tim_ndx(1) + 1
            call moz_findplb( forcings(m)%times(forcings(m)%tim_ndx(1)), tcnt, wrk_time, n )
            if( n < tcnt ) then
               last = forcings(m)%tim_ndx(1) + n - 1
               next = last + 1
               dels = (wrk_time - forcings(m)%times(last)) &
                      /(forcings(m)%times(next) - forcings(m)%times(last))
            else
               next = forcings(m)%tim_ndx(1)
               last = forcings(m)%tim_ndx(2)
               dels = wrk_time - forcings(m)%times(last)
               if( dels < 0. ) then
                  dels = 365. + dels
               end if
               dels = dels/(365. + forcings(m)%times(next) - forcings(m)%times(last))
            end if
         end if

         dels = max( min( 1.,dels ),0. )
         n    = forcings(m)%frc_ndx
         nlev = forcings(m)%nlev
         allocate( frc_data(nlev),data_dz(nlev),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'extfrc_set: failed to allocate frc_data; error = ',astat
            call endrun
         end if
         do i = 1,plonl
            frc_data(:) = forcings(m)%src(i,:,lat,ip,last)
            model_z(:)  = zint(i,plevp:1:-1)
#ifdef DIAGS
               write(*,*) '======================================================'
               write(*,*) 'extfrc_set: diags at lat,ip, i = ',lat,ip,i
               write(*,*) 'model_z'
               write(*,'(1p5g15.7)') model_z
               write(*,*) 'data_z'
               write(*,'(1p5g15.7)') forcings(m)%levi
#endif
            call rebin( nlev, plev, forcings(m)%levi, model_z, frc_data, frc_model )
#ifdef DIAGS
               write(*,*) ' '
               write(*,*) 'src frcing'
               write(*,'(1p5g15.7)') frc_data
               write(*,*) 'model frcing'
               write(*,'(1p5g15.7)') frc_model(:,1)
               data_dz(:)  = forcings(m)%levi(2:nlev+1) - forcings(m)%levi(1:nlev)
               model_dz(:) = model_z(2:plevp) - model_z(1:plev)
               data_sum  = dot_product( frc_data,data_dz )
               model_sum = dot_product( frc_model(:,1),model_dz )
               write(*,*) 'data_sum, model_sum = ',data_sum, model_sum
               write(*,*) '======================================================'
               call endrun
#endif
            frc_data(:) = forcings(m)%src(i,:,lat,ip,next)
            call rebin( nlev, plev, forcings(m)%levi, model_z, frc_data, frc_model(1,2) )
            frcing(i,plev:1:-1,n) = frc_model(:,1) + dels * (frc_model(:,2) - frc_model(:,1))
         end do
         deallocate( frc_data,data_dz )
      end do src_loop

      end subroutine extfrc_set

      subroutine rebin( nsrc, ntrg, src_x, trg_x, src, trg )
!---------------------------------------------------------------
!	... rebin src to trg
!---------------------------------------------------------------

      implicit none

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in)   :: nsrc                  ! dimension source array
      integer, intent(in)   :: ntrg                  ! dimension target array
      real, intent(in)      :: src_x(nsrc+1)         ! source coordinates
      real, intent(in)      :: trg_x(ntrg+1)         ! target coordinates
      real, intent(in)      :: src(nsrc)             ! source array
      real, intent(out)     :: trg(ntrg)             ! target array

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: i, l
      integer  :: si, si1
      integer  :: sil, siu
      real     :: y
      real     :: sl, su
      real     :: tl, tu

!---------------------------------------------------------------
!	... check interval overlap
!---------------------------------------------------------------
!     if( trg_x(1) < src_x(1) .or. trg_x(ntrg+1) > src_x(nsrc+1) ) then
!        write(*,*) 'rebin: target grid is outside source grid'
!        write(*,*) '       target grid from ',trg_x(1),' to ',trg_x(ntrg+1)
!        write(*,*) '       source grid from ',src_x(1),' to ',src_x(nsrc+1)
!        call endrun
!     end if

      do i = 1,ntrg
        tl = trg_x(i)
        if( tl < src_x(nsrc+1) ) then
           do sil = 1,nsrc+1
              if( tl <= src_x(sil) ) then
                 exit
              end if
           end do
           tu = trg_x(i+1)
           do siu = 1,nsrc+1
              if( tu <= src_x(siu) ) then
                 exit
              end if
           end do
           y   = 0.
           sil = max( sil,2 )
           siu = min( siu,nsrc+1 )
           do si = sil,siu
              si1 = si - 1
              sl  = max( tl,src_x(si1) )
              su  = min( tu,src_x(si) )
              y   = y + (su - sl)*src(si1)
           end do
           trg(i) = y/(trg_x(i+1) - trg_x(i))
        else
           trg(i) = 0.
        end if
      end do

      end subroutine rebin

      end module mo_extfrc
