
      module mo_flbc
!---------------------------------------------------------------
! 	... lower boundary module
!---------------------------------------------------------------

      use mo_grid,   only : plon, plat
      use mo_grid,   only : pcnst
      use m_types,   only : time_ramp

      implicit none

      type :: flbc
         integer           :: spc_ndx
         real, pointer     :: vmr(:,:,:,:)
         character(len=16) :: species
      end type flbc

      private
      public  :: flbc_inti, flbc_set, flbc_chk
      public  :: has_flbc

      save

      integer, parameter :: time_span = 12

      integer :: ntimes
      integer :: flbc_cnt
      integer :: gndx
      integer :: nlon
      integer :: tim_ndx(2)
      integer :: jlim(2)
      integer, allocatable  :: dates(:)
      real, allocatable     :: times(:)
      character(len=80) :: filename, lpath, mspath
      logical :: has_flbc(pcnst)

      type(time_ramp)             :: flbc_timing
      type(flbc), allocatable     :: flbcs(:)

      contains

      subroutine flbc_inti( plonl, platl, pplon, flbc_timing_in, ncdate, ncsec )
!-----------------------------------------------------------------------
! 	... initialize the fixed lower bndy cond
!-----------------------------------------------------------------------

      use m_tracname,    only : tracnam
      use mo_chem_utls,  only : get_spc_ndx
      use chem_mods,     only : adv_mass
      use chem_mods,     only : fbc_cnt, flbc_lst, flbc_alias_lst
      use mo_constants,  only : d2r, pi, rearth, latwts
      use mo_control,    only : lbc_flsp
      use mo_file_utils, only : open_netcdf_file
      use mo_charutl,    only : upcase
      use netcdf

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate
      integer, intent(in) :: ncsec
      type(time_ramp), intent(in) :: flbc_timing_in

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: astat
      integer :: j, l, m, n                           ! Indices
      integer :: t1, t2
      integer :: ncid
      integer :: dimid
      integer :: varid
      integer :: yr, wrk_date, wrk_sec
      real    :: seq
      real    :: wrk_time
      character(len=8)  :: species
      character(len=8)  :: spc_name
      character(len=8)  :: time_type

!-----------------------------------------------------------------------
! 	... check timing
!-----------------------------------------------------------------------
      flbc_timing = flbc_timing_in
      call upcase( flbc_timing%type, time_type )
      flbc_timing%type = time_type
      if( time_type /= 'SERIAL' .and. time_type /= 'CYCLICAL' &
                                .and. time_type /= 'FIXED' ) then
         write(*,*) 'lbc_inti: time type ',trim(time_type),' is not SERIAL,CYCLICAL, or FIXED'
         call endrun
      end if

      wrk_sec  = ncsec
      if( time_type == 'SERIAL' ) then
	 wrk_date = ncdate + flbc_timing%yr_offset*10000
      else if( time_type == 'CYCLICAL' ) then
         wrk_date = (flbc_timing%date/10000)*10000 + mod(ncdate,10000)
      else
         wrk_date = flbc_timing%date
         wrk_sec  = 0
      end if
      wrk_time = days0( wrk_date, wrk_sec )
      write(*,*) 'lbc_inti: wrk_date,wrk_sec,wrk_time = ',wrk_date,wrk_sec,wrk_time

      flbc_cnt = fbc_cnt(1)
!-----------------------------------------------------------------------
! 	... species with fixed lbc ?
!-----------------------------------------------------------------------
      if( flbc_cnt == 0 ) then
         return
      end if
!-----------------------------------------------------------------------
! 	... allocate type array
!-----------------------------------------------------------------------
      allocate( flbcs(flbc_cnt), stat=astat )
      if( astat/= 0 ) then
	 write(*,*) 'flbc_inti: failed to allocate flbc array; error = ',astat
	 call endrun
      end if

      has_flbc(:) = .false.
      do n = 1,flbc_cnt
         m = get_spc_ndx( trim(flbc_lst(n)) )
         flbcs(n)%spc_ndx = m
         has_flbc(m) = .true.
         if( flbc_alias_lst(n) == ' ' ) then
            flbcs(n)%species = trim( flbc_lst(n) )
         else
            flbcs(n)%species = trim( flbc_alias_lst(n) )
         end if
      end do

      write(*,*) ' '
      if( flbc_cnt > 0 ) then
         write(*,*) 'Species with specified lower boundary values'
         do m = 1,flbc_cnt
            if( flbc_alias_lst(m) == ' ' ) then
               write(*,*) trim( flbc_lst(m) )
            else
               write(*,*) trim( flbc_lst(m) ) // ' -> ' // trim( flbc_alias_lst(m) )
            end if
         end do
      else
         write(*,*) 'There are no species with specified lower boundary values'
      end if
      write(*,*) ' '

      lpath    = trim( lbc_flsp%local_path )
      mspath   = trim( lbc_flsp%remote_path )
      filename = trim( lbc_flsp%nl_filename )

!-----------------------------------------------------------------------
! 	... diagnostics
!-----------------------------------------------------------------------
      write(*,*) ' '
      write(*,*) 'lbc_inti: diagnostics'
      write(*,*) ' '
      write(*,*) 'lower bndy timing specs'
      write(*,*) 'type = ',flbc_timing%type
      if( time_type == 'SERIAL' ) then
         write(*,*) 'year offset = ',flbc_timing%yr_offset
      else if( time_type == 'CYCLICAL' ) then
         write(*,*) 'year = ',flbc_timing%date/10000
      else
         write(*,*) 'date = ',flbc_timing%date
      end if
      write(*,*) ' '
      write(*,*) 'there are ',flbc_cnt,' species with specified lower bndy values'
      write(*,*) ' '

!-----------------------------------------------------------------------
! 	... get timing information, allocate arrays, and read in dates
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( filename, lpath, mspath )
      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid ), &
                         'lbc_inti: failed to find time dimension' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, ntimes ), &
                         'lbc_inti: failed to get size of time dimension' )
      allocate( dates(ntimes),stat=astat )
      if( astat/= 0 ) then
	 write(*,*) 'lbc_inti: failed to allocate dates array; error = ',astat
	 call endrun
      end if
      allocate( times(ntimes),stat=astat )
      if( astat/= 0 ) then
	 write(*,*) 'lbc_inti: failed to allocate times array; error = ',astat
	 call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'date', varid ), &
                         'lbc_inti: Failed to find date variable' )
      call handle_ncerr( nf_get_var_int( ncid, varid, dates ), &
                         'lbc_inti: Failed to read date variable' )
      do n = 1,ntimes
         times(n) = days0( dates(n), 0 )
      end do
      if( time_type /= 'CYCLICAL' ) then
         if( wrk_time < times(1) .or. wrk_time > times(ntimes) ) then
            write(*,*) 'lbc_inti: time out of bounds for dataset = ',trim(filename)
            call endrun
         end if
         do n = 2,ntimes
            if( wrk_time <= times(n) ) then
               exit
            end if
         end do
         tim_ndx(1) = n - 1
      else
	 yr = flbc_timing%date/10000
         do n = 1,ntimes
            if( yr == dates(n)/10000 ) then
               exit
            end if
         end do
	 if( n >= ntimes ) then
            write(*,*) 'lbc_inti: time out of bounds for dataset = ',trim(filename)
            call endrun
	 end if
         tim_ndx(1) = n
      end if
      select case( time_type )
         case( 'FIXED' )
            tim_ndx(2) = n
         case( 'CYCLICAL' )
            do n = tim_ndx(1),ntimes
               if( yr /= dates(n)/10000 ) then
                  exit
               end if
            end do
            tim_ndx(2) = n - 1
	    if( (tim_ndx(2) - tim_ndx(1)) < 2 ) then
               write(*,*) 'lbc_inti: cyclical lb conds require at least two time points'
               call endrun
	    end if
         case( 'SERIAL' )
            tim_ndx(2) = min( ntimes,tim_ndx(1) + time_span )
      end select
      t1 = tim_ndx(1)
      t2 = tim_ndx(2)

      write(*,*) ' '
      write(*,*) 'flbc time cnt = ',ntimes
      write(*,*) 'flbc times'
      write(*,'(10i10)') dates(:)
      write(*,'(1p,5g15.7)') times(:)
      write(*,*) 'flbc time indicies = ',tim_ndx(:)
      write(*,'(10i10)') dates(tim_ndx(1):tim_ndx(2))
      write(*,*) ' '

      do m = 1,flbc_cnt
!-----------------------------------------------------------------------
! 	... allocate array
!-----------------------------------------------------------------------
         allocate( flbcs(m)%vmr(plonl,platl,pplon,t1:t2),stat=astat )
         if( astat/= 0 ) then
            write(*,*) 'lbc_inti: failed to allocate lbc vmr; error = ',astat
	    call endrun
         end if
!-----------------------------------------------------------------------
! 	... readin the flbc vmr
!-----------------------------------------------------------------------
         call flbc_get( plonl, platl, pplon, ncid, flbcs(m), .true. )
      end do

!-----------------------------------------------------------------------
! 	... close the file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_close( ncid ), &
                         'lbc_inti: Failed to close NetCDF file =' // trim(filename) )

      end subroutine flbc_inti

      subroutine flbc_chk( plonl, platl, pplon, ncdate, ncsec )
!-----------------------------------------------------------------------
!       ... check serial case for time span
!-----------------------------------------------------------------------

      use netcdf
      use mo_file_utils, only : open_netcdf_file

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate              ! present date (yyyymmdd)
      integer, intent(in) :: ncsec               ! seconds in present date

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer                     :: m
      integer                     :: t1, t2, tcnt
      integer                     :: astat
      integer                     :: ncid
      real                        :: wrk_time

      if( flbc_cnt > 0 .and. flbc_timing%type == 'SERIAL' ) then
         wrk_time = days0( ncdate + flbc_timing%yr_offset*10000, ncsec )
         if( wrk_time > times(tim_ndx(2)) ) then
            tcnt = tim_ndx(2) - tim_ndx(1)
            tim_ndx(1) = tim_ndx(2)
            tim_ndx(2) = min( ntimes,tim_ndx(1) + time_span )
            t1 = tim_ndx(1)
            t2 = tim_ndx(2)
!-----------------------------------------------------------------------
! 	... allocate array
!-----------------------------------------------------------------------
            do m = 1,flbc_cnt
               if( associated( flbcs(m)%vmr ) ) then
                  deallocate( flbcs(m)%vmr,stat=astat )
                  if( astat/= 0 ) then
	             write(*,*) 'lbc_chk: failed to deallocate flbc vmr; error = ',astat
	             call endrun
                  end if
	       end if
               allocate( flbcs(m)%vmr(plonl,platl,pplon,t1:t2),stat=astat )
               if( astat/= 0 ) then
	          write(*,*) 'lbc_chk: failed to allocate flbc vmr; error = ',astat
	          call endrun
               end if
	    end do

            ncid = open_netcdf_file( filename, lpath, mspath )
!-----------------------------------------------------------------------
! 	... readin the lb concentrations
!-----------------------------------------------------------------------
            do m = 1,flbc_cnt
               call flbc_get( plonl, platl, pplon, ncid, flbcs(m), .false. )
            end do

!-----------------------------------------------------------------------
! 	... close the file
!-----------------------------------------------------------------------
            call handle_ncerr( nf_close( ncid ), &
                               'lbc_chk: failed to close netcdf file = ' // trim(filename) )
         end if
      end if

      end subroutine flbc_chk

      subroutine flbc_get( plonl, platl, pplon, ncid, flbcs, initial )
!-----------------------------------------------------------------------
!       ... read lower bndy values
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_constants,  only : phi, lam, d2r
      use mo_file_utils, only : open_netcdf_file
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)           :: plonl
      integer, intent(in)           :: platl
      integer, intent(in)           :: pplon
      integer, intent(in)           :: ncid
      logical, intent(in)           :: initial
      type(flbc), intent(inout) :: flbcs

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer                     :: j, jl, ju, m               ! Indices
      integer                     :: t1, t2, tcnt
      integer                     :: ierr
      integer                     :: vid, nlat
      integer                     :: dimid_lat, dimid_lon
      real, allocatable           :: lat(:)
      real, allocatable           :: lon(:)
      real, allocatable           :: wrk(:,:,:)
      real                        :: wrk2d(plon,platl)
      character(len=nf_max_name)  :: varname

initialization : &
      if( initial ) then
!-----------------------------------------------------------------------
!       ... get grid dimensions from file
!-----------------------------------------------------------------------
!           latitudes
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                            'flbc_get: Failed to find dimension lat' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                            'flbc_get: Failed to get size of dimension lat' )
         allocate( lat(nlat),stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'flbc_get: lat allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                            'flbc_get: Failed to find variable lat' )
         call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                            'flbc_get: Failed to read variable lat' )
         lat(:nlat) = lat(:nlat) * d2r
 
!-----------------------------------------------------------------------
!           longitudes
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                            'flbc_get: Failed to find dimension lon' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                            'flbc_get: Failed to get size of dimension lon' )
         allocate( lon(nlon),stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'flbc_get: lon allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                            'flbc_get: Failed to find variable lon' )
         call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                            'flbc_get: Failed to read variable lon' )
         lon(:nlon) = lon(:nlon) * d2r
!-----------------------------------------------------------------------
!       ... set up regridding
!-----------------------------------------------------------------------
         gndx = regrid_inti( nlat, plat, &
                             nlon, plon, &
                             lon,  lam, &
                             lat,  phi, &
                             0, platl, &
                             do_lons=.true., do_lats=.true. )
         deallocate( lat,lon,stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'flbc_get: Failed to deallocate lat,lon; ierr = ',ierr
            call endrun
         end if
         if( gndx /= 0 ) then
            jlim = regrid_lat_limits( gndx )
         else
            jlim = (/ base_lat+1,base_lat+platl /)
         end if
         write(*,'(1x,''flbc_get: gndx='',i2,'', grid limits = '',2i4,'', jl,ju='',2i4)') gndx,jlim,base_lat+1,base_lat+platl
      end if initialization

!-----------------------------------------------------------------------
!       ... read data
!-----------------------------------------------------------------------
      t1 = tim_ndx(1)
      t2 = tim_ndx(2)
      tcnt = t2 - t1 + 1
      allocate( wrk(nlon,jlim(1):jlim(2),tcnt), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'flbc_get: wrk allocation error = ',ierr
         call endrun
      end if
      varname = trim(flbcs%species)
      call handle_ncerr( nf_inq_varid( ncid, trim(varname), vid ), &
                         'flbc_get: Failed to find variable ' // trim(varname) )
      call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                             (/ 1, jlim(1), t1/), &                     ! start
                                             (/ nlon, jlim(2)-jlim(1)+1, tcnt /), &  ! count
                                             wrk ), &
                         'flbc_get: Failed to read variable ' // trim( varname ) )
      jl   = base_lat + 1
      ju   = base_lat + platl
      do m = t1,t2
         call regrid_2d( wrk(:,jlim(1):jlim(2),m-t1+1), wrk2d, gndx, jl, ju, &
                         do_poles=.true. )
         flbcs%vmr(:,:,:,m) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
      end do
      deallocate( wrk,stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'flbc_get: Failed to deallocate wrk, ierr = ',ierr
         call endrun
      end if

      end subroutine flbc_get

      subroutine flbc_set( lat, ip, ncdate, ncsec, vmr, plonl )
!--------------------------------------------------------
!	... set the lower bndy values
!--------------------------------------------------------

      use mo_grid,      only : pcnstm1, plev
      use mo_mpi,       only : base_lat
      use m_tracname,   only : tracnam

      implicit none

!--------------------------------------------------------
!	... dummy arguments
!--------------------------------------------------------
      integer, intent(in) ::   plonl
      integer, intent(in) ::   lat                 ! latitude index
      integer, intent(in) ::   ip                  ! longitude tile index
      integer, intent(in) ::   ncdate              ! present date (yyyymmdd)
      integer, intent(in) ::   ncsec               ! seconds in present date
      real, intent(inout) ::   vmr(plonl,plev,pcnstm1)    ! lower bndy concentrations( mol/mol )

!--------------------------------------------------------
!	... local variables
!--------------------------------------------------------
      integer  ::  i, m, n
      integer  ::  last, next
      integer  ::  wrk_date, wrk_sec
      integer  ::  tcnt
      integer  ::  astat
      real     ::  dels
      real     ::  wrk_time

      if( flbc_cnt < 1 ) then
         return
      end if
!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
      wrk_sec  = ncsec
      select case( flbc_timing%type )
         case( 'SERIAL' )
            wrk_date = ncdate + flbc_timing%yr_offset*10000
         case( 'CYCLICAL' )
            wrk_date = (flbc_timing%date/10000)*10000 + mod( ncdate,10000 )
         case( 'FIXED' )
            wrk_date = flbc_timing%date
            wrk_sec  = 0
      end select
      wrk_time = days0( wrk_date, wrk_sec )

!--------------------------------------------------------
!	... set time interpolation factor
!--------------------------------------------------------
      if( flbc_timing%type /= 'CYCLICAL' ) then
         do n = tim_ndx(1)+1,tim_ndx(2)
            if( wrk_time <= times(n) ) then
               last = n - 1
               next = n
               exit
            end if
         end do
         if( n > ntimes ) then
            write(*,*) 'lbc_set: interp time is out of bounds'
            call endrun
         end if
         dels = (wrk_time - times(last))/(times(next) - times(last))
!        write(*,*) ' '
!        write(*,*) 'flbc_set: last,next,dels,ncdate,ncsec = ',last,next,dels,ncdate,ncsec
      else
         tcnt = tim_ndx(2) - tim_ndx(1) + 1
         call moz_findplb( times(tim_ndx(1)), tcnt, wrk_time, n )
         if( n < tcnt ) then
            last = tim_ndx(1) + n - 1
            next = last + 1
            dels = (wrk_time - times(last))/(times(next) - times(last))
         else
            next = tim_ndx(1)
            last = tim_ndx(2)
            dels = wrk_time - times(last)
            if( dels < 0. ) then
               dels = 365. + dels
            end if
            dels = dels/(365. + times(next) - times(last))
         end if
!        write(*,*) ' '
!        write(*,*) 'flbc_set: last,next,dels,ncdate,ncsec = ',last,next,dels,ncdate,ncsec
      end if

      dels = max( min( 1.,dels ),0. )

      do m = 1,flbc_cnt
         n = flbcs(m)%spc_ndx
         vmr(:,plev,n) = flbcs(m)%vmr(:,lat,ip,last) &
                         + dels * (flbcs(m)%vmr(:,lat,ip,next) - flbcs(m)%vmr(:,lat,ip,last))
      end do


      end subroutine flbc_set

      real function days0( ncdate, ncsec )
!----------------------------------------------------------------------- 
! Purpose: Convert date and seconds of day to floating point days since
!          00/01/01
! 
! Method: Use table of days per month to do conversion
! 
! Author: CCM Core Group
!-----------------------------------------------------------------------


      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)   :: ncdate      ! Current date as yymmdd or yyyymmdd
      integer, intent(in)   :: ncsec       ! Seconds of day for current date

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer  :: year        ! year
      integer  :: mnth        ! Month number
      integer  :: mday        ! Day number of month
      integer  :: jdcon(12) & ! Starting day number for each month
                       = (/ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 /)
      real     :: doy

!-----------------------------------------------------------------------
! 	... decode month and day
!-----------------------------------------------------------------------
      year = ncdate/10000
      mnth = mod( ncdate,10000 )/100
      if( mnth < 1 .or. mnth > 12) then
         write(*,*) 'days0: Bad month index = ', mnth
         call endrun
      end if
      mday = mod(ncdate,100)
      doy  = jdcon(mnth) + mday + ncsec/86400.

      if( doy < 1. .or. doy > 366. ) then
         write(*,*) 'days0: bad day of year = ',doy
         call endrun
      end if

      days0 = 365.*year + doy

      end function days0

      end module mo_flbc
