
#ifdef IBM
#define USE_ESSL
#endif
      module mo_jlong

      implicit none

      private
      public :: jlong_init
      public :: jlong_timestep_init
      public :: jlong
      public :: numj

      integer           :: nw                    ! wavelengths >200nm
      integer           :: nwe_max 		 ! last wave bin index inside woods grid
      integer           :: nt                    ! number of temperatures in xsection table
      integer           :: np_xs                 ! number of pressure levels in xs    table
      integer           :: numj                  ! number of photorates in xsqy, rsf
      integer           :: nump                  ! number of pressure levels in rsf
      integer           :: numsza                ! number of zen angles in rsf
      integer           :: numalb                ! number of albedos in rsf
      integer           :: numcolo3              ! number of o3 columns in rsf
      real, allocatable :: xsqy(:,:,:,:)
      real, allocatable :: wc(:)
      real, allocatable :: we(:)
      real, allocatable :: wlintv(:)
      real, allocatable :: etfphot(:)
      real, allocatable :: prs(:)
      real, allocatable :: dprs(:)
      real, allocatable :: p(:)
      real, allocatable :: del_p(:)
      real, allocatable :: sza(:)
      real, allocatable :: del_sza(:)
      real, allocatable :: alb(:)
      real, allocatable :: del_alb(:)
      real, allocatable :: o3rat(:)
      real, allocatable :: del_o3rat(:)
      real, allocatable :: colo3(:)
      real, allocatable :: rsf_tab(:,:,:,:,:)

      contains

      subroutine jlong_init( lng_indexer )
!------------------------------------------------------------------------------
!    ... initialize the long wavelength photolysis module
!------------------------------------------------------------------------------

      use chem_mods, only : phtcnt
      use m_types, only   : filespec
      use mo_solar_parms, only : rebin
      use woods,          only : nbins, woods_etf, woods_we => we
      use neckel,         only : neckel_nw => nw, neckel_we => we, neckel_etf => etf

      implicit none

!------------------------------------------------------------------------------
!    ... dummy arguments
!------------------------------------------------------------------------------
      integer, intent(inout) :: lng_indexer(:)

!------------------------------------------------------------------------------
!    ... local variables
!------------------------------------------------------------------------------
      real, parameter    :: woods_limit = 350.          ! limit of woods remapping (nm)
      logical            :: found

!------------------------------------------------------------------------------
!     ... read Cross Section * QY NetCDF file...
!         find temperature index for given altitude...
!         derive cross*QY results, returns xsqy(nj,nz,nw)
!------------------------------------------------------------------------------
      call get_xsqy( lng_indexer )

!------------------------------------------------------------------------------
!     ... read radiative source function NetCDF file
!------------------------------------------------------------------------------
      call get_rsf

      we(:nw)  = wc(:nw) - .5*wlintv(:nw)
      we(nw+1) = wc(nw) + .5*wlintv(nw)
!------------------------------------------------------------------------------
!     ... find last full interval in long grid inside woods grid
!------------------------------------------------------------------------------
      found = .false.
      do nwe_max = nw+1,2,-1
         if( we(nwe_max) <= woods_limit ) then
            found = .true.
            exit 
         end if
      end do
      if( .not. found ) then
         write(*,*) 'jlong_init: failed to place long wave spectra in woods'
         call endrun
      else
         write(*,*) 'jlong_init: nwe_max = ',nwe_max-1
      end if
      nwe_max = nwe_max - 1
      write(*,*) ' '
      write(*,*) '--------------------------------------------------'
      call rebin( nbins, nwe_max, woods_we, we, woods_etf, etfphot )
      write(*,*) 'jlong_init: etfphot after woods rebin'
      write(*,'(1p,5g15.7)') etfphot(:nwe_max)
      write(*,*) '--------------------------------------------------'
      write(*,*) ' '

      write(*,*) ' '
      write(*,*) '--------------------------------------------------'
      call rebin( neckel_nw, nw-nwe_max, neckel_we, we(nwe_max+1), neckel_etf, etfphot(nwe_max+1) )
      write(*,*) 'jlong_init: etfphot after neckel rebin'
      write(*,'(1p,5g15.7)') etfphot(nwe_max+1:)
      write(*,*) '--------------------------------------------------'
      write(*,*) ' '

      end subroutine jlong_init

      subroutine get_xsqy( lng_indexer )
!=============================================================================
!   Subroutine GET_XSQY                                                      
!=============================================================================
!   PURPOSE:                                                                 
!   Reads a NetCDF file that contains:                                       
!     cross section * QY temperature dependence, >200nm                      
!=============================================================================
!   EDIT HISTORY:                                                             
!   Created by Doug Kinnison, 3/14/2002                                       
!=============================================================================

      use netcdf
      use chem_mods,     only : phtcnt, pht_alias_lst, rxt_tag_lst
      use mo_control,    only : photo_xs_long_flsp
      use mo_file_utils, only : open_netcdf_file
      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_comm_comp, mpi_integer, mpi_double_precision
#endif

      implicit none

!------------------------------------------------------------------------------
!    ... dummy arguments
!------------------------------------------------------------------------------
      integer, intent(inout) :: lng_indexer(:)

!------------------------------------------------------------------------------
!       ... local variables
!------------------------------------------------------------------------------
      integer :: varid, dimid, ndx
      integer :: ncid
      integer :: iret
      integer :: i, k, m, n
      integer :: wrk_ndx(phtcnt)
      integer :: bcast_ndx(4)
      real, allocatable :: xsqy_species(:,:,:)
      character(len=80) :: err_msg

master_only : &
      if( masternode ) then
!------------------------------------------------------------------------------
!       ... open NetCDF File
!------------------------------------------------------------------------------
         ncid = open_netcdf_file( photo_xs_long_flsp%nl_filename, &
                                  photo_xs_long_flsp%local_path, &
                                  photo_xs_long_flsp%remote_path, masteronly=.true. )
!------------------------------------------------------------------------------
!       ... get dimensions
!------------------------------------------------------------------------------
         err_msg = 'get_xsqy: failed to get numprs id'
         call handle_ncerr( nf_inq_dimid( ncid, 'numprs', dimid ), trim(err_msg) )
         err_msg = 'get_xsqy: failed to get numprs dimension'
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, np_xs ), trim(err_msg) )
         err_msg = 'get_xsqy: failed to get numtemp id'
         call handle_ncerr( nf_inq_dimid( ncid, 'numtemp', dimid ), trim(err_msg) )
         err_msg = 'get_xsqy: failed to get numtemp dimension'
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nt ), trim(err_msg) )
         err_msg = 'get_xsqy: failed to get numwl id'
         call handle_ncerr( nf_inq_dimid( ncid, 'numwl', dimid ), trim(err_msg) )
         err_msg = 'get_xsqy: failed to get numwl dimension'
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nw ), trim(err_msg) )
!------------------------------------------------------------------------------
!       ... check for cross section in dataset
!------------------------------------------------------------------------------
         do m = 1,phtcnt
            if( pht_alias_lst(m,2) == ' ' ) then
               iret = nf_inq_varid( ncid, rxt_tag_lst(m), varid )
               if( iret == nf_noerr ) then 
                  lng_indexer(m) = varid
               end if
            else if( pht_alias_lst(m,2) == 'userdefined' ) then
               lng_indexer(m) = -1
            else
               iret = nf_inq_varid( ncid, pht_alias_lst(m,2), varid )
               if( iret == nf_noerr ) then 
                  lng_indexer(m) = varid
               else
	          write(*,*) 'get_xsqy : ',rxt_tag_lst(m)(:len_trim(rxt_tag_lst(m))),' alias ', &
                             pht_alias_lst(m,2)(:len_trim(pht_alias_lst(m,2))),' not in dataset'            
                  call endrun
               end if
            end if
         end do
         numj = 0
         do m = 1,phtcnt
            if( lng_indexer(m) > 0 ) then
               if( any( lng_indexer(:m-1) == lng_indexer(m) ) ) then
                  cycle
               end if
               numj = numj + 1
            end if 
         end do

!------------------------------------------------------------------------------
!       ... allocate arrays
!------------------------------------------------------------------------------
         allocate( xsqy(numj,nw,nt,np_xs),stat=iret )
         if( iret /= nf_noerr) then 
	    write(*,*) 'get_xsqy : failed to allocate xsqy ; error = ',iret
            call endrun
         end if
         allocate( xsqy_species(nw,nt,np_xs),stat=iret )
         if( iret /= nf_noerr) then 
	    write(*,*) 'get_xsqy : failed to allocate xsqy_species ; error = ',iret
            call endrun
         end if
         allocate( prs(np_xs),dprs(np_xs-1),stat=iret )
         if( iret /= nf_noerr) then 
	    write(*,*) 'get_xsqy : failed to allocate prs,dprs ; error = ',iret
            call endrun
         end if
!------------------------------------------------------------------------------
!       ... Read variables
!------------------------------------------------------------------------------
         err_msg = 'get_xsqy: failed to get pressure id'
         call handle_ncerr( nf_inq_varid( ncid, 'pressure', varid ), trim(err_msg) )
         err_msg = 'get_xsqy: failed to read pressure id'
         call handle_ncerr( nf_get_var_double( ncid, varid, prs ), trim(err_msg) )
         ndx = 0
         do m = 1,phtcnt
            if( lng_indexer(m) > 0 ) then
               if( any( lng_indexer(:m-1) == lng_indexer(m) ) ) then
                  cycle
               end if
               err_msg = 'get_xsqy: failed to read '// trim(rxt_tag_lst(m))
               call handle_ncerr( nf_get_var_double( ncid, varid, xsqy_species ), trim(err_msg) )
               ndx = ndx + 1
               xsqy(ndx,:,:,:) = xsqy_species(:,:,:)
            end if
         end do
         deallocate( xsqy_species )
         if( ndx /= numj ) then
	    write(*,*) 'get_xsqy : ndx count /= cross section count'
            call endrun
         end if
!------------------------------------------------------------------------------
!       ... setup final lng_indexer
!------------------------------------------------------------------------------
         ndx = 0
         wrk_ndx(:) = lng_indexer(:)
         do m = 1,phtcnt
            if( wrk_ndx(m) > 0 ) then
               ndx = ndx + 1
               i = wrk_ndx(m)
               where( wrk_ndx(:) == i )
                  lng_indexer(:) = ndx
                  wrk_ndx(:)     = -100000
               end where
            end if
         end do
         err_msg = 'get_xsqy: failed to close ' // trim( photo_xs_long_flsp%nl_filename )
         call handle_ncerr( nf_close( ncid ), trim(err_msg) )
      end if master_only
#ifdef USE_MPI
      bcast_ndx(:) = (/ numj, np_xs, nt, nw /)
      call mpi_bcast( bcast_ndx, 4, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_xsqy: failed to bcast bcast_ndx; error = ',iret
         call endrun
      end if
      call mpi_bcast( lng_indexer, phtcnt, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_xsqy: failed to bcast lng_indexer; error = ',iret
         call endrun
      end if
      if( .not. masternode ) then
         numj  = bcast_ndx(1)
         np_xs = bcast_ndx(2)
         nt    = bcast_ndx(3)
         nw    = bcast_ndx(4)
!------------------------------------------------------------------------------
!       ... allocate arrays
!------------------------------------------------------------------------------
         allocate( xsqy(numj,nw,nt,np_xs),stat=iret )
         if( iret /= nf_noerr) then 
	    write(*,*) 'get_xsqy : failed to allocate xsqy ; error = ',iret
            call endrun
         end if
         allocate( prs(np_xs),dprs(np_xs-1),stat=iret )
         if( iret /= nf_noerr) then 
	    write(*,*) 'get_xsqy : failed to allocate prs,dprs ; error = ',iret
            call endrun
         end if
      end if
!------------------------------------------------------------------------------
!       ... bcast arrays
!------------------------------------------------------------------------------
      call mpi_bcast( prs, np_xs, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_xsqy: failed to bcast prs; error = ',iret
         call endrun
      end if
      call mpi_bcast( xsqy, numj*nw*nt*np_xs, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_xsqy: failed to bcast xsqy; error = ',iret
         call endrun
      end if
#endif
      dprs(:np_xs-1) = 1./(prs(1:np_xs-1) - prs(2:np_xs))

      end subroutine get_xsqy

      subroutine get_rsf
!=============================================================================!
!   Subroutine get_rsf                                                        !
!=============================================================================!
!   PURPOSE:                                                                  !
!   Reads a NetCDF file that contains:
!     Radiative Souce function                                                !
!=============================================================================!
!   EDIT HISTORY:                                                             !
!   Created by Doug Kinnison, 3/14/2002                                       !
!=============================================================================!

      use netcdf
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : photo_rsf_flsp
      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_comm_comp, mpi_integer, mpi_double_precision
#endif

      implicit none

!------------------------------------------------------------------------------
!       ... local variables
!------------------------------------------------------------------------------
      integer :: varid, dimid
      integer :: ncid
      integer :: i, j, k, l, wn
      integer :: iret
      real, allocatable :: wrk(:)
      real, allocatable :: fnorm_in(:,:,:,:,:)
      character(len=80) :: err_msg

master_only : &
      if( masternode ) then
!------------------------------------------------------------------------------
!       ... open NetCDF File
!------------------------------------------------------------------------------
         ncid = open_netcdf_file( photo_rsf_flsp%nl_filename, &
                                  photo_rsf_flsp%local_path, &
                                  photo_rsf_flsp%remote_path, masteronly=.true. )
!------------------------------------------------------------------------------
!       ... get dimensions
!------------------------------------------------------------------------------
         err_msg = 'get_rsf: failed to get dimension numz id'
         call handle_ncerr( nf_inq_dimid( ncid, 'numz', dimid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get dimension numz'
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nump ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get dimension numsza id'
         call handle_ncerr( nf_inq_dimid( ncid, 'numsza', dimid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get dimension numsza'
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, numsza ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get dimension numalb id'
         call handle_ncerr( nf_inq_dimid( ncid, 'numalb', dimid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get dimension numalb'
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, numalb ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get dimension numcolo3fact id'
         call handle_ncerr( nf_inq_dimid( ncid, 'numcolo3fact', dimid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get dimension numcolo3fact'
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, numcolo3 ), trim(err_msg) )
!------------------------------------------------------------------------------
!       ... allocate arrays
!------------------------------------------------------------------------------
         allocate( wc(nw),wlintv(nw),etfphot(nw),we(nw+1),wrk(nw),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate wc ... wrk ; error = ',iret
            call endrun
         end if
         allocate( p(nump),del_p(nump-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate z,del_z ; error = ',iret
            call endrun
         end if
         allocate( sza(numsza),del_sza(numsza-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate sza,del_sza ; error = ',iret
            call endrun
         end if
         allocate( alb(numalb),del_alb(numalb-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate alb,del_alb ; error = ',iret
            call endrun
         end if
         allocate( o3rat(numcolo3),del_o3rat(numcolo3-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate o3rat,del_o3rat ; error = ',iret
            call endrun
         end if
         allocate( colo3(nump),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate colo3; error = ',iret
            call endrun
         end if
         allocate( fnorm_in(numalb,numcolo3,numsza,nump,nw),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate rsf_tab; error = ',iret
            call endrun
         end if
         allocate( rsf_tab(nw,nump,numsza,numcolo3,numalb),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate rsf_tab; error = ',iret
            call endrun
         end if
!------------------------------------------------------------------------------
!       ... read variables
!------------------------------------------------------------------------------
         err_msg = 'get_rsf: failed to get wc variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'wc', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read wc variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, wc ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get wlintv variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'wlintv', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read wlintv variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, wlintv ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get pm variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'pm', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read pm variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, p ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get sza variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'sza', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read sza variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, sza ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get alb variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'alb', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read alb variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, alb ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get colo3fact variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'colo3fact', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read colo3fact variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, o3rat ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get colo3 variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'colo3', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read colo3 variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, colo3 ), trim(err_msg) )
         err_msg = 'get_rsf: failed to get RSF variable id'
         call handle_ncerr( nf_inq_varid( ncid, 'RSF', varid ), trim(err_msg) )
         err_msg = 'get_rsf: failed to read RSF variable'
         call handle_ncerr( nf_get_var_double( ncid, varid, fnorm_in ), trim(err_msg) )
!------------------------------------------------------------------------------
!       ... form table rsf field
!------------------------------------------------------------------------------
         wrk(:) = wlintv(:)
         do i = 1,nump
            do j = 1,numsza
               do k = 1,numcolo3
                  do l = 1,numalb
                     rsf_tab(:,i,j,k,l) = wrk(:)*fnorm_in(l,k,j,i,:)
                  end do
               end do
            end do
         end do

         deallocate( fnorm_in, wrk )

         err_msg = 'get_rsf: failed to close ' // trim( photo_rsf_flsp%nl_filename )
         call handle_ncerr( nf_close( ncid ), trim(err_msg) )
      end if master_only

#ifdef USE_MPI
      call mpi_bcast( nump, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast nump; error = ',iret
         call endrun
      end if
      call mpi_bcast( numsza, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast numsza; error = ',iret
         call endrun
      end if
      call mpi_bcast( numalb, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast numalb; error = ',iret
         call endrun
      end if
      call mpi_bcast( numcolo3, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast numcolo3; error = ',iret
         call endrun
      end if
      if( .not. masternode ) then
!------------------------------------------------------------------------------
!       ... allocate arrays
!------------------------------------------------------------------------------
         allocate( wc(nw), wlintv(nw), etfphot(nw), we(nw+1), stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate wc ... we ; error = ',iret
            call endrun
         end if
         allocate( p(nump),del_p(nump-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate z,del_z ; error = ',iret
            call endrun
         end if
         allocate( sza(numsza),del_sza(numsza-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate sza,del_sza ; error = ',iret
            call endrun
         end if
         allocate( alb(numalb),del_alb(numalb-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate alb,del_alb ; error = ',iret
            call endrun
         end if
         allocate( o3rat(numcolo3),del_o3rat(numcolo3-1),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate o3rat,del_o3rat ; error = ',iret
            call endrun
         end if
         allocate( colo3(nump),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate colo3; error = ',iret
            call endrun
         end if
         allocate( rsf_tab(nw,nump,numsza,numcolo3,numalb),stat=iret )
         if( iret /= nf_noerr) then 
            write(*,*) 'get_rsf : failed to allocate rsf_tab; error = ',iret
            call endrun
         end if
      end if
      call mpi_bcast( wc, nw, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast wc; error = ',iret
         call endrun
      end if
      call mpi_bcast( wlintv, nw, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast wlintv; error = ',iret
         call endrun
      end if
      call mpi_bcast( p, nump, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast p; error = ',iret
         call endrun
      end if
      call mpi_bcast( sza, numsza, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast sza; error = ',iret
         call endrun
      end if
      call mpi_bcast( alb, numalb, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast alb; error = ',iret
         call endrun
      end if
      call mpi_bcast( o3rat, numcolo3, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast o3rat; error = ',iret
         call endrun
      end if
      call mpi_bcast( colo3, nump, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast colo3; error = ',iret
         call endrun
      end if
      call mpi_bcast( rsf_tab, nw*nump*numsza*numcolo3*numalb, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'get_rsf: failed to bcast rsf_tab; error = ',iret
         call endrun
      end if
#endif
!------------------------------------------------------------------------------
!       ... form secondary inverse delta variables
!------------------------------------------------------------------------------
      del_p(:nump-1)         = 1./(p(1:nump-1) - p(2:nump))
      del_sza(:numsza-1)     = 1./(sza(2:numsza) - sza(1:numsza-1))
      del_alb(:numalb-1)     = 1./(alb(2:numalb) - alb(1:numalb-1))
      del_o3rat(:numcolo3-1) = 1./(o3rat(2:numcolo3) - o3rat(1:numcolo3-1))

      end subroutine get_rsf

      subroutine jlong_timestep_init( ncsec )
!---------------------------------------------------------------
!	... update etfphot if required
!---------------------------------------------------------------

      use mo_solar_parms, only : rebin
      use woods,          only : nbins, woods_etf, woods_we => we
      use neckel,         only : neckel_nw => nw, neckel_we => we, neckel_etf => etf

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in) :: ncsec

      if( ncsec == 0 ) then
         call rebin( nbins, nwe_max, woods_we, we, woods_etf, etfphot )
         call rebin( neckel_nw, nw-nwe_max, neckel_we, we(nwe_max+1), neckel_etf, etfphot(nwe_max+1) )
      end if

      end subroutine jlong_timestep_init

      subroutine jlong( nlev, sza_in, alb_in, p_in, t_in, &
                        colo3_in, j_long )
!==============================================================================
!   Subroutine JLONG                                                           
!==============================================================================
!   Purpose:                                                                   
!     To calculate the total J for selective species longward of 200nm.        
!==============================================================================
!                                                                              
!   Approach:
!     1) Reads the Cross Section*QY NetCDF file
!     2) Given a temperature profile, derives the appropriate XS*QY
!
!     3) Reads the Radiative Source function (RSF) NetCDF file
!        Units = quanta cm-2 sec-1
!
!     4) Indices are supplied to select a RSF that is consistent with
!        the reference atmosphere in TUV (for direct comparision of J's).
!        This approach will be replaced in the global model. Here colo3, zenith
!        angle, and altitude will be inputed and the correct entry in the table
!        will be derived.
!==============================================================================
!   EDIT HISTORY:
!   Created by Doug Kinnison, 3/14/2002
!==============================================================================

        use mo_mpi,    only : thisnode

        implicit none

!------------------------------------------------------------------------------
!    	... dummy arguments
!------------------------------------------------------------------------------
      integer, intent (in) :: nlev               ! number vertical levels
      real, intent(in)     :: sza_in             ! solar zenith angle (degrees)
      real, intent(in)     :: alb_in(nlev)       ! albedo
      real, intent(in)     :: p_in(nlev)         ! midpoint pressure (hPa)
      real, intent(in)     :: t_in(nlev)         ! Temperature profile (K)
      real, intent(in)     :: colo3_in(nlev)     ! o3 column density (molecules/cm^3)
      real, intent(out)    :: j_long(:,:)        ! photo rates (1/s)

!----------------------------------------------------------------------
!    	... local variables
!----------------------------------------------------------------------
      integer  ::  is, iv, ial
      integer  ::  isp1, ivp1, ialp1
      integer  ::  astat
      integer  ::  iz, k, km, m
      integer  ::  izl
      integer  ::  pndx
      integer  ::  wn
      integer  ::  t_index                                      ! Temperature index
      integer  ::  ratindl, ratindu
      integer  ::  pind, albind
      real     ::  wrk0, wrk1, wght1
      real     ::  psum_u, p_tablel, p_tableu, col_ozonel, col_ozoneu
      real     ::  delp
      real     ::  v3ratl, v3ratu
      real     ::  dels2, psum_u2, psum_u46
      real, dimension(3)               :: dels
      real, dimension(0:1,0:1,0:1)     :: wghtl, wghtu
      real, allocatable                :: psum_l(:)
      real, allocatable                :: rsf(:,:)              ! Radiative source function
      real, allocatable                :: xswk(:,:)             ! working xsection array

!----------------------------------------------------------------------
!        ... allocate arrays
!----------------------------------------------------------------------
      allocate( psum_l(nw),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'jlong : failed to allocate work array psum_l : error = ',astat
	 call endrun
      end if
      allocate( rsf(nw,nlev),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'jlong : failed to allocate work array rsf : error = ',astat
         call endrun
      end if
      allocate( xswk(numj,nw),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'jlong : failed to allocate work array xswk : error = ',astat
	 call endrun
      end if
!----------------------------------------------------------------------
!        ... Interpolate for rsf
!----------------------------------------------------------------------
!----------------------------------------------------------------------
!        ... Find the zenith angle index ( same for all levels )
!----------------------------------------------------------------------
#ifdef DEBUG
      if( sza_in < sza(1) .or. sza_in > sza(numsza) ) then
	 write(*,*) 'jlong: sza = ',sza_in,' is out of bounds at lon,lat,ip = ',long,lat,ip
      end if
#endif
      do is = 1,numsza
         if( sza(is) > sza_in ) then
            exit
         end if
      end do
      is   = max( min( is,numsza ) - 1,1 )
      isp1 = is + 1
      dels(1)  = max( 0.,min( 1.,(sza_in - sza(is)) * del_sza(is) ) )
      wrk0     = 1. - dels(1)

      izl = 2
Level_loop : &
      do k = nlev,1,-1
!----------------------------------------------------------------------
!        ... Find albedo indicies
!----------------------------------------------------------------------
#ifdef DEBUG
         if( alb_in(k) < alb(1) .or. alb_in(k) > alb(numalb) ) then
	    write(*,*) 'jlong: alb = ',alb_in(k),' is out of bounds at lon,lat,ip,k = ',long,lat,ip,k
         end if
#endif
         do ial = 1,numalb
            if( alb(ial) > alb_in(k) ) then
               exit
            end if
         end do
         albind = max( min( ial,numalb ) - 1,1 )
!----------------------------------------------------------------------
!        ... Find pressure level indicies
!----------------------------------------------------------------------
#ifdef DEBUG
         if( p_in(k) > p(1) .or. p_in(k) < p(nump) ) then
            write(*,*) 'jlong: p = ',p_in(k),' is out of bounds at lon,lat,ip,k = ',long,lat,ip,k
         end if
#endif
         if( p_in(k) > p(1) ) then
            pind  = 2
            wght1 = 1.
         else if( p_in(k) <= p(nump) ) then
            pind  = nump
            wght1 = 0.
         else
            do iz = izl,nump
               if( p(iz) < p_in(k) ) then
	          izl = iz
	          exit
               end if
            end do
            pind  = max( min( iz,nump ),2 )
            wght1 = max( 0.,min( 1.,(p_in(k) - p(pind)) * del_p(pind-1) ) )
         end if
!----------------------------------------------------------------------
!        ... Find "o3 ratio" indicies; lower then upper
!----------------------------------------------------------------------
         v3ratl = colo3_in(k) / colo3(pind)
#ifdef DEBUG
         if( v3ratl < o3rat(1) .or. v3ratl > o3rat(numcolo3) ) then
	    write(*,*) 'jlong: v3ratl = ',v3ratl,' is out of bounds at lon,lat,ip,k = ',long,lat,ip,k
         end if
#endif
         do iv = 1,numcolo3
            if( o3rat(iv) > v3ratl ) then
               exit
            end if
         end do
         ratindl = max( min( iv,numcolo3 ) - 1,1 )

         v3ratu  = colo3_in(k) / colo3(pind-1)
#ifdef DEBUG
         if( v3ratu < o3rat(1) .or. v3ratu > o3rat(numcolo3) ) then
            write(*,*) 'jlong: v3ratu = ',v3ratu,' is out of bounds at lon,lat,ip,k = ',long,lat,ip,k
         end if
#endif
         do iv = 1,numcolo3
            if( o3rat(iv) > v3ratu ) then
               exit
            end if
         end do
         ratindu = max( min( iv,numcolo3 ) - 1,1 )

!----------------------------------------------------------------------
!        ... Compute the weigths
!----------------------------------------------------------------------
	 ial   = albind
	 ialp1 = ial + 1
	 iv    = ratindl

         dels(2)  = max( 0.,min( 1.,(v3ratl - o3rat(iv)) * del_o3rat(iv) ) )
	 dels2    = dels(2)
         dels(3)  = max( 0.,min( 1.,(alb_in(k) - alb(ial)) * del_alb(ial) ) )

	 wrk1         = (1. - dels(2))*(1. - dels(3))
	 wghtl(0,0,0) = wrk0*wrk1
	 wghtl(1,0,0) = dels(1)*wrk1
	 wrk1         = (1. - dels(2))*dels(3)
	 wghtl(0,0,1) = wrk0*wrk1
	 wghtl(1,0,1) = dels(1)*wrk1
	 wrk1         = dels(2)*(1. - dels(3))
	 wghtl(0,1,0) = wrk0*wrk1
	 wghtl(1,1,0) = dels(1)*wrk1
	 wrk1         = dels(2)*dels(3)
	 wghtl(0,1,1) = wrk0*wrk1
	 wghtl(1,1,1) = dels(1)*wrk1

	 iv  = ratindu
         dels(2)  = max( 0.,min( 1.,(v3ratu - o3rat(iv)) * del_o3rat(iv) ) )

	 wrk1         = (1. - dels(2))*(1. - dels(3))
	 wghtu(0,0,0) = wrk0*wrk1
	 wghtu(1,0,0) = dels(1)*wrk1
	 wrk1         = (1. - dels(2))*dels(3)
	 wghtu(0,0,1) = wrk0*wrk1
	 wghtu(1,0,1) = dels(1)*wrk1
	 wrk1         = dels(2)*(1. - dels(3))
	 wghtu(0,1,0) = wrk0*wrk1
	 wghtu(1,1,0) = dels(1)*wrk1
	 wrk1         = dels(2)*dels(3)
	 wghtu(0,1,1) = wrk0*wrk1
	 wghtu(1,1,1) = dels(1)*wrk1

	 iz   = pind
	 iv   = ratindl
	 ivp1 = iv + 1
         do wn = 1,nw
            psum_l(wn) = wghtl(0,0,0) * rsf_tab(wn,iz,is,iv,ial) &
                         + wghtl(0,0,1) * rsf_tab(wn,iz,is,iv,ialp1) &
                         + wghtl(0,1,0) * rsf_tab(wn,iz,is,ivp1,ial) &
                         + wghtl(0,1,1) * rsf_tab(wn,iz,is,ivp1,ialp1) &
                         + wghtl(1,0,0) * rsf_tab(wn,iz,isp1,iv,ial) &
                         + wghtl(1,0,1) * rsf_tab(wn,iz,isp1,iv,ialp1) &
                         + wghtl(1,1,0) * rsf_tab(wn,iz,isp1,ivp1,ial) &
                         + wghtl(1,1,1) * rsf_tab(wn,iz,isp1,ivp1,ialp1)
         end do

	 iz   = iz - 1
	 iv   = ratindu
	 ivp1 = iv + 1
         do wn = 1,nw
            psum_u = wghtu(0,0,0) * rsf_tab(wn,iz,is,iv,ial) &
                     + wghtu(0,0,1) * rsf_tab(wn,iz,is,iv,ialp1) &
                     + wghtu(0,1,0) * rsf_tab(wn,iz,is,ivp1,ial) &
                     + wghtu(0,1,1) * rsf_tab(wn,iz,is,ivp1,ialp1) &
                     + wghtu(1,0,0) * rsf_tab(wn,iz,isp1,iv,ial) &
                     + wghtu(1,0,1) * rsf_tab(wn,iz,isp1,iv,ialp1) &
                     + wghtu(1,1,0) * rsf_tab(wn,iz,isp1,ivp1,ial) &
                     + wghtu(1,1,1) * rsf_tab(wn,iz,isp1,ivp1,ialp1)
            rsf(wn,k) = (psum_l(wn) + wght1*(psum_u - psum_l(wn)))
         end do
         rsf(:,k) = etfphot(:) * rsf(:,k)
#ifdef DEBUG_1
!       if( k == 23 .and. thisnode == 0 .and. lat == 2 .and. (ip-1)*plonl+long == 97 ) then
	if( k == 23 .and. lat == 2 .and. (ip-1)*plonl+long == 97 ) then
	   write(*,*) 'jlong: diagnostics for rsf @ k = ',k
	   write(*,*) 'p_in'
	   write(*,'(1x,1p,10g12.5)') p_in
	   write(*,*) 't_in'
	   write(*,'(1x,1p,10g12.5)') t_in
	   write(*,*) 'colo3_in'
	   write(*,'(1x,1p,10g12.5)') colo3_in
	   write(*,*) '       sza_in,alb_in(k),p_in(k),colo3_in(k)'
	   write(*,'(1x,1p,10g12.5)') sza_in,alb_in(k),p_in(k),colo3_in(k)
	   write(*,'('' v3rat = '',1p,2g12.5)') v3ratl, v3ratu
	   write(*,'('' dels  = '',1p,5g12.5)') wght1,dels(1),dels2,dels(2),dels(3)
	   write(*,'('' wghtl = '',1p,8g12.5)') wghtl
	   write(*,'('' wghtu = '',1p,8g12.5)') wghtu
	   write(*,*) '       pind,ratindl,ratindu = ',pind,ratindl,ratindu
	   write(*,*) '       iz,is,iv,ial = ',iz,is,iv,ial
	   write(*,*) '       table rsf(2) at lower level'
	   write(*,'(1x,1p,8g12.5)') rsf_tab(2,pind-1,is,ratindl,ial), rsf_tab(2,pind-1,is,ratindl,ialp1), &
	                             rsf_tab(2,pind-1,is,ratindl+1,ial), rsf_tab(2,pind-1,is,ratindl+1,ialp1), &
	                             rsf_tab(2,pind-1,isp1,ratindl,ial), rsf_tab(2,pind-1,isp1,ratindl,ialp1), &
	                             rsf_tab(2,pind-1,isp1,ratindl+1,ial), rsf_tab(2,pind-1,isp1,ratindl+1,ialp1)
	   write(*,*) '       table rsf(2) at upper level'
	   write(*,'(1x,1p,8g12.5)') rsf_tab(2,pind,is,ratindu,ial), rsf_tab(2,pind,is,ratindu,ialp1), &
	                             rsf_tab(2,pind,is,ratindu+1,ial), rsf_tab(2,pind,is,ratindu+1,ialp1), &
	                             rsf_tab(2,pind,isp1,ratindu,ial), rsf_tab(2,pind,isp1,ratindu,ialp1), &
	                             rsf_tab(2,pind,isp1,ratindu+1,ial), rsf_tab(2,pind,isp1,ratindu+1,ialp1)
	   write(*,*) '       table rsf(46) at lower level'
	   write(*,'(1x,1p,8g12.5)') rsf_tab(46,pind-1,is,ratindl,ial), rsf_tab(46,pind-1,is,ratindl,ialp1), &
	                             rsf_tab(46,pind-1,is,ratindl+1,ial), rsf_tab(46,pind-1,is,ratindl+1,ialp1), &
	                             rsf_tab(46,pind-1,isp1,ratindl,ial), rsf_tab(46,pind-1,isp1,ratindl,ialp1), &
	                             rsf_tab(46,pind-1,isp1,ratindl+1,ial), rsf_tab(46,pind-1,isp1,ratindl+1,ialp1)
	   write(*,*) '       table rsf(46) at upper level'
	   write(*,'(1x,1p,8g12.5)') rsf_tab(46,pind,is,ratindu,ial), rsf_tab(46,pind,is,ratindu,ialp1), &
	                             rsf_tab(46,pind,is,ratindu+1,ial), rsf_tab(46,pind,is,ratindu+1,ialp1), &
	                             rsf_tab(46,pind,isp1,ratindu,ial), rsf_tab(46,pind,isp1,ratindu,ialp1), &
	                             rsf_tab(46,pind,isp1,ratindu+1,ial), rsf_tab(46,pind,isp1,ratindu+1,ialp1)
	   write(*,*) '       psum_l(2),psum_u(2),psum_l(46),psum_u(46)'
	   write(*,'(1x,1p,4g12.5)') psum_l(2), psum_u2, psum_l(46),psum_u46
	   write(*,*) '       rsf(2,k), rsf(46,k)'
	   write(*,'(1x,1p,2g12.5)') rsf(2,k), rsf(46,k)
	end if
#endif
      end do Level_loop

!------------------------------------------------------------------------------
!     ... calculate total Jlong for wavelengths >200nm
!------------------------------------------------------------------------------
!     LLNL LUT approach to finding temperature index...
!     Calculate the temperature index into the cross section
!     data which lists coss sections for temperatures from
!     150 to 350 degrees K.  Make sure the index is a value
!     between 1 and 201.
!------------------------------------------------------------------------------
level_loop2 : &
      do k = 1,nlev
!----------------------------------------------------------------------
!    	... get index into xsqy_in
!----------------------------------------------------------------------
         t_index = t_in(k) - 148.5
         t_index = min( 201,max( t_index,1) )
!----------------------------------------------------------------------
! 	... find pressure
!----------------------------------------------------------------------
         if( p_in(k) >= prs(1) ) then
            do wn = 1,nw
               xswk(:numj,wn) = xsqy(:numj,wn,t_index,1)
            end do
         else if( p_in(k) <= prs(np_xs) ) then
            do wn = 1,nw
               xswk(:numj,wn) = xsqy(:numj,wn,t_index,np_xs)
            end do
         else
            do km = 2,np_xs
              if( p_in(k) >= prs(km) ) then
                pndx = km - 1
                delp = (prs(pndx) - p_in(k))*dprs(pndx)
                exit
              end if
            end do
            do wn = 1,nw
               xswk(:numj,wn) = xsqy(:numj,wn,t_index,pndx) &
                                + delp*(xsqy(:numj,wn,t_index,pndx+1) - xsqy(:numj,wn,t_index,pndx))
            end do
         end if
#ifdef USE_ESSL
         call dgemm( 'N', 'N', numj, 1, nw, &
                     1., xswk, numj, rsf(1,k), nw, &
                     0., j_long(1,k), numj )
#else
         j_long(1:numj,k) = matmul( xswk(1:numj,1:nw),rsf(1:nw,k) )
#endif
#ifdef DEBUG
         if( k == 1 .and. thisnode == 0 .and. lat == 1 .and. ip == 7 ) then
            write(*,*) 'jlong: diagnostics for xsqy(1,:) @ t_index = ',t_index
            write(*,'(1x,1p,10g12.5)') xswk(1,:)
            write(*,*) 'count of non-zero xs = ',count( xswk(1,:) /= 0. )
            write(*,*) 'jlong: diagnostics for rsf(:,k)'
            write(*,'(1x,1p,10g12.5)') rsf(:,k)
            write(*,*) 'jlong: diagnostics for jlong(:,k)'
            write(*,'(1x,1p,10g12.5)') j_long(:,k)
         end if
#endif
      end do level_loop2

      deallocate( psum_l, rsf, xswk )

      end subroutine jlong

      end module mo_jlong
