
      module mo_lghtning
!----------------------------------------------------------------------
!       ... the lightning module
!----------------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

      private
      public  :: lght_inti
      public  :: lght_no_prod
      public  :: lght_diagnostics
      public  :: prod_no
      public  :: do_lghtning

      save

      real :: csrf
      real :: factor = 1.                  ! user-controlled scaling factor to achieve arbitrary no prod.
      real :: geo_factor                   ! latitude grid cell factor
      real :: vdist(16,3)                  ! vertical distribution of lightning
      real, allocatable :: prod_no(:,:,:,:)
      real, allocatable :: glob_prod_no_col(:,:,:)
      real, allocatable :: flash_freq(:,:,:)
      real, allocatable :: lght_lndmsk(:,:,:)
      logical :: do_lghtning

      contains

      subroutine lght_inti( plonl, platl, pplon, lght_no_prd_factor )
!----------------------------------------------------------------------
!       ... initialize the lightning module
!----------------------------------------------------------------------

      use netcdf
      use mo_mpi,        only : base_lat
      use mo_constants,  only : phi, lam, d2r, pi, twopi, rearth, latwts
      use mo_grid,       only : plon, plat
      use mo_chem_utls,  only : get_spc_ndx
      use mo_control,    only : surf_flsp
      use mo_file_utils, only : open_netcdf_file
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits

      implicit none

!----------------------------------------------------------------------
!	... dummy args
!----------------------------------------------------------------------
      integer, intent(in)        :: platl, plonl, pplon
      real, intent(in), optional :: lght_no_prd_factor        ! lightning no production factor

!----------------------------------------------------------------------
!	... local variables
!----------------------------------------------------------------------
      integer :: astat
      integer :: ncid
      integer :: dimid
      integer :: vid
      integer :: gndx
      integer :: jl, ju
      integer :: nlat, nlon
      integer :: jlim(2)
      real    :: wrk2d(plon,platl)
      real, allocatable :: lats(:)
      real, allocatable :: lons(:)
      real, allocatable :: landmask(:,:)

      do_lghtning = get_spc_ndx( 'NO' ) > 0
      if( present( lght_no_prd_factor ) ) then
         if( lght_no_prd_factor /= 1. ) then
            factor = lght_no_prd_factor
         end if
      end if
      write(*,*) 'lght_inti: lightning no production scaling factor = ',factor
      csrf = twopi*rearth*rearth/real(plon)                            ! rearth in m
!----------------------------------------------------------------------
!       ... vdist(kk,itype) = % of lightning nox between (kk-1) and (kk)
!           km for profile itype
!----------------------------------------------------------------------
      vdist(:,1) = (/  3.0, 3.0, 3.0, 3.0, 3.4, 3.5, 3.6, 4.0, &       ! midlat cont
                       5.0, 7.0, 9.0, 14.0, 16.0, 14.0, 8.0, 0.5 /)
      vdist(:,2) = (/  2.5, 2.5, 2.5, 2.5, 2.5, 2.5, 2.5, 6.1, &       ! trop marine
                       17.0, 15.4, 14.5, 13.0, 12.5, 2.8, 0.9, 0.3 /)
      vdist(:,3) = (/  2.0, 2.0, 2.0, 1.5, 1.5, 1.5, 3.0, 5.8, &       ! trop cont
                       7.6, 9.6, 11.0, 14.0, 14.0, 14.0, 8.2, 2.3 /)

      allocate( prod_no(plonl,platl,plev,pplon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'lght_inti: failed to allocate prod_no; error = ',astat
	 call endrun
      end if
      allocate( flash_freq(plonl,platl,pplon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'lght_inti: failed to allocate flash_freq; error = ',astat
	 call endrun
      end if
      allocate( glob_prod_no_col(plonl,platl,pplon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'lght_inti: failed to allocate glob_prod_no_col; error = ',astat
	 call endrun
      end if
      allocate( lght_lndmsk(plonl,platl,pplon),stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'lght_inti: failed to allocate lght_lndmsk; error = ',astat
	 call endrun
      end if
      prod_no(:,:,:,:)   = 0.
      flash_freq(:,:,:)  = 0.

!-----------------------------------------------------------------------
!	... open lightning landmask netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( 'lightning_landmask.nc', surf_flsp%local_path, surf_flsp%remote_path )
!-----------------------------------------------------------------------
!       ... get grid dimensions from file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), &
                         'lght_inti: failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat ), &
                         'lght_inti: failed to get length of dimension lat' )
      allocate( lats(nlat), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'lght_inti: lats allocation error = ',astat
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'lght_inti: failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lats ), &
                         'lght_inti: failed to read variable lat' )
      lats(:nlat) = lats(:nlat) * d2r
 
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid ), &
                         'lght_inti: failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlon ), &
                         'lght_inti: failed to get length of dimension lon' )
      allocate( lons(nlon), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'lght_inti: lons allocation error = ',astat
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                         'lght_inti: failed to find variable lon' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lons ), &
                         'lght_inti: failed to read variable lon' )
      lons(:nlon) = lons(:nlon) * d2r
!-----------------------------------------------------------------------
!       ... set up regridding
!-----------------------------------------------------------------------
      gndx = regrid_inti( nlat, plat, &
                          nlon, plon, &
                          lons,  lam, &
                          lats,  phi, &
                          0, platl, &
                          do_lons=.true.,do_lats=.true. )
      deallocate( lats, lons, stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'lght_inti: failed to deallocate lats,lons; ierr = ',astat
         call endrun
      end if
      jl   = base_lat + 1
      ju   = base_lat + platl
      if( gndx /= 0 )then
         jlim = regrid_lat_limits( gndx )
      else
         jlim = (/ jl,ju /)
      end if
      allocate( landmask(nlon,jlim(1):jlim(2)),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'lght_inti: landmask allocation error = ',astat
         call endrun
      end if
!-----------------------------------------------------------------------
!	... read lightning landmask
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'LANDMASK', vid ), &
                         'lght_inti: failed to get id for LANDMASK' )
      call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                             (/ 1, jlim(1) /), &                ! start
                                             (/ nlon, jlim(2)-jlim(1)+1 /), &   ! count
                                             landmask ), &
                         'lght_inti: failed to read LANDMASK' )
!-----------------------------------------------------------------------
!	... regrid landmask
!-----------------------------------------------------------------------
      call regrid_2d( landmask(:,jlim(1):jlim(2)), wrk2d, gndx, jl, ju, do_poles=.true. )
!-----------------------------------------------------------------------
!	... force partial land to all land
!-----------------------------------------------------------------------
      where( wrk2d(:,:) /= 0. .and. wrk2d(:,:) /= 1. )
         wrk2d(:,:) = 1.
      endwhere
      lght_lndmsk(:,:,:) = reshape( wrk2d, (/plonl,platl,pplon/), order = (/ 1, 3, 2/) )

      deallocate( landmask,stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'lght_inti: failed to deallocate landmask; error = ',astat
         call endrun
      end if

      call handle_ncerr( nf_close( ncid ), 'lght_inti: failed to close netcdf file' )

      geo_factor = 1./(sum(latwts)/real(plat))

      end subroutine lght_inti

      subroutine lght_no_prod( ncdate, ncsec, caldayf, cldtop, &
			       cldbot, zm, zint, t, plonl, &
                               lat, ip )
!----------------------------------------------------------------------
!	... set no production from lightning
!----------------------------------------------------------------------

      use mo_mpi
      use mo_constants, only : dayspy, phi, latwts, lat25
      use mo_grid,      only : plat, plev, plevp, plon
      use mo_histout,   only : outfld, match_file_cnt

      implicit none

!----------------------------------------------------------------------
!	... dummy args
!----------------------------------------------------------------------
      integer, intent(in) :: ncdate                        ! date of current step (yyyymmdd)
      integer, intent(in) :: ncsec                         ! seconds of current step
      integer, intent(in) :: plonl
      integer, intent(in) :: lat
      integer, intent(in) :: ip
      real, intent(in)    :: caldayf                       ! day of year at endpoint
      real, intent(in)    :: cldtop(plonl)                 ! cloud top level index
      real, intent(in)    :: cldbot(plonl)                 ! cloud bottom level index
      real, intent(in)    :: zm(plonl,plev)                ! geopot height above surface at midpoints (m)
      real, intent(in)    :: zint(plonl,plevp)             ! geopot height above surface at interfaces (m)
      real, intent(in)    :: t(plonl,plev)                 ! temperature

!----------------------------------------------------------------------
!	... local variables
!----------------------------------------------------------------------
      real, parameter    :: land   = 1.
      real, parameter    :: secpyr = dayspy * 8.64e4

      integer :: i, istat
      integer :: cldtind, &         ! level index for cloud top
                 cldbind, &         ! level index for cloud base > 273k
                 surf_type, &
                 file               ! file index
      integer :: k, kk, zlow_ind, zhigh_ind, itype
      real    :: glob_flashfreq, &  ! global flash frequency [s-1]
                 glob_noprod, &     ! global rate of no production [as tgn/yr]
                 frac_sum           ! work variable
      real    :: zlow
      real    :: zhigh
      real    :: zlow_scal
      real    :: zhigh_scal
      real    :: fraction
      real    :: dchgz
      real    :: dchgzone(plonl)           ! depth of discharge zone [km]
      real    :: cldhgt(plonl)             ! cloud top height [km]
      real    :: cgic(plonl)               ! cloud-ground/intracloud discharge ratio
      real    :: flash_energy(plonl)       ! energy of flashes per second
      real    :: prod_no_col(plonl)        ! global no production rate for diagnostics
      real    :: wrk(plonl,plev)           ! no production rate for diagnostics

!----------------------------------------------------------------------
! 	... parameters to determine cg/ic ratio [price and rind, 1993]
!----------------------------------------------------------------------
      real, parameter  :: ca = .021
      real, parameter  :: cb = -.648
      real, parameter  :: cc = 7.49
      real, parameter  :: cd = -36.54
      real, parameter  :: ce = 64.09
      real, parameter  :: t0 = 273.
      real, parameter  :: m2km  = 1.e-3
      real, parameter  :: km2cm = 1.e5

!----------------------------------------------------------------------
!	... initialization
!----------------------------------------------------------------------
      flash_freq(:,lat,ip) = 0.
      cldhgt(:)            = 0.
      dchgzone(:)          = 0.
      cgic(:)              = 0.
      flash_energy(:)      = 0.
      prod_no_col(:)       = 0.
      glob_prod_no_col(:,lat,ip) = 0.
      do k = 1,plev
         prod_no(:,lat,k,ip)  = 0.
      end do

!--------------------------------------------------------------------------------
!	... estimate flash frequency and resulting no emissions
!           [price, penner, prather, 1997 (jgr)]
!    lightning only occurs in convective clouds with a discharge zone, i.e.
!    an altitude range where liquid water, ice crystals, and graupel coexist.
!    we test this by examining the temperature at the cloud base.
!    it is assumed that only one thunderstorm occurs per grid box, and its
!    flash frequency is determined by the maximum cloud top height (not the
!    depth of the discharge zone). this is somewhat speculative but yields
!    reasonable results.
!
!       the cg/ic ratio is determined by an empirical formula from price and
!    rind [1993]. the average energy of a cg flash is estimated as 6.7e9 j,
!    and the average energy of a ic flash is assumed to be 1/10 of that value.
!       the no production rate is assumed proportional to the discharge energy
!    with 1e17 n atoms per j. the total number of n atoms is then distributed
!    over the complete column of grid boxes.
!--------------------------------------------------------------------------------
long_loop : &
      do i = 1,plonl
!--------------------------------------------------------------------------------
! 	... find cloud top and bottom level above 273k
!--------------------------------------------------------------------------------
	 cldtind = nint( cldtop(i) )
         cldbind = nint( cldbot(i) )
         do
            if( cldbind <= cldtind .or. t(i,cldbind) < t0 ) then
	       exit
	    end if
            cldbind = cldbind - 1
         end do
cloud_layer : &
         if( cldtind < plev .and. cldtind > 0 .and. cldtind < cldbind ) then
!--------------------------------------------------------------------------------
!       ... compute cloud top height and depth of charging zone
!--------------------------------------------------------------------------------
	    cldhgt(i)   = m2km * max( 0.,zint(i,cldtind) )
            dchgz       = cldhgt(i) - m2km*zm(i,cldbind)
            dchgzone(i) = dchgz
!--------------------------------------------------------------------------------
!       ... compute flash frequency for given cloud top height
!           (flashes storm^-1 min^-1)
!--------------------------------------------------------------------------------
	    if( lght_lndmsk(i,lat,ip) == land ) then
	       flash_freq(i,lat,ip) = 3.44e-5 * cldhgt(i)**4.9 
	    else
	       flash_freq(i,lat,ip) = 6.40e-4 * cldhgt(i)**1.7
	    end if
!--------------------------------------------------------------------------------
!       ... compute cg/ic ratio
!           cgic = proportion of cg flashes (=pg from ppp paper)
!--------------------------------------------------------------------------------
            cgic(i) = 1./((((ca*dchgz + cb)*dchgz + cc) *dchgz + cd)*dchgz + ce)
            if( dchgz < 5.5 ) then
	       cgic(i) = 0.
            else if( dchgz > 14. ) then
	       cgic(i) = .02
            end if
!--------------------------------------------------------------------------------
!       ... compute flash energy (cg*6.7e9 + ic*6.7e8)
!           and convert to total energy per second
!           set ic = cg
!--------------------------------------------------------------------------------
            flash_energy(i) = 6.7e9 * flash_freq(i,lat,ip)/60.
!--------------------------------------------------------------------------------
!       ... LKE Aug 23, 2005: scale production to account for different grid
!           box sizes. This requires a reduction in the overall fudge factor 
!           (e.g., from 1.2 to 0.5)
!--------------------------------------------------------------------------------
            flash_energy(i) =  flash_energy(i) * latwts(base_lat+lat) * geo_factor
!--------------------------------------------------------------------------------
! 	... compute number of n atoms produced per second
!           and convert to n atoms per second per cm2 and apply fudge factor
!--------------------------------------------------------------------------------
            prod_no_col(i) = 1.e17*flash_energy(i) &
                                  /(1.e4*csrf*latwts(base_lat+lat)) * factor
!--------------------------------------------------------------------------------
! 	... compute global no production rate in tgn/yr:
!           tgn per second: * 14.00674 * 1.65979e-24 * 1.e-12
!             nb: 1.65979e-24 = 1/avo
!           tgn per year: * secpyr
!--------------------------------------------------------------------------------
            glob_prod_no_col(i,lat,ip) = 1.e17*flash_energy(i) &
                                              * 14.00674 * 1.65979e-24 * 1.e-12 * secpyr * factor
	 end if cloud_layer
      end do long_loop

!--------------------------------------------------------------------------------
!	... distribute production up to cloud top [pickering et al., 1998 (jgr)]
!--------------------------------------------------------------------------------
      do i = 1,plonl
 	 if( prod_no_col(i) > 0. .and. cldhgt(i) > 0. ) then
            if( abs( phi(base_lat+lat) ) > lat25 ) then
               itype = 1                              ! midlatitude continental
            else if( lght_lndmsk(i,lat,ip) == land ) then
               itype = 3                              ! tropical continental
            else
               itype = 2                              ! topical marine
            end if
            frac_sum = 0.
	    cldtind = nint( cldtop(i) )
level_loop : &
            do k = cldtind,plev
               zlow       = zint(i,k+1) * m2km                                  ! lower interface height (km)
               zlow_scal  = zlow * 16./cldhgt(i)                                ! scale to 16 km convection height
               zlow_ind   = max( 1,int(zlow_scal)+1 )                           ! lowest vdist index to include in layer
               zhigh      = zint(i,k) * m2km                                    ! upper interface height (km)
               zhigh_scal = zhigh * 16./cldhgt(i)                               ! height (km) scaled to 16km convection height
               zhigh_ind  = max( 1,min( 16,int(zhigh_scal)+1 ) )                ! highest vdist index to include in layer
               do kk = zlow_ind,zhigh_ind
                  fraction = min( zhigh_scal,real(kk) ) &                       ! fraction of vdist in this model layer
                             - max( zlow_scal,real(kk-1) )
                  fraction = max( 0., min( 1.,fraction ) )
                  frac_sum = frac_sum + fraction*vdist(kk,itype)
                  prod_no(i,lat,k,ip) = prod_no(i,lat,k,ip) &                   ! sum the fraction of column nox in layer k
                                        + fraction*vdist(kk,itype)*.01
               end do
               prod_no(i,lat,k,ip) = prod_no_col(i) * prod_no(i,lat,k,ip) &     ! multiply fraction by column amount
                                                    / (km2cm*(zhigh - zlow))    ! and convert to atom n cm^-3 s^-1
            end do level_loop
	 end if
      end do

!--------------------------------------------------------------------------------
!       ... output lightning no production to history file
!--------------------------------------------------------------------------------
      do k = 1,plev
         wrk(:,k) = prod_no(:,lat,k,ip)
      end do
      do file = 1,match_file_cnt
         call outfld( 'LNO_COL_PROD', glob_prod_no_col(:,lat,ip), plonl, ip, lat, file )
         call outfld( 'LNO_PROD', wrk, plonl, ip, lat, file )
         call outfld( 'FLASHFRQ', flash_freq(:,lat,ip), plonl, ip, lat, file )
         call outfld( 'CLDHGT', cldhgt, plonl, ip, lat, file )
         call outfld( 'DCHGZONE', dchgzone, plonl, ip, lat, file )
         call outfld( 'CGIC', cgic, plonl, ip, lat, file )
      end do

      end subroutine lght_no_prod

      subroutine lght_diagnostics( plonl, platl, pplon )
!----------------------------------------------------------------------
!	... output lightning diagnostics
!----------------------------------------------------------------------

      use mo_mpi
      use mo_grid,      only : plat, plon

      implicit none

!----------------------------------------------------------------------
!	... dummy args
!----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon

!----------------------------------------------------------------------
!	... local variables
!----------------------------------------------------------------------
      integer :: i, ip, j
      integer :: istat
      integer :: node
      real    :: glob_flashfreq
      real    :: glob_noprod
      real    :: wrk
      real    :: gather_buff(plonl,platl,pplon,maxnodes)

!--------------------------------------------------------------------------------
! 	... accumulate global total, convert to flashes per second
!--------------------------------------------------------------------------------
      glob_flashfreq = sum( flash_freq(:,:,:) )/60.

!--------------------------------------------------------------------------------
! 	... accumulate global no production rate
!--------------------------------------------------------------------------------
      glob_noprod = 0.
#ifdef USE_MPI
      call mpi_reduce( glob_flashfreq, wrk, 1, mpi_double_precision, mpi_sum, 0, mpi_comm_comp, istat )
      if( istat /= mpi_success ) then
         write(*,*) 'lght_diagnostics: mpi_allreduce for flashfreq failed; error = ',istat
	 call endrun
      end if
      call mpi_gather( glob_prod_no_col, plon*platl, mpi_double_precision, &
		       gather_buff, plon*platl, mpi_double_precision, 0, mpi_comm_comp, istat )
      if( istat /= mpi_success ) then
         write(*,*) 'lght_diagnostics: mpi_gather for prod_no_col failed; error = ',istat
	 call endrun
      end if
      if( masternode ) then
         do node = 1,maxnodes
	    do j = 1,platl
	       do ip = 1,pplon
	          do i = 1,plonl
		     glob_noprod = glob_noprod + gather_buff(i,j,ip,node)
	          end do
	       end do
	    end do
         end do
      end if
#else
      wrk = glob_flashfreq
      if( pplon > 1 ) then
         do j = 1,plat
            do ip = 1,pplon
	       do i = 1,plonl
                  glob_noprod = glob_noprod + glob_prod_no_col(i,j,ip)
	       end do
            end do
         end do
      else
         do j = 1,plat
	    do i = 1,plonl
               glob_noprod = glob_noprod + glob_prod_no_col(i,j,1)
	    end do
         end do
      end if
#endif

      if( masternode ) then
         write(*,*) ' '
!        write(*,'('' global flash freq (/s), lightning nox (tgn/y) = '',2f10.4)') &
!                     glob_flashfreq, glob_noprod
         write(*,*) 'lght_diagnostics : global flash freq (/s), lightning nox (tgn/y) = ', wrk, glob_noprod
         write(*,*) ' '
      end if

      end subroutine lght_diagnostics

      end module mo_lghtning
