
      module mo_photo
!----------------------------------------------------------------------
!	... photolysis interp table and related arrays
!----------------------------------------------------------------------

      implicit none

      private
      public :: photo_inti, table_photo, xactive_photo
      public :: set_ub_col, setcol, diurnal_geom, sundis, photo_timestep_init

      save

      real, parameter :: kg2g = 1.e3

      integer ::  jno_ndx
      integer ::  jonitr_ndx
      integer ::  jho2no2_ndx, jho2no2_a_ndx
      integer ::  jch3cho_a_ndx, jch3cho_b_ndx, jch3cho_c_ndx
      integer ::  ox_ndx, o3_ndx, o3_inv_ndx, o3rad_ndx
      integer ::  oc1_ndx, oc2_ndx
      integer ::  cb1_ndx, cb2_ndx
      integer ::  soa_ndx
      integer ::  ant_ndx
      integer ::  so4_ndx
      integer ::  sa1_ndx, sa2_ndx, sa3_ndx, sa4_ndx
      integer ::  n2_ndx, no_ndx, o2_ndx
      integer, allocatable :: lng_indexer(:)

      integer              :: ki
      integer              :: last
      integer              :: next
      integer              :: nlev
      real                 :: delp
      real                 :: dels
      real, allocatable    :: days(:)
      real, allocatable    :: levs(:)
      real, allocatable    :: o2_exo_coldens(:,:,:)
      real, allocatable    :: o3_exo_coldens(:,:,:)
      logical              :: o2_is_inv
      logical              :: n2_is_inv
      logical              :: o3_is_inv
      logical              :: no_is_inv
      logical              :: has_o2_col
      logical              :: has_o3_col
      logical              :: has_fixed_press

      contains

      subroutine photo_inti( plonl, platl, pplon )
!----------------------------------------------------------------------
!	... initialize photolysis module
!----------------------------------------------------------------------

      use netcdf
      use mo_control,    only : photo_flsp, xactive_prates
      use mo_constants,  only : d2r, phi
      use mo_mpi,        only : base_lat
      use mo_grid,       only : plat, plevp
      use mo_photoin,    only : photoin_inti
      use mo_tuv_inti,   only : tuv_inti
      use mo_tuv_inti,   only : nlng
      use mo_seto2,      only : o2_xsect_inti
      use mo_file_utils, only : open_netcdf_file
      use mo_regrider,   only : regrid_inti, regrid_1d, regrid_lat_limits
      use chem_mods,     only : ncol_abs, phtcnt
      use chem_mods,     only : rxt_tag_lst, pht_alias_lst, pht_alias_mult
      use mo_setinv,     only : o2_ndx
      use mo_calendar,   only : caldayr
      use plevs,         only : ps0, hybi, hyai
      use m_types,       only : filespec
      use mo_chem_utls,  only : get_rxt_ndx, get_spc_ndx, get_inv_ndx
      use mo_jlong,      only : jlong_init

      implicit none

!----------------------------------------------------------------------
!	... dummy arguments
!----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon

!----------------------------------------------------------------------
!	... local variables
!----------------------------------------------------------------------
      real, parameter   :: hPa2Pa = 100.
      integer           :: k, n
      integer           :: ncid
      integer           :: vid
      integer           :: dimid
      integer           :: nlat
      integer           :: ntimes
      integer           :: astat
      integer           :: gndx
      integer           :: ndx
      integer           :: spc_ndx
      integer           :: jl, ju
      integer           :: jlim(2)
      integer, allocatable :: dates(:)
      real              :: pinterp
      real              :: wrk(platl)
      real, allocatable :: lats(:)
      real, allocatable :: coldens(:,:)
      character(len=80) :: lpath
      character(len=80) :: mspath

      allocate( lng_indexer(phtcnt),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'photo_inti: failed to allocate lng_indexer; error = ',astat
         call endrun
      end if
      lng_indexer(:) = 0
!----------------------------------------------------------------------
!	... call module initializers
!----------------------------------------------------------------------
      jno_ndx  = get_rxt_ndx( 'jno' )
      lpath    = photo_flsp%local_path
      mspath   = photo_flsp%remote_path
is_xactive : &
      if( xactive_prates ) then
         jch3cho_a_ndx = get_rxt_ndx( 'jch3cho_a' )
         jch3cho_b_ndx = get_rxt_ndx( 'jch3cho_b' )
         jch3cho_c_ndx = get_rxt_ndx( 'jch3cho_c' )
         jonitr_ndx    = get_rxt_ndx( 'jonitr' )
         jho2no2_ndx   = get_rxt_ndx( 'jho2no2' )
         call tuv_inti( plevp, lng_indexer )
      else is_xactive
         jho2no2_a_ndx = get_rxt_ndx( 'jho2no2_a' )
         call jlong_init( lng_indexer )
      end if is_xactive
!----------------------------------------------------------------------
!        ... check that each photorate is in long dataset
!----------------------------------------------------------------------
      if( any( abs(lng_indexer(:)) == 0 ) ) then
         write(*,*) ' '
         write(*,*) 'photo_inti: the following photorate(s) are not in'
         write(*,*) '            the xactive cross section dataset'
         write(*,*) ' '
         do ndx = 1,phtcnt
            if( abs(lng_indexer(ndx)) == 0 ) then
               write(*,*) '           ',trim( rxt_tag_lst(ndx) )
            end if
         end do
         call endrun
      end if
!----------------------------------------------------------------------
!        ... output any aliased photorates
!----------------------------------------------------------------------
      if( any( pht_alias_lst(:,2) /= ' ' ) ) then
         write(*,*) ' '
         write(*,*) 'photo_inti: the following long photorate(s) are aliased'
         write(*,*) ' '
         do ndx = 1,phtcnt
            if( pht_alias_lst(ndx,2) /= ' ' ) then
               if( pht_alias_mult(ndx,2) == 1. ) then
                  write(*,*) '           ',trim(rxt_tag_lst(ndx)),' -> ',trim(pht_alias_lst(ndx,2))
               else
                  write(*,*) '           ',trim(rxt_tag_lst(ndx)),' -> ',pht_alias_mult(ndx,2),'*',trim(pht_alias_lst(ndx,2))
               end if
            end if
         end do
      end if
      if( xactive_prates ) then
         call o2_xsect_inti
         call photoin_inti( platl, nlng, lng_indexer )
      end if

      ox_ndx     = get_spc_ndx( 'OX' )
      if( ox_ndx < 1 ) then
         ox_ndx  = get_spc_ndx( 'O3' )
      end if
      o3_ndx     = get_spc_ndx( 'O3' )
      o3rad_ndx  = get_spc_ndx( 'O3RAD' )
      o3_inv_ndx = get_inv_ndx( 'O3' )

      n2_ndx     = get_inv_ndx( 'N2' )
      n2_is_inv  = n2_ndx > 0
      if( .not. n2_is_inv ) then
         n2_ndx = get_spc_ndx( 'N2' )
      end if
      o2_ndx     = get_inv_ndx( 'O2' )
      o2_is_inv  = o2_ndx > 0
      if( .not. o2_is_inv ) then
         o2_ndx = get_spc_ndx( 'O2' )
      end if
      no_ndx     = get_spc_ndx( 'NO' )
      no_is_inv  = no_ndx < 1
      if( no_is_inv ) then
         no_ndx = get_inv_ndx( 'NO' )
      end if
      o3_is_inv  = o3_ndx < 1

      oc1_ndx    = get_spc_ndx( 'OC1' )
      oc2_ndx    = get_spc_ndx( 'OC2' )
      cb1_ndx    = get_spc_ndx( 'CB1' )
      cb2_ndx    = get_spc_ndx( 'CB2' )
      soa_ndx    = get_spc_ndx( 'SOA' )
      ant_ndx    = get_spc_ndx( 'NH4NO3' )
      so4_ndx    = get_spc_ndx( 'SO4' )
      sa1_ndx    = get_spc_ndx( 'SA1' )
      sa2_ndx    = get_spc_ndx( 'SA2' )
      sa3_ndx    = get_spc_ndx( 'SA3' )
      sa4_ndx    = get_spc_ndx( 'SA4' )

!----------------------------------------------------------------------
!	... check for o2, o3 absorber columns
!----------------------------------------------------------------------
      if( ncol_abs > 0 ) then
         spc_ndx = ox_ndx
         if( spc_ndx < 1 ) then
            spc_ndx = o3_ndx
         end if
         if( spc_ndx > 0 ) then
            has_o3_col = .true.
         else
            has_o3_col = .false.
         end if
         if( ncol_abs > 1 ) then
            if( o2_ndx > 1 ) then
               has_o2_col = .true.
            else
               has_o2_col = .false.
            end if
         else
            has_o2_col = .false.
         end if
      else
         has_o2_col = .false.
         has_o3_col = .false.
      end if

has_abs_columns : &
      if( has_o2_col .or. has_o3_col ) then
!-----------------------------------------------------------------------
!	... open exo coldens file
!-----------------------------------------------------------------------
         ncid = open_netcdf_file( 'exo_coldens.nc', lpath, mspath )
!-----------------------------------------------------------------------
!       ... get grid dimensions from file
!-----------------------------------------------------------------------
!       ... timing
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'month', dimid ), &
                            'photo_inti: failed to find dimension month' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, ntimes ), &
                            'photo_inti: failed to get length of dimension month' )
         if( ntimes /= 12 ) then
            write(*,*) 'photo_inti: exo coldens is not annual period'
            call endrun
         end if
         allocate( dates(ntimes),days(ntimes),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photo_inti: dates,days allocation error = ',astat
            call endrun
         end if
         dates(:) = (/ 116, 214, 316, 415,  516,  615, &
                       716, 816, 915, 1016, 1115, 1216 /)
!-----------------------------------------------------------------------
!	... initialize the monthly day of year times
!-----------------------------------------------------------------------
	 do n = 1,ntimes
	    days(n) = caldayr( dates(n), 0 )
	 end do
         deallocate( dates )
!-----------------------------------------------------------------------
!       ... latitudes
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), &
                            'photo_inti: failed to find dimension lat' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat ), &
                            'photo_inti: failed to get length of dimension lat' )
         allocate( lats(nlat), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photo_inti: lats allocation error = ',astat
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                            'photo_inti: failed to find variable lat' )
         call handle_ncerr( nf_get_var_double( ncid, vid, lats ), &
                            'photo_inti: failed to read variable lat' )
         lats(:nlat) = lats(:nlat) * d2r
!-----------------------------------------------------------------------
!       ... levels
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lev', dimid ), &
                            'photo_inti: failed to find dimension lev' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlev ), &
                            'photo_inti: failed to get length of dimension lev' )
         allocate( levs(nlev), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photo_inti: levs allocation error = ',astat
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lev', vid ), &
                            'photo_inti: failed to find variable lev' )
         call handle_ncerr( nf_get_var_double( ncid, vid, levs ), &
                            'photo_inti: failed to read variable lev' )
         levs(:nlev) = levs(:nlev) * hPa2Pa
!-----------------------------------------------------------------------
!       ... set up regridding
!-----------------------------------------------------------------------
         gndx = regrid_inti( nlat, plat, &
                             nlat, plat, &
                             lats,  phi, &
                             lats,  phi, &
                             0, platl, &
                             do_lons = .false., &
                             do_lats = .true. )
         deallocate( lats,stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photo_inti: failed to deallocate lats; error = ',astat
            call endrun
         end if
         jl   = base_lat + 1
         ju   = base_lat + platl
         if( gndx /= 0 )then
            jlim = regrid_lat_limits( gndx )
         else
            jlim = (/ jl,ju /)
         end if
         allocate( coldens(jlim(1):jlim(2),nlev),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photo_inti: coldens allocation error = ',astat
            call endrun
         end if
         if( has_o2_col ) then
            allocate( o2_exo_coldens(nlev,platl,ntimes),stat=astat )
            if( astat /= 0 ) then
               write(*,*) 'photo_inti: o2_exo_coldens allocation error = ',astat
               call endrun
            end if
         end if
         if( has_o3_col ) then
            allocate( o3_exo_coldens(nlev,platl,ntimes),stat=astat )
            if( astat /= 0 ) then
               write(*,*) 'photo_inti: o3_exo_coldens allocation error = ',astat
               call endrun
            end if
         end if
!-----------------------------------------------------------------------
!	... read exo coldens
!-----------------------------------------------------------------------
         do n = 1,ntimes
            if( has_o2_col ) then
               call handle_ncerr( nf_inq_varid( ncid, 'O2_column_density', vid ), &
                                  'photo_inti: failed to get id for O2_column_density' )
               call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                                      (/ jlim(1), 1, n /), &                ! start
                                                      (/ jlim(2)-jlim(1)+1, nlev, 1 /), &   ! count
                                                      coldens ), &
                                  'photo_inti: failed to read O2_column_density' )
!-----------------------------------------------------------------------
!	... regrid exo coldens
!-----------------------------------------------------------------------
               do k = 1,nlev
                  call regrid_1d( coldens(jlim(1):jlim(2),k), &
                                  wrk, gndx, do_lat=.true., to_lat_min=jl, to_lat_max=ju )
                  o2_exo_coldens(k,:,n) = wrk(:)
               end do
            end if
            if( has_o3_col ) then
               call handle_ncerr( nf_inq_varid( ncid, 'O3_column_density', vid ), &
                                  'photo_inti: failed to get id for O3_column_density' )
               call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                                      (/ jlim(1), 1, n /), &                ! start
                                                      (/ jlim(2)-jlim(1)+1, nlev, 1 /), &   ! count
                                                      coldens ), &
                                  'photo_inti: failed to read O3_column_density' )
!-----------------------------------------------------------------------
!	... regrid exo coldens
!-----------------------------------------------------------------------
               do k = 1,nlev
                  call regrid_1d( coldens(jlim(1):jlim(2),k), &
                                  wrk, gndx, do_lat=.true., to_lat_min=jl, to_lat_max=ju )
                  o3_exo_coldens(k,:,n) = wrk(:)
               end do
            end if
         end do

         call handle_ncerr( nf_close( ncid ), 'photo_inti: failed to close netcdf file' )

         deallocate( coldens,stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photo_inti: failed to deallocate coldens; error = ',astat
            call endrun
         end if
         has_fixed_press = hybi(1) == 0.
!-----------------------------------------------------------------------
!	... setup the pressure interpolation
!-----------------------------------------------------------------------
         if( has_fixed_press ) then
            pinterp = ps0 * hyai(1)
            if( pinterp <= levs(1) ) then
               ki   = 1
               delp = 0.
            else
               do ki = 2,nlev
                  if( pinterp <= levs(ki) ) then
                     delp = log( pinterp/levs(ki-1) )/log( levs(ki)/levs(ki-1) )
                     exit
                  end if
               end do
            end if
#ifdef DEBUG
            write(*,*) '-----------------------------------'
            write(*,*) 'photo_inti: diagnostics'
            write(*,*) 'ki, delp = ',ki,delp
            write(*,*) 'pinterp,levs(ki-1:ki) = ',pinterp,levs(ki-1:ki)
            write(*,*) '-----------------------------------'
#endif
         end if
      end if has_abs_columns

      end subroutine photo_inti

      subroutine table_photo( photos, pmid, pdel, temper, zmid, &
                              col_dens, zen_angle, srf_alb, lwc, clouds, &
                              esfact, vmr, invariants, plonl )
!-----------------------------------------------------------------
!   	... table photorates
!-----------------------------------------------------------------

      use mo_grid,      only : plev
      use chem_mods,    only : ncol_abs, phtcnt, nfs, indexm, pcnstm1
      use chem_mods,    only : pht_alias_mult
      use mo_constants, only : r2d
      use mo_jlong,     only : jlong
      use mo_jlong,     only : nlng => numj

      implicit none

!-----------------------------------------------------------------
!   	... dummy arguments
!-----------------------------------------------------------------
      integer, intent(in) :: plonl
      real, intent(in)    :: esfact                              ! earth sun distance factor
      real, intent(in)    :: vmr(plonl,plev,max(1,pcnstm1))      ! species concentrations (mol/mol)
      real, intent(in)    :: invariants(plonl,plev,max(1,nfs))   ! invariant concentrations (molec/cm^3)
      real, intent(in)    :: col_dens(:,:,:)                     ! column densities (molec/cm^2)
      real, intent(in)    :: zen_angle(plonl)                    ! solar zenith angle (radians)
      real, intent(in)    :: srf_alb(plonl)                      ! surface albedo
      real, intent(in)    :: lwc(plonl,plev)                     ! liquid water content (kg/kg)
      real, intent(in)    :: clouds(plonl,plev)                  ! cloud fraction
      real, intent(in)    :: pmid(plonl,plev)                    ! midpoint pressure (Pa)
      real, intent(in)    :: pdel(plonl,plev)                    ! pressure delta about midpoint (Pa)
      real, intent(in)    :: zmid(plonl,plev)                    ! midpoint height (km)
      real, intent(in)    :: temper(plonl,plev)                  ! midpoint temperature (K)
      real, intent(inout) :: photos(:,:,:)                       ! photodissociation rates (1/s)

!-----------------------------------------------------------------
!    	... local variables
!-----------------------------------------------------------------
      real, parameter :: Pa2mb         = 1.e-2       ! pascals to mb
      real, parameter :: max_zen_angle = 88.85       ! degrees

      integer ::  i, k, m, n                 ! indicies
      integer ::  astat
      real    ::  sza
      real    ::  alias_factor
      real    ::  fac1(plev)                 ! work space for j(no) calc
      real    ::  fac2(plev)                 ! work space for j(no) calc
      real    ::  colo3(plev)                ! vertical o3 column density
      real    ::  parg(plev)                 ! vertical pressure array (hPa)
      real    ::  tline(plev)                ! vertical temperature array
      real    ::  cld_line(plev)             ! vertical cloud array
      real    ::  lwc_line(plev)             ! vertical lwc array
      real    ::  eff_alb(plev)              ! effective albedo from cloud modifications
      real    ::  cld_mult(plev)             ! clould multiplier
      real    ::  tmp(plonl,plev)            ! wrk array
      real, allocatable ::  lng_prates(:,:)  ! photorates matrix (1/s)

!-----------------------------------------------------------------
!	... allocate long rates work array
!-----------------------------------------------------------------
      allocate( lng_prates(nlng,plev),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'table_photo: failed to allocate lng_prates; error = ',astat
         call endrun
      end if

!-----------------------------------------------------------------
!	... zero all photorates
!-----------------------------------------------------------------
      do m = 1,max(1,phtcnt)
         do k = 1,plev
            photos(:,k,m) = 0.
         end do
      end do

long_loop : &
      do i = 1,plonl
        sza = zen_angle(i)*r2d
daylight : &
        if( sza >= 0. .and. sza < max_zen_angle ) then
           parg(:)     = Pa2mb*pmid(i,:)
           colo3(:)    = col_dens(i,:,1)
           fac1(:)     = pdel(i,:)
           tline(:)    = temper(i,:)
           lwc_line(:) = lwc(i,:)
           cld_line(:) = clouds(i,:)
           call cloud_mod( zen_angle(i), cld_line, lwc_line, fac1, srf_alb(i), &
                           eff_alb, cld_mult )
           cld_mult(:) = esfact * cld_mult(:)
           call jlong( plev, sza, eff_alb, parg, tline, &
                       colo3, lng_prates )
           do m = 1,phtcnt
              if( lng_indexer(m) > 0 ) then
                 alias_factor = pht_alias_mult(m,2)
                 if( alias_factor == 1. ) then
                    photos(i,:,m) = photos(i,:,m) + lng_prates(lng_indexer(m),:)
                 else
                    photos(i,:,m) = photos(i,:,m) + alias_factor * lng_prates(lng_indexer(m),:)
                 end if
              end if
           end do
           do m = 1,phtcnt
              photos(i,:,m) = photos(i,:,m)*cld_mult(:)
           end do
!-----------------------------------------------------------------
!	... calculate j(no) from formula
!-----------------------------------------------------------------
           if( jno_ndx > 0 ) then
              if( has_o2_col .and. has_o3_col ) then
                 fac1(:) = 1.e-8 * (col_dens(i,:,2)/cos(zen_angle(i)))**.38
                 fac2(:) = 5.e-19 * col_dens(i,:,1) / cos(zen_angle(i))
                 photos(i,:,jno_ndx) = 4.5e-6 * exp( -(fac1(:) + fac2(:)) )
              end if
           end if
!-----------------------------------------------------------------
! 	... add near IR correction to ho2no2
!-----------------------------------------------------------------
           if( jho2no2_a_ndx > 0 ) then
              photos(i,:,jho2no2_a_ndx) = photos(i,:,jho2no2_a_ndx) + 1.e-5*cld_mult(:)
           end if
        end if daylight
      end do long_loop

      deallocate( lng_prates )

      end subroutine table_photo

      subroutine xactive_photo( photos, vmr, temper, cwat, cldfr, &
                                pmid, zmid, col_dens, zen_angle, srf_alb, &
                                tdens, ps, ts, lat, ip, &
                                ncdate, ncsec, sunon, sunoff, esfact, &
                                relhum, dust, dt_diag, fracday, plonl )
!-----------------------------------------------------------------
!   	... fast online photo rates
!-----------------------------------------------------------------

      use mo_grid,      only : plev, plevp, pcnstm1
      use mo_control,   only : use_dust
      use chem_mods,    only : ncol_abs, phtcnt
      use chem_mods,    only : pht_alias_mult
      use mo_dust,      only : nqdust
      use mo_params,    only : kz, kw
      use mo_wavelen,   only : nw
      use mo_constants, only : phi, lam, r2d, boltz
      use mo_photoin,   only : photoin
      use mo_tuv_inti,  only : nlng
      use mo_mpi,       only : base_lat, thisnode

      implicit none

!-----------------------------------------------------------------
!   	... dummy arguments
!-----------------------------------------------------------------
      integer, intent(in) :: plonl                         ! long tile dim
      integer, intent(in) :: lat                           ! lat tile index
      integer, intent(in) :: ip                            ! long tile index
      integer, intent(in) :: ncdate, ncsec                 ! current date(yyyymmdd) and seconds in date
      real, intent(in)    :: sunon                         ! angle for sunrise (radians)
      real, intent(in)    :: sunoff                        ! angle for sunset (radians)
      real, intent(in)    :: esfact                        ! earth sun distance factor
      real, intent(in)    :: ps(plonl)                     ! surface pressure (Pa)
      real, intent(in)    :: ts(plonl)                     ! surface temperature (K)
      real, intent(in)    :: col_dens(plonl,plev,ncol_abs) ! column densities (molecules/cm^2)
      real, intent(in)    :: zen_angle(plonl)              ! solar zenith angle (radians)
      real, intent(in)    :: srf_alb(plonl)                ! surface albedo
      real, intent(in)    :: tdens(plonl,plev)             ! total atms density (molecules/cm^3)
      real, intent(in)    :: vmr(plonl,plev,pcnstm1)       ! species concentration (mol/mol)
      real, intent(in)    :: pmid(plonl,plev)              ! midpoint pressure (Pa)
      real, intent(in)    :: zmid(plonl,plev)              ! midpoint height (m)
      real, intent(in)    :: temper(plonl,plev)            ! midpoint temperature (K)
      real, intent(in)    :: relhum(plonl,plev)            ! relative humidity
      real, intent(in)    :: cwat(plonl,plev)              ! cloud water (kg/kg)
      real, intent(in)    :: cldfr(plonl,plev)             ! cloud fraction
      real, intent(in)    :: dust(plonl,plev,nqdust)       ! dust concentration (mol/mol)
      real, intent(inout) :: photos(plonl,plev,phtcnt)     ! photodissociation rates (1/s)
      real, intent(inout) :: dt_diag(plonl,7)              ! od diagnostics
      real, intent(out)   :: fracday(plonl)                ! fraction of day
!-----------------------------------------------------------------
!    	... local variables
!-----------------------------------------------------------------
      integer, parameter ::  k_diag = 3

      integer  ::  i                      ! index
      integer  ::  k                      ! index
      integer  ::  m                      ! index
      integer  ::  spc_ndx                ! index
      integer  ::  file                   ! index
      integer  ::  astat                  ! allocate return code
      logical  ::  zagtz(plonl)           ! zenith angle > 0 flag array

      real    ::   secant
      real    ::   alat
      real    ::   along
      real    ::   ut
      real    ::   alias_factor
      real    ::   dt_xdiag(7)                   ! wrk array
      real    ::   fac1(plev)                    ! work space for j(no) calc
      real    ::   fac2(plev)                    ! work space for j(no) calc
      real    ::   tlay(plev)                    ! vertical temperature array at layer midpoint
      real    ::   tline(plevp)                  ! vertical temperature array
      real    ::   xlwc(plevp)                   ! cloud water (kg/kg)
      real    ::   xfrc(plevp)                   ! cloud fraction      xuexi
      real    ::   airdens(plevp)                ! atmospheric density
      real    ::   o3line(plevp)                 ! vertical o3 vmr
      real    ::   aerocs1(plevp)   
      real    ::   aerocs2(plevp)   
      real    ::   aercbs1(plevp)   
      real    ::   aercbs2(plevp)   
      real    ::   aersoa(plevp)   
      real    ::   aerant(plevp)   
      real    ::   aerso4(plevp)   
      real    ::   aerds1(plevp)   
      real    ::   aerds2(plevp)   
      real    ::   aerds3(plevp)   
      real    ::   aerds4(plevp)   
      real    ::   rh(plevp)   
      real    ::   zarg(plevp)                   ! vertical height array
      real    ::   aersal(plevp,4)
      real    ::   albedo(kw)                    ! wavelength dependent albedo
      real, allocatable :: prates(:,:)           ! photorates matrix

!-----------------------------------------------------------------
!	... allocate "working" rate array
!-----------------------------------------------------------------
      allocate( prates(plevp,nlng), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'xactive_photo: failed to allocate prates; error = ',astat
         call endrun
      end if
!-----------------------------------------------------------------
!	... zero all photorates
!-----------------------------------------------------------------
      do m = 1,phtcnt
         do k = 1,plev
            photos(:,k,m) = 0.
         end do
      end do
      zagtz(:) = zen_angle(:) > 0.
      fracday(:) = 0.
      ut   = real(ncsec)/3600.
      alat = r2d * phi(base_lat+lat)
#ifdef DEBUG
      write(*,*) 'photo: entered for lat,ip = ',lat,ip
      write(*,*) 'photo: nj = ',nlng
      write(*,*) 'photo: esfact = ',esfact
#endif
Long_loop : &
      do i = 1,plonl
daylight : &
         if( zagtz(i) ) then
            secant = 1. / cos( zen_angle(i) )
            if( secant <= 50. ) then
               fracday(i) = 1.
               zarg(plevp:2:-1)     = zmid(i,:)
               zarg(1)              = 0.
               airdens(plevp:2:-1)  = tdens(i,:)
               airdens(1)           = 10. * ps(i) / (boltz*ts(i))
               if( o3rad_ndx > 0 ) then
                  spc_ndx = o3rad_ndx
               else
                  spc_ndx = ox_ndx
               end if
               if( spc_ndx < 1 ) then
	          spc_ndx = o3_ndx
               end if
               if( spc_ndx > 0 ) then
                  o3line(plevp:2:-1) = vmr(i,:,spc_ndx)
               else
                  o3line(plevp:2:-1) = 0.
               end if
               o3line(1)            = o3line(2)
               tline(plevp:2:-1)    = temper(i,:)
               tline(1)             = tline(2)
               rh(plevp:2:-1)       = relhum(i,:)
               rh(1)                = rh(2)
               xlwc(plevp:2:-1)     = cwat(i,:) * pmid(i,:)/(temper(i,:)*287.) * kg2g  !! TIE
               xlwc(1)              = xlwc(2)
               xfrc(plevp:2:-1)     = cldfr(i,:)                      ! cloud fraction
               xfrc(1)              = xfrc(2)
               tlay(1:plev)         = .5*(tline(1:plev) + tline(2:plevp))
               albedo(1:nw)         = srf_alb(i)
               along                = r2d * lam((ip-1)*plonl+i)
               if( oc1_ndx > 0 ) then
                  aerocs1(plevp:2:-1) = vmr(i,:,oc1_ndx)
               else
                  aerocs1(plevp:2:-1) = 0.
               end if
               aerocs1(1)            = aerocs1(2)
               if( oc2_ndx > 0 ) then
                  aerocs2(plevp:2:-1) = vmr(i,:,oc2_ndx)
               else
                  aerocs2(plevp:2:-1) = 0.
               end if
               aerocs2(1)          = aerocs2(2)
               if( cb1_ndx > 0 ) then
                  aercbs1(plevp:2:-1) = vmr(i,:,cb1_ndx)
               else
                  aercbs1(plevp:2:-1) = 0.
               end if
               aercbs1(1)          = aercbs1(2)
               if( cb2_ndx > 0 ) then
                  aercbs2(plevp:2:-1) = vmr(i,:,cb2_ndx)
               else
                  aercbs2(plevp:2:-1) = 0.
               end if
               aercbs2(1)          = aercbs2(2)
               if( soa_ndx > 0 ) then
                  aersoa(plevp:2:-1) = vmr(i,:,soa_ndx)
               else
                  aersoa(plevp:2:-1) = 0.
               end if
               aersoa(1)          = aersoa(2)
               if( ant_ndx > 0 ) then
                  aerant(plevp:2:-1) = vmr(i,:,ant_ndx)
               else
                  aerant(plevp:2:-1) = 0.
               end if
               aerant(1)            = aerant(2)
               if( so4_ndx > 0 ) then
                  aerso4(plevp:2:-1) = vmr(i,:,so4_ndx)
               else
                  aerso4(plevp:2:-1) = 0.
               end if
               aerso4(1)            = aerso4(2)
               if( use_dust ) then
                  aerds1(plevp:2:-1) = dust(i,:,1)
                  aerds2(plevp:2:-1) = dust(i,:,2)
                  aerds3(plevp:2:-1) = dust(i,:,3)
                  aerds4(plevp:2:-1) = dust(i,:,4)
                  aerds1(1)          = aerds1(2)
                  aerds2(1)          = aerds2(2)
                  aerds3(1)          = aerds3(2)
                  aerds4(1)          = aerds4(2)
               end if
               if( sa1_ndx > 0 ) then
                  aersal(plevp:2:-1,1) = vmr(i,:,sa1_ndx)
               else
                  aersal(plevp:2:-1,1) = 0.
               end if
               if( sa2_ndx > 0 ) then
                  aersal(plevp:2:-1,2) = vmr(i,:,sa2_ndx)
               else
                  aersal(plevp:2:-1,2) = 0.
               end if
               if( sa3_ndx > 0 ) then
                  aersal(plevp:2:-1,3) = vmr(i,:,sa3_ndx)
               else
                  aersal(plevp:2:-1,3) = 0.
               end if
               if( sa4_ndx > 0 ) then
                  aersal(plevp:2:-1,4) = vmr(i,:,sa4_ndx)
               else
                  aersal(plevp:2:-1,4) = 0.
               end if
               aersal(1,:) = aersal(2,:)
               call photoin( lat, (ip-1)*plonl+i, ncdate, alat, along, &
                             ut, esfact, col_dens(i,1,1), col_dens(i,1,2), albedo, &
                             zarg, tline, tlay, xlwc, xfrc, &
                             airdens, aerocs1, aerocs2, aercbs1, aercbs2, &
                             aersoa, aerant, aerso4, aersal, aerds1, &
                             aerds2, aerds3, aerds4, o3line, rh, &
                             prates, nw, dt_xdiag )
               dt_diag(i,:) = dt_xdiag(:) 
#ifdef DEBUG
               if( do_diag ) then
                  write(*,*) 'xactive_photo: prates at lat,lon = ',lat,i
               end if
#endif
               do m = 1,phtcnt
                  if( lng_indexer(m) > 0 ) then
                     alias_factor = pht_alias_mult(m,2)
                     if( alias_factor == 1. ) then
                        photos(i,:,m) = prates(1:plev,lng_indexer(m))
                     else
                        photos(i,:,m) = alias_factor * prates(1:plev,lng_indexer(m))
                     end if
                  end if
#ifdef DEBUG
                  if( do_diag ) then
                     write(*,'(''xactive_photo: prates('',i2,'',.)'')') m
                     write(*,'(1p,5e21.13)') photos(i,:plev,m)
                     write(*,*) ' '
                  end if
#endif
               end do
!-----------------------------------------------------------------
!	... set jonitr
!-----------------------------------------------------------------
               if( jonitr_ndx > 0 ) then
                  if( jch3cho_a_ndx > 0 ) then
                     photos(i,1:plev,jonitr_ndx) = photos(i,1:plev,jch3cho_a_ndx)
                  end if
                  if( jch3cho_b_ndx > 0 ) then
                     photos(i,1:plev,jonitr_ndx) = photos(i,1:plev,jonitr_ndx) + photos(i,1:plev,jch3cho_b_ndx)
                  end if
                  if( jch3cho_c_ndx > 0 ) then
                     photos(i,1:plev,jonitr_ndx) = photos(i,1:plev,jonitr_ndx) + photos(i,1:plev,jch3cho_c_ndx)
                  end if
               end if
!-----------------------------------------------------------------
!	... calculate j(no) from formula
!-----------------------------------------------------------------
               if( jno_ndx > 0 ) then
                  if( has_o2_col .and. has_o3_col ) then
                     fac1(:) = 1.e-8 * (col_dens(i,:,2)/cos(zen_angle(i)))**.38
                     fac2(:) = 5.e-19 * col_dens(i,:,1) / cos(zen_angle(i))
                     photos(i,:,jno_ndx) = 4.5e-6 * exp( -(fac1(:) + fac2(:)) )
                  end if
               end if
!-----------------------------------------------------------------
! 	... add near IR correction to ho2no2
!-----------------------------------------------------------------
               if( jho2no2_ndx > 0 ) then
                  photos(i,:,jho2no2_ndx) = photos(i,:,jho2no2_ndx) + 1.e-5
               endif
!-----------------------------------------------------------------
! 	... impose zero floor on photorates
!-----------------------------------------------------------------
               photos(i,:,:) = max( 0.,photos(i,:,:) )
            end if
         end if daylight
      end do Long_loop

      deallocate( prates )
        
      end subroutine xactive_photo

      subroutine cloud_mod( zen_angle, clouds, lwc, delp, srf_alb, &
                            eff_alb, cld_mult )
!-----------------------------------------------------------------------
! 	... cloud alteration factors for photorates and albedo
!-----------------------------------------------------------------------

      use mo_grid,      only : plev, plevm
      use mo_constants, only : rgrav

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      real, intent(in)    ::  zen_angle         ! zenith angle
      real, intent(in)    ::  srf_alb           ! surface albedo
      real, intent(in)    ::  clouds(plev)       ! cloud fraction
      real, intent(in)    ::  lwc(plev)          ! liquid water content (mass mr)
      real, intent(in)    ::  delp(plev)         ! del press about midpoint in pascals
      real, intent(out)   ::  eff_alb(plev)      ! effective albedo
      real, intent(out)   ::  cld_mult(plev)     ! photolysis mult factor

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: k
      real    :: coschi
      real    :: del_lwp(plev)
      real    :: del_tau(plev)
      real    :: above_tau(plev)
      real    :: below_tau(plev)
      real    :: above_cld(plev)
      real    :: below_cld(plev)
      real    :: above_tra(plev)
      real    :: below_tra(plev)
      real    :: fac1(plev)
      real    :: fac2(plev)

!---------------------------------------------------------
!	... modify lwc for cloud fraction and form
!	    liquid water path for each layer
!---------------------------------------------------------
      where( clouds(:) /= 0. )
         del_lwp(:) = rgrav * lwc(:) * delp(:) * 1.e3 / clouds(:)
      elsewhere
	 del_lwp(:) = 0.
      endwhere
!---------------------------------------------------------
!    	... form tau for each model layer
!---------------------------------------------------------
      where( clouds(:) /= 0. )
	 del_tau(:) = del_lwp(:) *.155 * clouds(:)**1.5
      elsewhere
	 del_tau(:) = 0.
      end where
!---------------------------------------------------------
!    	... form integrated tau from top down
!---------------------------------------------------------
      above_tau(1) = 0.
      do k = 1,plevm
	 above_tau(k+1) = del_tau(k) + above_tau(k)
      end do
!---------------------------------------------------------
!    	... form integrated tau from bottom up
!---------------------------------------------------------
      below_tau(plev) = 0.
      do k = plevm,1,-1
	 below_tau(k) = del_tau(k+1) + below_tau(k+1)
      end do
!---------------------------------------------------------
!	... form vertically averaged cloud cover above and below
!---------------------------------------------------------
      above_cld(1) = 0.
      do k = 1,plevm
	 above_cld(k+1) = clouds(k) * del_tau(k) + above_cld(k)
      end do
      do k = 2,plev
	 if( above_tau(k) /= 0. ) then
	    above_cld(k) = above_cld(k) / above_tau(k)
	 else
	    above_cld(k) = above_cld(k-1)
	 end if
      end do
      below_cld(plev) = 0.
      do k = plevm,1,-1
	 below_cld(k) = clouds(k+1) * del_tau(k+1) + below_cld(k+1)
      end do
      do k = plevm,1,-1
	 if( below_tau(k) /= 0. ) then
	    below_cld(k) = below_cld(k) / below_tau(k)
	 else
	    below_cld(k) = below_cld(k+1)
	 end if
      end do
!---------------------------------------------------------
!	... modify above_tau and below_tau via jfm
!---------------------------------------------------------
      where( above_cld(2:plev) /= 0. )
	 above_tau(2:plev) = above_tau(2:plev) / above_cld(2:plev)
      end where
      where( below_cld(:plevm) /= 0. )
         below_tau(:plevm) = below_tau(:plevm) / below_cld(:plevm)
      end where
      where( above_tau(2:plev) < 5. )
	    above_cld(2:plev) = 0.
      end where
      where( below_tau(:plevm) < 5. )
	 below_cld(:plevm) = 0.
      end where
!---------------------------------------------------------
!	... form transmission factors
!---------------------------------------------------------
      above_tra(:) = 11.905 / (9.524 + above_tau(:))
      below_tra(:) = 11.905 / (9.524 + below_tau(:))
!---------------------------------------------------------
!	... form effective albedo
!---------------------------------------------------------
      where( below_cld(:) /= 0. )
	 eff_alb(:) = srf_alb + below_cld(:) * (1. - below_tra(:)) &
                                             * (1. - srf_alb)
      elsewhere
	 eff_alb(:) = srf_alb
      end where
      coschi = max( cos( zen_angle ),.5 )
      where( del_lwp(:)*.155 < 5. )
	 fac1(:) = 0.
      elsewhere
	 fac1(:) = 1.4 * coschi - 1.
      end where
      fac2(:)     = min( 0.,1.6*coschi*above_tra(:) - 1. )
      cld_mult(:) = 1. + fac1(:) * clouds(:) + fac2(:) * above_cld(:)
      cld_mult(:) = max( .05,cld_mult(:) )

      end subroutine cloud_mod

      subroutine set_ub_col( col_delta, vmr, invariants, ptop, pdel, &
                             plonl, lat )
!---------------------------------------------------------------
!        ... set the column densities at the upper boundary
!---------------------------------------------------------------

      use chem_mods, only : nfs, ncol_abs, indexm
      use mo_grid,   only : plev, pcnstm1
      use mo_setinv, only : o2_ndx

      implicit none

!---------------------------------------------------------------
!        ... dummy args
!---------------------------------------------------------------
      integer, intent(in) ::  plonl                                    ! long tile dim
      integer, intent(in) ::  lat                                      ! lat tile index
      real, intent(in)    ::  vmr(plonl,plev,pcnstm1)                  ! species concentration (mol/mol)
      real, intent(in)    ::  invariants(plonl,plev,max(1,nfs))        ! invariant concentration (/cm^3)
      real, intent(in)    ::  ptop(plonl)                              ! pressure at upper interface (Pa)
      real, intent(in)    ::  pdel(plonl,plev)                         ! pressure delta about midpoints (Pa)
      real, intent(out)   ::  col_delta(:,0:,:)                        ! column layer density (/cm^2)

!---------------------------------------------------------------
!        ... local variables
!---------------------------------------------------------------
!---------------------------------------------------------------
!        note: xfactor = 10.*r/(k*g) in cgs units.
!              the factor 10. is to convert pdel
!              from pascals to dyne/cm**2.
!---------------------------------------------------------------
      real, parameter :: xfactor = 2.8704e21/(9.80616*1.38044)
      integer :: k, kl, spc_ndx
      integer :: ku(plonl)
      real    :: dp(plonl)
      real    :: tint_vals(2)
      real    :: o2_exo_col(plonl)
      real    :: o3_exo_col(plonl)

!---------------------------------------------------------------
!        ... assign column density at the upper boundary
!            the first column is o3 and the second is o2.
!            add 10 du o3 column above top of model.
!---------------------------------------------------------------
has_abs_cols : &
      if( ncol_abs > 0 ) then
!---------------------------------------------------------------
!	... set exo absorber columns
!---------------------------------------------------------------
         if( has_o2_col .or. has_o3_col ) then
            if( has_fixed_press ) then
               kl = ki - 1
               if( has_o2_col ) then
	          tint_vals(1) = o2_exo_coldens(kl,lat,last) &
                                 + delp * (o2_exo_coldens(ki,lat,last) &
                                           - o2_exo_coldens(kl,lat,last))
	          tint_vals(2) = o2_exo_coldens(kl,lat,next) &
                                 + delp * (o2_exo_coldens(ki,lat,next) &
                                           - o2_exo_coldens(kl,lat,next))
	          o2_exo_col(:) = tint_vals(1) + dels * (tint_vals(2) - tint_vals(1))
               else
                  o2_exo_col(:) = 0.
               end if
               if( has_o3_col ) then
	          tint_vals(1) = o3_exo_coldens(kl,lat,last) &
                                 + delp * (o3_exo_coldens(ki,lat,last) &
                                           - o3_exo_coldens(kl,lat,last))
	          tint_vals(2) = o3_exo_coldens(kl,lat,next) &
                                 + delp * (o3_exo_coldens(ki,lat,next) &
                                           - o3_exo_coldens(kl,lat,next))
	          o3_exo_col(:) = tint_vals(1) + dels * (tint_vals(2) - tint_vals(1))
               else
                  o3_exo_col(:) = 0.
               end if
#ifdef DEBUG
               write(*,*) '-----------------------------------'
               write(*,*) 'set_ub_col: diagnostics @ lat = ',lat
               write(*,*) 'o2_exo_col'
               write(*,'(1p,5g15.7)') o2_exo_col(:)
               write(*,*) 'o3_exo_col'
               write(*,'(1p,5g15.7)') o3_exo_col(:)
               write(*,*) '-----------------------------------'
#endif
            else
!---------------------------------------------------------------
!	... do pressure interpolation
!---------------------------------------------------------------
               call p_interp( lat, plonl, ptop, o2_exo_col, o3_exo_col )
            end if
         end if

         if( o3rad_ndx > 0 ) then
            spc_ndx = o3rad_ndx
         else
            spc_ndx = ox_ndx
         end if
         if( spc_ndx < 1 ) then
	    spc_ndx = o3_ndx
         end if
         if( spc_ndx > 0 ) then
            col_delta(:,0,1) = o3_exo_col(:)
            do k = 1,plev
               col_delta(:,k,1) = xfactor * pdel(:,k) * vmr(:,k,spc_ndx)
            end do
         else if( o3_inv_ndx > 0 ) then
            col_delta(:,0,1) = o3_exo_col(:)
            do k = 1,plev
               col_delta(:,k,1) = xfactor * pdel(:,k) * invariants(:,k,o3_inv_ndx)/invariants(:,k,indexm)
            end do
         else
             col_delta(:,:,1) = 0.
         end if
         if( ncol_abs > 1 ) then
            if( o2_ndx > 1 ) then
               col_delta(:,0,2) = o2_exo_col(:)
               do k = 1,plev
                  col_delta(:,k,2) = xfactor * pdel(:,k) * invariants(:,k,o2_ndx)/invariants(:,k,indexm)
               end do
            else
             col_delta(:,:,2) = 0.
            end if
         end if
      end if has_abs_cols

      end subroutine set_ub_col

      subroutine p_interp( lat, plonl, ptop, o2_exo_col, o3_exo_col )
!---------------------------------------------------------------
!     	... pressure interpolation for exo col density
!---------------------------------------------------------------

      implicit none

!---------------------------------------------------------------
!     	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in)  :: lat
      integer, intent(in)  :: plonl
      real, intent(in)     :: ptop(plonl)                ! top interface pressure (Pa)
      real, intent(out)    :: o2_exo_col(plonl)          ! exo model o2 column density (molecules/cm^2)
      real, intent(out)    :: o3_exo_col(plonl)          ! exo model o3 column density (molecules/cm^2)

!---------------------------------------------------------------
!     	... local variables
!---------------------------------------------------------------
      integer :: i, k, ki, kl
      integer :: ku(plonl)                               ! interpolation index
      real    :: pinterp
      real    :: delp
      real    :: tint_vals(2)
      real    :: dp(plonl)                               ! pressure interpolation factor

long_loop : &
      do i = 1,plonl
         pinterp = ptop(i)
         if( pinterp < levs(1) ) then
            ki   = 0
            delp = 0.
         else
	    do ki = 2,nlev
               if( pinterp <= levs(ki) ) then
		  delp = log( pinterp/levs(ki-1) )/log( levs(ki)/levs(ki-1) )
		  exit
	       end if
	    end do
         end if
         kl = ki - 1
         if( has_o2_col ) then
	    tint_vals(1) = o2_exo_coldens(kl,lat,last) &
                           + delp * (o2_exo_coldens(ki,lat,last) &
                                     - o2_exo_coldens(kl,lat,last))
	    tint_vals(2) = o2_exo_coldens(kl,lat,next) &
                           + delp * (o2_exo_coldens(ki,lat,next) &
                                     - o2_exo_coldens(kl,lat,next))
	    o2_exo_col(i) = tint_vals(1) + dels * (tint_vals(2) - tint_vals(1))
         else
            o2_exo_col(i) = 0.
         end if
         if( has_o3_col ) then
	    tint_vals(1) = o3_exo_coldens(kl,lat,last) &
                           + delp * (o3_exo_coldens(ki,lat,last) &
                                     - o3_exo_coldens(kl,lat,last))
	    tint_vals(2) = o3_exo_coldens(kl,lat,next) &
                           + delp * (o3_exo_coldens(ki,lat,next) &
                                     - o3_exo_coldens(kl,lat,next))
	    o3_exo_col(i) = tint_vals(1) + dels * (tint_vals(2) - tint_vals(1))
         else
            o3_exo_col(i) = 0.
         end if
      end do long_loop

      end subroutine p_interp

      subroutine setcol( col_delta, col_dens, vmr, pdel, plonl )
!---------------------------------------------------------------
!     	... set the column densities
!---------------------------------------------------------------

      use chem_mods, only : ncol_abs
      use mo_grid,   only : plev, pcnstm1

      implicit none

!---------------------------------------------------------------
!     	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in) :: plonl                             ! long tile dim
      real, intent(in)    :: vmr(plonl,plev,pcnstm1)           ! xported species vmr
      real, intent(in)    :: pdel(plonl,plev)                  ! delta about midpoints
      real, intent(in)    :: col_delta(:,0:,:)                 ! layer column densities (molecules/cm^2)
      real, intent(out)   :: col_dens(:,:,:)                   ! column densities ( /cm**2 )

!---------------------------------------------------------------
!        the local variables
!---------------------------------------------------------------
      integer  ::   i, k, km1      ! long, alt indicies

!---------------------------------------------------------------
!        note: xfactor = 10.*r/(k*g) in cgs units.
!              the factor 10. is to convert pdel
!              from pascals to dyne/cm**2.
!---------------------------------------------------------------
      real, parameter :: xfactor = 2.8704e21/(9.80616*1.38044)

!---------------------------------------------------------------
!   	... compute column densities down to the
!           current eta index in the calling routine.
!           the first column is o3 and the second is o2.
!---------------------------------------------------------------
      if( ncol_abs > 0 ) then
         col_dens(:,1,1) = col_delta(:,0,1) + .5 * col_delta(:,1,1)
         do k = 2,plev
	    km1 = k - 1
	    col_dens(:,k,1) = col_dens(:,km1,1) + .5 * (col_delta(:,km1,1) + col_delta(:,k,1))
         end do
         if( ncol_abs > 1 ) then
            col_dens(:,1,2) = col_delta(:,0,2) + .5 * col_delta(:,1,2)
            do k = 2,plev
	       km1 = k - 1
	       col_dens(:,k,2) = col_dens(:,km1,2) + .5 * (col_delta(:,km1,2) + col_delta(:,k,2))
            end do
         end if
      end if

      end subroutine setcol

      subroutine diurnal_geom( ip, lat, time_of_year, polar_night, polar_day, &
                               sunon, sunoff, loc_angle, zen_angle, plonl )
!------------------------------------------------------------------
!    	... diurnal geometry factors
!------------------------------------------------------------------

      use mo_constants, only : pi, twopi, pid2, dayspy, d2r, phi
      use mo_grid,      only : plong => plon
      use mo_mpi,       only : base_lat

      implicit none

!------------------------------------------------------------------
!    	... dummy arguments
!------------------------------------------------------------------
      integer, intent(in)  ::     ip                 ! longitude index
      integer, intent(in)  ::     lat                ! latitude index
      integer, intent(in)  ::     plonl
      real, intent(in)     ::     time_of_year       ! time of year
      real, intent(out)    ::     sunon           ! sunrise angle in radians
      real, intent(out)    ::     sunoff          ! sunset angle in radians
      real, intent(out)    ::     zen_angle(plonl) ! solar zenith angle
      real, intent(out)    ::     loc_angle(plonl) ! "local" time angle
      logical, intent(out) ::     polar_day       ! continuous daylight flag
      logical, intent(out) ::     polar_night     ! continuous night flag

!------------------------------------------------------------------
!        ... local variables
!------------------------------------------------------------------
      integer ::  i
      real    ::  dec_max
      real    ::  declination
      real    ::  latitude
      real    ::  doy_loc            ! day of year
      real    ::  tod                ! time of day
      real    ::  sin_dec, cos_dec   ! sin, cos declination
      real    ::  cosphi             ! cos latitude
      real    ::  sinphi             ! sin latitude

      dec_max     = 23.45 * d2r
      latitude    = phi(base_lat + lat)
      sinphi      = sin( latitude )
      cosphi      = cos( latitude )
      polar_day   = .false.
      polar_night = .false.
!------------------------------------------------------------------
!        note: this formula assumes a 365 day year !
!------------------------------------------------------------------
      doy_loc     = aint( time_of_year )
      declination = dec_max * cos((doy_loc - 172.)*twopi/dayspy)
!------------------------------------------------------------------
!        determine if in polar day or night
!        if not in polar day or night then
!        calculate terminator longitudes
!------------------------------------------------------------------
      if( abs(latitude) >= (pid2 - abs(declination)) ) then
	 if( sign(1.,declination) == sign(1.,latitude) ) then
	    polar_day = .true.
	    sunoff    = 2.*twopi
	    sunon     = -twopi
         else
	    polar_night  = .true.
            zen_angle(:) = -1.0
	    return
         end if
      else
         sunoff = acos( -tan(declination)*tan(latitude) )
         sunon  = twopi - sunoff
      end if

      sin_dec = sin( declination )
      cos_dec = cos( declination )
!------------------------------------------------------------------
!	... compute base for zenith angle
!------------------------------------------------------------------
      tod = (time_of_year - doy_loc) + .5
!-------------------------------------------------------------------
!        note: longitude 0 (greenwich) at 0:00 hrs
!              maps to local angle = pi
!-------------------------------------------------------------------
      loc_angle(:) = (/ ((tod + real(i+(ip-1)*plonl-1)/real(plong))*twopi,i = 1,plonl) /)
      loc_angle(:) = mod( loc_angle(:),twopi )

      if( polar_day ) then
	 zen_angle(:) = acos( sinphi*sin_dec + cosphi*cos_dec*cos(loc_angle(:)) )
      else
	 where( loc_angle(:) <= sunoff .or. loc_angle(:) >= sunon )
	    zen_angle(:) = acos( sinphi*sin_dec + cosphi*cos_dec*cos(loc_angle(:)) )
	 elsewhere
            zen_angle(:) = -1.
	 endwhere
      end if

      end subroutine diurnal_geom

      real function sundis( idate )
!-----------------------------------------------------------------------------
!=  purpose:                                                                 =*
!=  calculate earth-sun distance variation for a given date.  based on       =*
!=  fourier coefficients originally from:  spencer, j.w., 1971, fourier      =*
!=  series representation of the position of the sun, search, 2:172          =*
!-----------------------------------------------------------------------------*
!=  parameters:                                                              =*
!=  idate  - integer, specification of the date, from yymmdd              (i)=*
!=  esrm2  - real, variation of the earth-sun distance                    (o)=*
!=           esrm2 = (average e/s dist)^2 / (e/s dist on day idate)^2        =*
!-----------------------------------------------------------------------------*
!=  edit history:                                                            =*
!=  01/95  changed computation of trig function values                       =*
!-----------------------------------------------------------------------------*
!= this program is free software;  you can redistribute it and/or modify     =*
!= it under the terms of the gnu general public license as published by the  =*
!= free software foundation;  either version 2 of the license, or (at your   =*
!= option) any later version.                                                =*
!= the tuv package is distributed in the hope that it will be useful, but    =*
!= without any warranty;  without even the implied warranty of merchantibi-  =*
!= lity or fitness for a particular purpose.  see the gnu general public     =*
!= license for more details.                                                 =*
!= to obtain a copy of the gnu general public license, write to:             =*
!= free software foundation, inc., 675 mass ave, cambridge, ma 02139, usa.   =*
!-----------------------------------------------------------------------------*
!= to contact the authors, please mail to:                                   =*
!= sasha madronich, ncar/acd, p.o.box 3000, boulder, co, 80307-3000, usa  or =*
!= send email to:  sasha@ucar.edu                                            =*
!-----------------------------------------------------------------------------*
!= copyright (c) 1994,95,96  university corporation for atmospheric research =*
!-----------------------------------------------------------------------------

      use mo_constants, only : pi

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: idate

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: iyear, imonth, iday, mday, month, jday
      integer, save :: imn(12) = (/ 31,28,31,30,31,30,31,31,30,31,30,31 /)
      real    :: dayn, thet0
      real    :: sinth, costh, sin2th, cos2th

!-----------------------------------------------------------------------------
! 	... parse date to find day number (julian day)
!-----------------------------------------------------------------------------
      iyear  = int( idate/10000 )
      imonth = int( (idate - 10000*iyear)/100 )
      iday   = idate - (10000*iyear + 100*imonth)

      if( imonth > 12 ) then
         write(*,*) 'month in date exceeds 12'
         write(*,*) 'date = ', idate
         write(*,*) 'month = ', imonth
         call endrun
      end if

      if( mod(iyear,4) == 0 ) then
         imn(2) = 29
      else
         imn(2) = 28
      end if

      if( iday > imn(imonth) ) then
         write(*,*) 'day in date exceeds days in month'
         write(*,*) 'date = ', idate
         write(*,*) 'day = ', iday
         call endrun()
      end if

      mday = 0
      do month = 1,imonth-1
         mday = mday + imn(month)	  	   
      end do
      jday = mday + iday
      dayn = real(jday - 1) + .5

!-----------------------------------------------------------------------------
! 	... define angular day number and compute esrm2:
!-----------------------------------------------------------------------------
      thet0 = 2.*pi*dayn/365.

!-----------------------------------------------------------------------------
! 	... calculate sin(2*thet0), cos(2*thet0) 
!-----------------------------------------------------------------------------
      sinth   = sin( thet0 )
      costh   = cos( thet0 )
      sin2th  = 2.*sinth*costh
      cos2th  = costh*costh - sinth*sinth
      sundis  = 1.000110 + .034221*costh  +  .001280*sinth + .000719*cos2th +  .000077*sin2th

      end function sundis

      subroutine photo_timestep_init( calday, ncsec )
!-----------------------------------------------------------------------------
!	... setup the time interpolation
!-----------------------------------------------------------------------------

      use mo_control,     only : xactive_prates
      use mo_solar_parms, only : get_solar_parms
      use woods,          only : woods_set_etf
      use neckel,         only : neckel_scale_etf

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) ::  ncsec          ! model simulation time of day (s)
      real, intent(in)    ::  calday         ! day of year at end of present time step

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: m
      real    :: f107
      real    :: f107a

      if( has_o2_col .or. has_o3_col ) then
         if( calday < days(1) ) then
	    next = 1
	    last = 12
	    dels = (365. + calday - days(12)) / (365. + days(1) - days(12))
         else if( calday >= days(12) ) then
	    next = 1
	    last = 12
	    dels = (calday - days(12)) / (365. + days(1) - days(12))
         else
            do m = 11,1,-1
	       if( calday >= days(m) ) then
	          exit
	       end if
            end do
	    last = m
	    next = m + 1
	    dels = (calday - days(m)) / (days(m+1) - days(m))
         end if
#ifdef DEBUG
         write(*,*) '-----------------------------------'
         write(*,*) 'photo_timestep_init: diagnostics'
         write(*,*) 'calday, last, next, dels = ',calday,last,next,dels
         write(*,*) '-----------------------------------'
#endif
      end if

!----------------------------------------------------------------------
!        ... check for etf update
!----------------------------------------------------------------------
      if( .not. xactive_prates ) then
         if( ncsec == 0 ) then
            call get_solar_parms( f107_s = f107, f107a_s = f107a )
            call woods_set_etf( f107, f107a )
            call neckel_scale_etf( f107, f107a )
         end if
      end if

      end subroutine photo_timestep_init

      end module mo_photo
