
      module mo_photoin

      implicit none

      save

      public :: photoin_inti
      public :: photoin
      private

      integer              :: jo2_ndx = 0
      real, allocatable    :: max_o2col(:)
      real, allocatable    :: min_o2col(:)
      logical, allocatable :: z_dep(:)
      character(len=32), allocatable :: pht_tag(:)

      contains

      subroutine photoin_inti( platl, nlng, lng_indexer )
!-------------------------------------------------------------
! 	... assign use masks
!-------------------------------------------------------------

      use mo_params,   only : largest
      use mo_setcld,   only : setcld_inti
      use chem_mods,   only : phtcnt, rxt_tag_lst

      implicit none

!-------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------
      integer, intent(in) :: platl
      integer, intent(in) :: nlng
      integer, intent(in) :: lng_indexer(:)

!-------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------
      integer :: astat
      integer :: m
      integer :: ndx
      character(len=32) :: jname

!-------------------------------------------------------------
! 	... allocate module arrays
!-------------------------------------------------------------
has_photorates : &
      if( nlng > 0 ) then
         allocate( z_dep(nlng), pht_tag(nlng), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photoin_inti: failed to allocate z_dep; error = ',astat
	    call endrun
         end if
         ndx = 0
         do m = 1,phtcnt
            if( lng_indexer(m) > 0 ) then
               if( any( lng_indexer(:m-1) == lng_indexer(m) ) ) then
                  cycle
               end if
               ndx = ndx + 1
               pht_tag(ndx) = trim( rxt_tag_lst(m))
            end if
         end do
         if( ndx /= nlng ) then
            write(*,*) 'photo_inti: corrupted lng_indexer'
            call endrun
         end if
         write(*,*) ' '
         write(*,*) 'photo_inti: lng_indexer name mapping'
         write(*,'(5a)') pht_tag(:)
         write(*,*) ' '
!-------------------------------------------------------------
! 	... search for jo2
!-------------------------------------------------------------
         do m = 1,nlng
            if( pht_tag(m) == 'jo2' ) then
               jo2_ndx = m
               exit
            end if
         end do
         write(*,*) ' '
         write(*,*) 'photo_inti: jo2 index = ',jo2_ndx
         write(*,*) ' '
!-------------------------------------------------------------
! 	... set altitude dependence logical array
!-------------------------------------------------------------
         z_dep(:) = .true.
         do m = 1,nlng
            jname = pht_tag(m)
            select case( jname )
               case( 'jno2', 'jno3', 'jho2', 'jhno2', 'jho2no2' )
                  z_dep(m) = .false.
               case( 'jc2h5cho', 'jchocho', 'jch3ooh' )
                  z_dep(m) = .false.
            end select
         end do

         allocate( max_o2col(platl), min_o2col(platl), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'photoin_inti: failed to allocate max,min o2col; error = ',astat
	    call endrun
         end if

         min_o2col(:) = largest
         max_o2col(:) = 0.
!-------------------------------------------------------------
! 	... intialize cloud layer module
!-------------------------------------------------------------
         call setcld_inti

      end if has_photorates
      
      end subroutine photoin_inti

      subroutine photoin( lat, nlong, idate, alat, along, &
                          ut, esfact, o3top, o2top, albedo, &
                          z, tlev, tlay, xlwc, xfrc, &
                          airlev, aocs1, aocs2, acbs1, acbs2, &
                          asoa, aant, aso4, asal, ads1, &
                          ads2, ads3, ads4, o3, rh, &
                          prate, nw, dt_xdiag )
!----------------------------------------------------------
!     	... interactive photolysis interface routine
!----------------------------------------------------------

      use mo_control,  only : use_dust
      use mo_tuv_inti, only : nlng
      use mo_params,   only : kj, kw
      use mo_wavelen,  only : deltaw, sflx, wc, wl, wu
      use mo_wavelab , only : sj
      use mo_zadj,     only : adj_coeffs
      use mo_zenith,   only : zenith
      use mo_setair,   only : setair
      use mo_setozo,   only : setozo
      use mo_pchem,    only : pchem
      use mo_sphers,   only : sphers
      use mo_airmas,   only : airmas
      use mo_setz,     only : setz
      use mo_seto2,    only : set_o2_xsect
      use mo_rtlink,   only : rtlink
      use mo_setcld,   only : setcld   !, mreg
      use mo_setaer,   only : setaer
      use mo_grid,     only : plev, plevp
      use mo_timer,    only : elapsed
      use mo_mpi,      only : base_lat

      implicit none

!----------------------------------------------------------
!     	... dummy arguments
!----------------------------------------------------------
      integer, intent(in) ::  lat
      integer, intent(in) ::  nlong
      integer, intent(in) ::  idate
      integer, intent(in) ::  nw
      real, intent(in)    ::  alat, along, o3top, o2top
      real, intent(in)    ::  ut, esfact
      real, intent(in)    ::  albedo(kw)
      real, intent(in)    ::  tlay(plev)
      real, intent(in)    ::  xlwc(plevp)           ! cloud water (g/m3)
      real, intent(in)    ::  xfrc(plevp)           ! cloud fraction
      real, intent(in)    ::  tlev(plevp)
      real, intent(in)    ::  airlev(plevp)
      real, intent(in)    ::  z(plevp)
      real, intent(in)    ::  aocs1(plevp)
      real, intent(in)    ::  aocs2(plevp)
      real, intent(in)    ::  acbs1(plevp)
      real, intent(in)    ::  acbs2(plevp)
      real, intent(in)    ::  asoa(plevp)
      real, intent(in)    ::  aant(plevp)
      real, intent(in)    ::  aso4(plevp)
      real, intent(in)    ::  asal(plevp,4)
      real, intent(in)    ::  ads1(plevp)
      real, intent(in)    ::  ads2(plevp)
      real, intent(in)    ::  ads3(plevp)
      real, intent(in)    ::  ads4(plevp)
      real, intent(in)    ::  rh(plevp)
      real, intent(inout) ::  o3(plevp)
      real, intent(out)   ::  prate(plevp,nlng)
      real, intent(out)   ::  dt_xdiag(:)

!----------------------------------------------------------
!     	... local variables
!----------------------------------------------------------
      integer :: i, j, k, km, wn, n, astat
      real    :: factor, delzint
      real    :: wcen
      real, allocatable :: xs(:,:,:)
      real, allocatable :: adjcoe(:,:)    ! ftuv adjustment factor
      character(len=8 ) :: cdate(4)
      character(len=10) :: ctime(4)

!----------------------------------------------------------
! 	... altitude grid
!----------------------------------------------------------
      real    :: colinc(plevp)
      real    :: vcol(plevp)
      real    :: scol(plevp)
      real    :: to3(plevp)

!----------------------------------------------------------
! 	... solar zenith angle
!           slant pathlengths in spherical geometry
!----------------------------------------------------------
      integer :: nid(0:plev)
      real    :: zen
      real    :: dsdh(0:plev,plev)

!----------------------------------------------------------
! 	... extra terrestrial solar flux and earth-sun distance ^-2
!----------------------------------------------------------
      real    :: etf(nw)
      real    :: delw(nw)
      real    :: xsec(nw)

!--------------------------------------------------------------
! 	... atmospheric optical parameters:
!--------------------------------------------------------------
      integer, parameter :: mreg = 16   
      integer :: nreg                  ! regions at each grid
      real    :: dtrl(plev,nw)
      real    :: dto3(plev,nw)
      real    :: dto2(plev,nw)
      real    :: dtcld(plev,nw)
      real    :: omcld(plev,nw)
      real    :: gcld(plev,nw)

      real    :: dtcbs1(plev,nw)
      real    :: dtcbs2(plev,nw)
      real    :: omcbs1(plev,nw)
      real    :: omcbs2(plev,nw)
      real    :: gcbs1(plev,nw)
      real    :: gcbs2(plev,nw)

      real    :: dtocs1(plev,nw)
      real    :: dtocs2(plev,nw)
      real    :: omocs1(plev,nw)
      real    :: omocs2(plev,nw)
      real    :: gocs1(plev,nw)
      real    :: gocs2(plev,nw)

      real    :: dtant(plev,nw)
      real    :: omant(plev,nw)
      real    :: gant(plev,nw)

      real    :: dtso4(plev,nw)
      real    :: omso4(plev,nw)
      real    :: gso4(plev,nw)

      real    :: dtsal(plev,nw,4)
      real    :: omsal(plev,nw,4)
      real    :: gsal(plev,nw,4)

      real    :: dtds1(plev,nw)
      real    :: dtds2(plev,nw)
      real    :: dtds3(plev,nw)
      real    :: dtds4(plev,nw)
      real    :: omds1(plev,nw)
      real    :: omds2(plev,nw)
      real    :: omds3(plev,nw)
      real    :: omds4(plev,nw)
      real    :: gds1(plev,nw)
      real    :: gds2(plev,nw)
      real    :: gds3(plev,nw)
      real    :: gds4(plev,nw)

      real    :: optr(plev,mreg)         ! cld opt (z dependent) at each region
      real    :: fp(mreg)                ! probability at each region

!--------------------------------------------------------------
! 	... spectral irradiance and actinic flux (scalar irradiance):
!--------------------------------------------------------------
      real    :: radfld(plevp,nw)
      real    :: radxx(plevp,nw)

!-------------------------------------------------------------
!  	... j-values:
!-------------------------------------------------------------
      integer :: jn, m

!-------------------------------------------------------------
! 	... location and time
!-------------------------------------------------------------
      integer :: iyear, imonth, iday
      real    :: dtime, ut0

      call date_and_time( cdate(1), ctime(1) )
!-------------------------------------------------------------
! 	... allocate wrking xsection array
!-------------------------------------------------------------
      allocate( xs(nw,plevp,nlng), adjcoe(plevp,nlng), stat=astat )
      if( astat /= 0 ) then
	 write(*,*) 'photoin: failed to allocate xs, adjcoe; error = ',astat
	 call endrun
      end if
!-------------------------------------------------------------
! 	... solar zenith angle calculation:
!-------------------------------------------------------------
#ifdef DEBUG
      if( do_diag ) then
      write(*,*) 'photoin : called zenith with alat,along,idate,ut :'
      write(*,'(5(1x,z16))') alat,along,idate,ut
      end if
#endif
      call zenith( alat, along, idate, ut, zen )
#ifdef DEBUG
      if( do_diag ) then
      write(*,*) 'photoin: zenith angle'
      write(*,'(1x,z16)') zen
      end if
#endif

      etf(1:nw) = sflx(1:nw) * esfact   ! earth-sun distance effect
!-------------------------------------------------------
!  	... air profile and rayleigh optical depths (inter-face)
!-------------------------------------------------------
      call setair( z, nw, wc, airlev, dtrl, colinc, o2top )

!-------------------------------------------------------------
! 	... ozone optical depths (must give temperature) (inter-face)
!-------------------------------------------------------------
      call setozo( z, nw, wl, tlay, dto3, to3, o3, airlev, o3top )

!-------------------------------------------------------------
! 	... cloud optical depths
!-------------------------------------------------------------
      call setcld( z, xlwc, xfrc, nreg, fp, optr )

!-------------------------------------------------------------
! 	... aerosol optical depths
!-------------------------------------------------------------
      call setaer( z, airlev, rh, aocs1, aocs2, &
                   acbs1, acbs2,&
                   aant, aso4, asal, ads1, ads2, &
                   ads3, ads4, asoa, &
                   dtcbs1, dtcbs2, omcbs1, omcbs2, gcbs1, gcbs2, &
                   dtocs1, dtocs2, omocs1, omocs2, gocs1, gocs2, &
                   dtant, omant, gant, &
                   dtso4, omso4, gso4, &
                   dtsal, omsal, gsal, &
                   dtds1, dtds2, dtds3, dtds4, &
                   omds1, omds2, omds3, omds4, &
                   gds1, gds2, gds3, gds4, nw )
      dt_xdiag(1) = sum( dtcbs1(:,16) + dtcbs2(:,16) )
      dt_xdiag(2) = sum( dtocs1(:,16) + dtocs2(:,16) )
      dt_xdiag(3) = sum( dtso4(:,16) )
      dt_xdiag(4) = sum( dtant(:,16) )
      dt_xdiag(5) = sum( dtsal(:,16,1) + dtsal(:,16,2) + dtsal(:,16,3) + dtsal(:,16,4) )
      dt_xdiag(6) = sum( dtds1(:,16) + dtds2(:,16) + dtds3(:,16) + dtds4(:,16) )
      dt_xdiag(7) = sum( dt_xdiag(1:6) )
#ifdef DEBUG
      if( do_diag ) then
      write(*,*) 'photoin: nreg = ',nreg
      end if
#endif

!------------------------------------------------------------
! 	... photo-chemical and photo-biological weigting functions. 
!           for pchem, need to know temperature and pressure profiles.
!           output:
!           from pchem:  sj(kj,kz,kw) - for each reaction
!-------------------------------------------------------------
      xs(:,:,1:nlng) = sj(:,:,1:nlng)
      call pchem( nw, wl, wc, wu, tlev, &
                  airlev, nlng, pht_tag, xs )

!-------------------------------------------------------------
! 	... slant path lengths for spherical geometry
!-------------------------------------------------------------
       call sphers( z, zen, dsdh, nid )
       call airmas( z, zen, dsdh, nid, colinc, vcol, scol )

#ifdef DEBUG
       if( do_diag ) then
       write(*,*) '----------------------------------------'
       write(*,*) 'photoin: diagnostics'
       write(*,*) 'photoin: colinc'
       write(*,'(5(1x,z16))') colinc(:)
       write(*,*) 'photoin: to3'
       write(*,'(5(1x,z16))') to3(:)
       write(*,*) 'photoin: scol'
       write(*,'(5(1x,z16))') scol(:)
       write(*,*) '----------------------------------------'
       end if
#endif

!---------------------------------------------------------------
!    	... modification of coefficent of j-vales function of to3 and zenith
!---------------------------------------------------------------
      call setz( to3, tlev, adj_coeffs, zen, adjcoe, pht_tag )

!------------------------------------------------------------------
! 	... effective o2 optical depth (sr bands, must know zenith angle!)
!           assign o2 cross section to sj(1,*,*)
!------------------------------------------------------------------
       if( jo2_ndx > 0 ) then
          min_o2col(lat) = min( min_o2col(lat),.2905*minval( scol(1:plevp),mask=scol(1:plevp) /= 0. ) )
          max_o2col(lat) = max( max_o2col(lat),.2905*maxval( scol(1:plevp) ) )
          call set_o2_xsect( lat, z, nw, wl, colinc, &
			     vcol, scol, dto2, xs(1,1,jo2_ndx) )
       end if

      delw(:nw) = deltaw(:nw) * etf(:nw)

!---------------------------------------------------
!  	... monochromatic radiative transfer:
!           outputs are  fdir, fdn, fup
!---------------------------------------------------
      call date_and_time( cdate(2), ctime(2) )

! set for cloud only

      do wn = 1,nw
         radfld(:,wn) = 0. 
         omcld(:,wn)  = .9999
         gcld (:,wn)  = .85
      end do

Cld_reg_loop : &
      do n = 1,nreg
	 factor = fp(n)
         do wn = 1,nw
            dtcld(:,wn) = optr(:,n)
         end do
#ifdef DEBUG
       if( do_diag ) then
       write(*,*) '----------------------------------------'
       write(*,*) 'photoin: diagnostics'
       write(*,*) 'photoin: z'
       write(*,'(5(1x,z16))') z(:)
       write(*,*) 'photoin: nid'
       write(*,'(5(1x,z16))') nid(:)
       write(*,*) 'photoin: dtrl'
       write(*,'(5(1x,z16))') dtrl(3,:)
       write(*,*) 'photoin: dto3'
       write(*,'(5(1x,z16))') dto3(3,:)
       write(*,*) 'photoin: dto2'
       write(*,'(5(1x,z16))') dto2(3,:)
       write(*,*) 'photoin: dtcld'
       write(*,'(5(1x,z16))') dtcld(3,:)
       write(*,*) '----------------------------------------'
       end if
#endif

#ifdef NO_AEROSOL
       dtcbs1(:,:)  = 0.
       dtcbs2(:,:)  = 0.
       dtocs1(:,:)  = 0.
       dtocs2(:,:)  = 0.
       dtant(:,:)   = 0.
       dtso4(:,:)   = 0.
       dtsal(:,:,:) = 0.
       dtds1(:,:)   = 0.
       dtds2(:,:)   = 0.
       dtds3(:,:)   = 0.
       dtds4(:,:)   = 0.

       omcbs1(:,:)  = 0.
       omcbs2(:,:)  = 0.
       omocs1(:,:)  = 0.
       omocs2(:,:)  = 0.
       omant(:,:)   = 0.
       omso4(:,:)   = 0.
       omsal(:,:,:) = 0.
       omds1(:,:)   = 0.
       omds2(:,:)   = 0.
       omds3(:,:)   = 0.
       omds4(:,:)   = 0.

       gcbs1(:,:)  = 0.
       gcbs2(:,:)  = 0.
       gocs1(:,:)  = 0.
       gocs2(:,:)  = 0.
       gant(:,:)   = 0.
       gso4(:,:)   = 0.
       gsal(:,:,:) = 0.
       gds1(:,:)   = 0.
       gds2(:,:)   = 0.
       gds3(:,:)   = 0.
       gds4(:,:)   = 0.
#endif
         call rtlink( z, nw, albedo, zen, dsdh, &
                      nid, dtrl, dto3, dto2, & 
                      dtcld, omcld, gcld, &
                      dtcbs1, omcbs1, gcbs1, &
                      dtcbs2, omcbs2, gcbs2, &
                      dtocs1, omocs1, gocs1, &
                      dtocs2, omocs2, gocs2, &
                      dtant, omant, gant, &
                      dtso4, omso4, gso4, &
                      dtsal, omsal, gsal, &
                      dtds1, omds1, gds1, &
                      dtds2, omds2, gds2, &
                      dtds3, omds3, gds3, &
                      dtds4, omds4, gds4, radxx )
         do wn = 1,nw
            radfld(:,wn) = radfld(:,wn) + radxx(:,wn)*factor
         end do
      end do Cld_reg_loop

!----------------------------------------------------------
!     	... interplation at the top level
!----------------------------------------------------------
      delzint = (z(plev-1) - z(plev-2))/(z(plev) - z(plev-1))
      do wn = 1,nw
         radfld(1,wn) = radfld(2,wn) + (radfld(2,wn) - radfld(3,wn))*delzint
         radfld(1,wn) = max( radfld(1,wn),radfld(2,wn) )
      end do

      call date_and_time( cdate(3), ctime(3) )
!     elapsed(29) = elapsed(29) + time_diff( ctime(2),ctime(3) )
!----------------------------------------------------------
!   	... j-val calculation
!           spherical irradiance (actinic flux)
!           as a function of altitude
!           convert to quanta s-1 nm-1 cm-2
!           (1.e-4 * (wc*1e-9) / (hc = 6.62e-34 * 2.998e8))
!----------------------------------------------------------
      call date_and_time( cdate(2), ctime(2) )
!     elapsed(27) = elapsed(27) + time_diff( ctime(1),ctime(2) )
      call date_and_time( cdate(1), ctime(1) )
rate_loop : &
      do m = 1,nlng
         if( .not. z_dep(m) ) then
            xsec(:nw)     = xs(:nw,1,m) * delw(:nw)
            prate(:plevp,m) = matmul( radfld, xsec )
         else
            do k = 1,plevp
	       km = plevp - k + 1
               xsec(:nw) = xs(:nw,km,m) * delw(:nw)
               prate(k,m) = dot_product( radfld(k,:nw), xsec(:nw) )
#ifdef DEBUG
       if( do_diag .and. m == 2 ) then
       write(*,*) '----------------------------------------'
       write(*,*) 'photoin: diagnostics at k = ',k
       write(*,*) 'photoin: delw'
       write(*,'(5(1x,z16))') delw(:nw)
       write(*,*) 'photoin: sj'
       write(*,'(5(1x,z16))') sj(:nw,km,m)
       write(*,*) 'photoin: xsec'
       write(*,'(5(1x,z16))') xsec(:nw)
       write(*,*) 'photoin: radfld'
       write(*,'(5(1x,z16))') radfld(k,:nw)
       write(*,*) 'photoin: prate'
       write(*,'(5(1x,z16))') prate(k,m)
       write(*,*) 'photoin: adjcoe'
       write(*,'(5(1x,z16))') adjcoe(km,m)
       write(*,*) '----------------------------------------'
       end if
#endif
            end do
         end if
         prate(1:plevp,m) = prate(1:plevp,m) * adjcoe(plevp:1:-1,m)  
      end do rate_loop

      call date_and_time( cdate(2), ctime(2) )
!     elapsed(28) = elapsed(28) + time_diff( ctime(1),ctime(2) )

      deallocate( xs, adjcoe )

      end subroutine photoin

      end module mo_photoin
