
      module mo_ps2str

      private
      public :: ps2str

      contains

      subroutine ps2str( nw, zen, rsfc, tauu, omu, &
                         gu, dsdh, nid, radfld )
!-----------------------------------------------------------------------------
!   purpose:
!   solve two-stream equations for multiple layers.  the subroutine is based
!   on equations from:  toon et al., j.geophys.res., v94 (d13), nov 20, 1989.
!   it contains 9 two-stream methods to choose from.  a pseudo-spherical
!   correction has also been added.
!-----------------------------------------------------------------------------
!   parameters:
!   nlevel  - integer, number of specified altitude levels in the working (i)
!             grid
!   zen     - real, solar zenith angle (degrees)                          (i)
!   rsfc    - real, surface albedo at current wavelength                  (i)
!   tauu    - real, unscaled optical depth of each layer                  (i)
!   omu     - real, unscaled single scattering albedo of each layer       (i)
!   gu      - real, unscaled asymmetry parameter of each layer            (i)
!   dsdh    - real, slant path of direct beam through each layer crossed  (i)
!             when travelling from the top of the atmosphere to layer i;
!             dsdh(i,j), i = 0..nz-1, j = 1..nz-1
!   nid     - integer, number of layers crossed by the direct beam when   (i)
!             travelling from the top of the atmosphere to layer i;
!             nid(i), i = 0..nz-1
!   delta   - logical, switch to use delta-scaling                        (i)
!             .true. -> apply delta-scaling
!             .false.-> do not apply delta-scaling
!   fdr     - real, contribution of the direct component to the total     (o)
!             actinic flux at each altitude level
!   fup     - real, contribution of the diffuse upwelling component to    (o)
!             the total actinic flux at each altitude level
!   fdn     - real, contribution of the diffuse downwelling component to  (o)
!             the total actinic flux at each altitude level
!   edr     - real, contribution of the direct component to the total     (o)
!             spectral irradiance at each altitude level
!   eup     - real, contribution of the diffuse upwelling component to    (o)
!             the total spectral irradiance at each altitude level
!   edn     - real, contribution of the diffuse downwelling component to  (o)
!             the total spectral irradiance at each altitude level
!-----------------------------------------------------------------------------

      use mo_params,    only : smallest, largest
      use mo_constants, only : d2r
      use mo_grid,      only : plev, plevp, plevm
      use mo_trislv,    only : tridec, trislv

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: nw
      integer, intent(in) :: nid(0:plev)
      real, intent(in)    :: zen
      real, intent(in)    :: rsfc(nw)
      real, dimension(plev,nw), intent(in) :: tauu, omu, gu
      real, intent(in)    :: dsdh(0:plev,plev)
      real, intent(out)   :: radfld(plevp,nw)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
!-----------------------------------------------------------------------------
! 	... mu = cosine of solar zenith angle
!           rsfc = surface albedo
!           tauu =  unscaled optical depth of each layer
!           omu  =  unscaled single scattering albedo
!           gu   =  unscaled asymmetry factor
!           klev = max dimension of number of layers in atmosphere
!           nlayer = number of layers in the atmosphere
!           nlevel = nlayer + 1 = number of levels
!-----------------------------------------------------------------------------
      integer, parameter :: mrows = 2*plev
      real, parameter    :: eps = 1.e-3
      real, parameter    :: pifs = 1.      
      real, parameter    :: fdn0 = 0.

      integer :: row
      integer :: lev
      integer :: i, ip1, iw
      integer :: j, jl, ju

      real :: precis, wrk
      real :: tempg
      real :: mu, suma
      real :: g, om
      real :: gam1, gam2, gam3, gam4
      real, dimension(plev) :: f, gi, omi
      real, dimension(0:plevp) :: tauc, mu2
      real, dimension(plev) :: lam, taun, bgam
      real, dimension(plev) :: cdn
      real, dimension(0:plevp,nw) :: tausla
      real, dimension(plev,nw) :: cup, cuptn, cdntn
      real, dimension(plev,nw) :: e1, e2, e3, e4
      real, dimension(mrows) :: a, b, d, e
      real, dimension(nw,mrows) :: sub, main, super, y
!-----------------------------------------------------------------------------
! 	... for calculations of associated legendre polynomials for gama1,2,3,4
!           in delta-function, modified quadrature, hemispheric constant,
!           hybrid modified eddington-delta function metods, p633,table1.
!           w.e.meador and w.r.weaver, gas,1980,v37,p.630
!           w.j.wiscombe and g.w. grams, gas,1976,v33,p2440, 
!           uncomment the following two lines and the appropriate statements
!           further down.
!-----------------------------------------------------------------------------
      real :: expon, expon0, expon1, divisr, temp, up, dn
      real :: ssfc

!-----------------------------------------------------------------------------
! 	... initial conditions:  pi*solar flux = 1;  diffuse incidence = 0
!-----------------------------------------------------------------------------
      precis = epsilon( precis )

      mu = cos( zen*d2r )
wave_loop : &
      do iw = 1,nw
!-----------------------------------------------------------------------------
!	... compute coefficients for each layer:
!           gam1 - gam4 = 2-stream coefficients, different for different approximations
!           expon0 = calculation of e when tau is zero
!           expon1 = calculation of e when tau is taun
!           cup and cdn = calculation when tau is zero
!           cuptn and cdntn = calc. when tau is taun
!           divisr = prevents division by zero
!-----------------------------------------------------------------------------
         tauc(0:plevp)      = 0.
         tausla(0:plevp,iw) = 0.
         mu2(0:plevp)       = sqrt( smallest )

!-----------------------------------------------------------------------------
! 	... delta-scaling. have to be done for delta-eddington approximation, 
!           delta discrete ordinate, practical improved flux method, delta function,
!           and hybrid modified eddington-delta function methods approximations
!-----------------------------------------------------------------------------
         f(1:plev)    = gu(:,iw)*gu(:,iw)
         gi(1:plev)   = (gu(:,iw) - f(1:plev))/(1. - f(1:plev))
         omi(1:plev)  = (1. - f(1:plev))*omu(1:plev,iw)/(1. - omu(1:plev,iw)*f(1:plev))       
         taun(1:plev) = (1. - omu(1:plev,iw)*f(1:plev))*tauu(1:plev,iw)

!-----------------------------------------------------------------------------
! 	... calculate slant optical depth at the top of the atmosphere when zen>90.
!           in this case, higher altitude of the top layer is recommended which can 
!           be easily changed in gridz.f.
!-----------------------------------------------------------------------------
         if( zen > 90. ) then
            if( nid(0) < 0 ) then
               tausla(0,iw) = largest
            else
	       ju = nid(0)
               tausla(0,iw) = 2.*dot_product( taun(1:ju),dsdh(0,1:ju) )
            end if
         end if
level_loop : &  
         do i = 1,plev
            g = gi(i)
            om = omi(i)
            tauc(i) = tauc(i-1) + taun(i)
!-----------------------------------------------------------------------------
! 	... stay away from 1 by precision.  for g, also stay away from -1
!-----------------------------------------------------------------------------
            tempg = min( abs(g),1. - precis )
            g     = sign( tempg,g )
            om    = min( om,1.-precis )
!-----------------------------------------------------------------------------
! 	... calculate slant optical depth
!-----------------------------------------------------------------------------
            if( nid(i) < 0 ) then
               tausla(i,iw) = largest
            else
               ju = min( nid(i),i )
               suma = dot_product( taun(1:ju),dsdh(i,1:ju) )
               jl = min( nid(i),i ) + 1
               tausla(i,iw) = suma + 2.*dot_product( taun(jl:nid(i)),dsdh(i,jl:nid(i)) )
               if( tausla(i,iw) == tausla(i-1,iw) ) then
                 mu2(i) = sqrt( largest )
               else
                 mu2(i) = (tauc(i) - tauc(i-1))/(tausla(i,iw) - tausla(i-1,iw))
                 mu2(i) = sign( max( abs(mu2(i)),sqrt(smallest) ),mu2(i) )
               end if
            end if
!-----------------------------------------------------------------------------
!	... the following gamma equations are from pg 16,289, table 1
!           eddington approximation(joseph et al., 1976, jas, 33, 2452):
!-----------------------------------------------------------------------------
            gam1 =  .25*(7. - om*(4. + 3.*g))
            gam2 = -.25*(1. - om*(4. - 3.*g))
            gam3 = .25*(2. - 3.*g*mu)
            gam4 = 1. - gam3
!-----------------------------------------------------------------------------
! 	... lambda = pg 16,290 equation 21
!           big gamma = pg 16,290 equation 22
!-----------------------------------------------------------------------------
            lam(i) = sqrt( gam1*gam1 - gam2*gam2 )
            bgam(i) = (gam1 - lam(i))/gam2
	    wrk = lam(i)*taun(i)
	    if( wrk < 500. ) then
               expon = exp( -wrk )
	    else
               expon = 0.
	    end if
!-----------------------------------------------------------------------------
! 	... e1 - e4 = pg 16,292 equation 44
!-----------------------------------------------------------------------------
            e1(i,iw) = 1. + bgam(i)*expon
            e2(i,iw) = 1. - bgam(i)*expon
            e3(i,iw) = bgam(i) + expon
            e4(i,iw) = bgam(i) - expon
!-----------------------------------------------------------------------------
! 	... the following sets up for the c equations 23, and 24
!           found on page 16,290
!           prevent division by zero (if lambda=1/mu, shift 1/mu^2 by eps = 1.e-3
!           which is approx equiv to shifting mu by 0.5*eps* (mu)**3
!-----------------------------------------------------------------------------
	    if( tausla(i-1,iw) < 500. ) then
               expon0 = exp( -tausla(i-1,iw) )
	    else
               expon0 = 0.
	    end if
	    if( tausla(i,iw) < 500. ) then
               expon1 = exp( -tausla(i,iw) )
	    else
               expon1 = 0.
	    end if
            divisr = lam(i)*lam(i) - 1./(mu2(i)*mu2(i))
            temp = max( eps,abs(divisr) )
            divisr = 1./sign( temp,divisr )
            up = om*pifs*((gam1 - 1./mu2(i))*gam3 + gam4*gam2)*divisr
            dn = om*pifs*((gam1 + 1./mu2(i))*gam4 + gam2*gam3)*divisr
!-----------------------------------------------------------------------------
! 	... cup and cdn are when tau is equal to zero
!           cuptn and cdntn are when tau is equal to taun
!-----------------------------------------------------------------------------
            cup(i,iw) = up*expon0
            cdn(i)    = dn*expon0
            cuptn(i,iw) = up*expon1
            cdntn(i,iw) = dn*expon1
         end do level_loop

!-----------------------------------------------------------------------------
!	... set up matrix
!           ssfc = pg 16,292 equation 37  where pi fs is one (unity).
!-----------------------------------------------------------------------------
	if( tausla(plev,iw) < 500. ) then
           ssfc = rsfc(iw)*mu*exp( -tausla(plev,iw) )*pifs
	else
           ssfc = 0.
	end if

!-----------------------------------------------------------------------------
! 	... the following are from pg 16,292  equations 39 - 43.
!           set up first row of matrix:
!-----------------------------------------------------------------------------
        a(1) = 0.
        b(1) = e1(1,iw)
        d(1) = -e2(1,iw)
        e(1) = fdn0 - cdn(1)

!-----------------------------------------------------------------------------
! 	... set up odd rows 3 thru (mrows - 1):
!-----------------------------------------------------------------------------
        a(3:mrows-1:2) = e2(1:plevm,iw)*e3(1:plevm,iw) - e4(1:plevm,iw)*e1(1:plevm,iw)
        b(3:mrows-1:2) = e1(1:plevm,iw)*e1(2:plev,iw) - e3(1:plevm,iw)*e3(2:plev,iw)
        d(3:mrows-1:2) = e3(1:plevm,iw)*e4(2:plev,iw) - e1(1:plevm,iw)*e2(2:plev,iw)
        e(3:mrows-1:2) = e3(1:plevm,iw)*(cup(2:plev,iw) - cuptn(1:plevm,iw)) + e1(1:plevm,iw)*(cdntn(1:plevm,iw) - cdn(2:plev))

!-----------------------------------------------------------------------------
! 	... set up even rows 2 thru (mrows - 2): 
!-----------------------------------------------------------------------------
        a(2:mrows-2:2) = e2(2:plev,iw)*e1(1:plevm,iw) - e3(1:plevm,iw)*e4(2:plev,iw)
        b(2:mrows-2:2) = e2(1:plevm,iw)*e2(2:plev,iw) - e4(1:plevm,iw)*e4(2:plev,iw)
        d(2:mrows-2:2) = e1(2:plev,iw)*e4(2:plev,iw) - e2(2:plev,iw)*e3(2:plev,iw)
        e(2:mrows-2:2) = (cup(2:plev,iw) - cuptn(1:plevm,iw))*e2(2:plev,iw) - (cdn(2:plev) - cdntn(1:plevm,iw))*e4(2:plev,iw)

!-----------------------------------------------------------------------------
! 	... set up last row of matrix at mrows:
!-----------------------------------------------------------------------------
        a(mrows) = e1(plev,iw) - rsfc(iw)*e3(plev,iw)
        b(mrows) = e2(plev,iw) - rsfc(iw)*e4(plev,iw)
        d(mrows) = 0.
        e(mrows) = ssfc - cuptn(plev,iw) + rsfc(iw)*cdntn(plev,iw)

	sub(iw,1:mrows)   = a(1:mrows)
	main(iw,1:mrows)  = b(1:mrows)
	super(iw,1:mrows) = d(1:mrows)
	y(iw,1:mrows)     = e(1:mrows)
      end do wave_loop

!-----------------------------------------------------------------------------
! 	... solve the system
!-----------------------------------------------------------------------------
      call tridec( nw, mrows, sub, main, super )
      call trislv( nw, mrows, sub, main, super, y )

!-----------------------------------------------------------------------------
!	... unfold solution of matrix, compute output fluxes
!-----------------------------------------------------------------------------
      do iw = 1,nw
!-----------------------------------------------------------------------------
! 	... the following equations are from pg 16,291  equations 31 & 32
!-----------------------------------------------------------------------------
	 e(:mrows) = y(iw,:mrows)
	 if( tausla(0,iw) < 500. ) then
            radfld(1,iw) = 2.*(fdn0 +  e(1)*e3(1,iw) - e(2)*e4(1,iw) + cup(1,iw)) + exp( -tausla(0,iw) )
	 else
            radfld(1,iw) = 2.*(fdn0 +  e(1)*e3(1,iw) - e(2)*e4(1,iw) + cup(1,iw))
	 end if
	 where( tausla(1:plev,iw) < 500. )
	    cdn(1:plev) = exp( -tausla(1:plev,iw) )
	 elsewhere
	    cdn(1:plev) = 0.
	 endwhere
         radfld(2:plevp,iw) = 2.*(e(1:mrows-1:2)*(e3(1:plev,iw) + e1(1:plev,iw)) &
                            + e(2:mrows:2)*(e4(1:plev,iw) + e2(1:plev,iw))       &
                            + cdntn(1:plev,iw) + cuptn(1:plev,iw)) + cdn(1:plev)
      end do

      end subroutine ps2str

      end module mo_ps2str
