
      module mo_setair

      private
      public :: setair

      contains

      subroutine setair( z, nw, wc, airlev, dtrl, &
                         cz, o2top )
!-----------------------------------------------------------------------------
!   purpose:
!   set up an altitude profile of air molecules.  subroutine includes a
!   shape-conserving scaling method that allows scaling of the entire
!   profile to a given sea-level pressure.
!-----------------------------------------------------------------------------
!   parameters:
!   pmbnew  - real, sea-level pressure (mb) to which profile should be
!             scaled.  if pmbnew < 0, no scaling is done
!   nz      - integer, number of specified altitude levels in the working (i)
!             grid
!   z       - real, specified altitude working grid (km)                  (i)
!   nw      - integer, number of specified intervals + 1 in working       (i)
!             wavelength grid
!   wl      - real, vector of lower limits of wavelength intervals in     (i)
!             working wavelength grid
!   airlev  - real, air density (molec/cc) at each specified altitude     (o)
!   dtrl    - real, rayleigh optical depth at each specified altitude     (o)
!             and each specified wavelength
!   cz      - real, number of air molecules per cm^2 at each specified    (o)
!             altitude layer
!-----------------------------------------------------------------------------

      use mo_params, only : kw
      use mo_grid,   only : plev, plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: nw
      real, intent(in)    :: o2top
      real, intent(in)    :: wc(kw)
      real, intent(in)    :: z(plevp)
      real, intent(in)    :: airlev(plevp)
!-----------------------------------------------------------------------------
! 	... air density (molec cm-3) at each grid level
!           rayleigh optical depths
!-----------------------------------------------------------------------------
      real, intent(out) :: dtrl(plev,nw)
      real, intent(out) :: cz(plevp)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: i, ip1, iw
      real    :: hscale
      real    :: srayl
      real    :: deltaz
      real    :: wmicrn, xx 

!-----------------------------------------------------
! 	... compute column increments (logarithmic integrals)
!-----------------------------------------------------
      do i = 1,plev
         ip1 = i + 1
         deltaz = 1.e5 * (z(ip1) - z(i)) 
         cz(i)  =  (airlev(ip1) - airlev(i)) / log(airlev(ip1)/airlev(i)) * deltaz
      end do

!-----------------------------------------------------
! 	... include exponential tail integral from infinity to 50 km,
!           fold tail integral into top layer
!           specify scale height near top of data.  (scale height at 40km????)
!-----------------------------------------------------
!     hscale = 8.05e5
      cz(plevp) = o2top/.2095

!-----------------------------------------------------
! 	... compute rayleigh cross sections and depths:
!-----------------------------------------------------
      do iw = 1,nw
!-----------------------------------------------------
! 	... rayleigh scattering cross section from wmo 1985 (originally from
!           nicolet, m., on the molecular scattering in the terrestrial atmosphere:
!           an empirical formula for its calculation in the homoshpere, planet.
!           space sci., 32, 1467-1468, 1984.
!-----------------------------------------------------
         wmicrn =  1.e-3*wc(iw)
         if( wmicrn <= .55 ) then
            xx = 3.6772 + 0.389*wmicrn + 0.09426/wmicrn
         else
            xx = 4.04
         end if
         srayl = 4.02e-28/(wmicrn)**xx
         dtrl(1:plev-1,iw) = cz(1:plev-1)*srayl
         dtrl(plev,iw)     = (cz(plev) + cz(plevp))*srayl
      end do

      end subroutine setair

      end module mo_setair
