
      module mo_sethet

! corrected NH4 washout (LKE, 5/1/07)

      private
      public :: sethet_inti, sethet

      save

      integer :: h2o2_ndx, hno3_ndx, ch2o_ndx, ch3ooh_ndx, ch3coooh_ndx, &
		 ho2no2_ndx, ch3cocho_ndx, xooh_ndx, onitr_ndx, glyald_ndx, &
		 ch3cho_ndx, mvk_ndx, macr_ndx, pooh_ndx, c2h5ooh_ndx, &
		 c3h7ooh_ndx, rooh_ndx, isopno3_ndx, onit_ndx, Pb_ndx, &
		 macrooh_ndx, isopooh_ndx, ch3oh_ndx, c2h5oh_ndx, hyac_ndx, hydrald_ndx
      integer :: spc_h2o2_ndx, spc_hno3_ndx
      integer :: alkooh_ndx, mekooh_ndx, tolooh_ndx, terpooh_ndx, ch3cooh_ndx
      integer :: so2_ndx, soa_ndx, so4_ndx, nh3_ndx, nh4_ndx, nh4no3_ndx, &
                 cb2_ndx, oc2_ndx, sa1_ndx, sa2_ndx, sa3_ndx, sa4_ndx


      contains

      subroutine sethet_inti
!-----------------------------------------------------------------------      
!       ... intialize the wet removal rate constants routine
!-----------------------------------------------------------------------      

      use mo_chem_utls, only : get_het_ndx, get_spc_ndx

      spc_h2o2_ndx = get_spc_ndx( 'H2O2' )
      spc_hno3_ndx = get_spc_ndx( 'HNO3' )

      h2o2_ndx   = get_het_ndx( 'H2O2' )
      hno3_ndx   = get_het_ndx( 'HNO3' )
      ch2o_ndx   = get_het_ndx( 'CH2O' )
      ch3ooh_ndx = get_het_ndx( 'CH3OOH' )
      ch3coooh_ndx = get_het_ndx( 'CH3COOOH' )
      ho2no2_ndx  = get_het_ndx( 'HO2NO2' )
      ch3cocho_ndx = get_het_ndx( 'CH3COCHO' )
      xooh_ndx    = get_het_ndx( 'XOOH' )
      onitr_ndx   = get_het_ndx( 'ONITR' )
      glyald_ndx  = get_het_ndx( 'GLYALD' )
      ch3cho_ndx  = get_het_ndx( 'CH3CHO' )
      mvk_ndx     = get_het_ndx( 'MVK' )
      macr_ndx    = get_het_ndx( 'MACR' )
      pooh_ndx    = get_het_ndx( 'POOH' )
      c2h5ooh_ndx = get_het_ndx( 'C2H5OOH' )
      c3h7ooh_ndx = get_het_ndx( 'C3H7OOH' )
      rooh_ndx    = get_het_ndx( 'ROOH' )
      isopno3_ndx = get_het_ndx( 'ISOPNO3' )
      onit_ndx    = get_het_ndx( 'ONIT' )
      Pb_ndx      = get_het_ndx( 'Pb' )
      macrooh_ndx = get_het_ndx( 'MACROOH' )
      isopooh_ndx = get_het_ndx( 'ISOPOOH' )
      ch3oh_ndx   = get_het_ndx( 'CH3OH' )
      c2h5oh_ndx  = get_het_ndx( 'C2H5OH' )
      hyac_ndx    = get_het_ndx( 'HYAC' )
      hydrald_ndx = get_het_ndx( 'HYDRALD' )
      alkooh_ndx  = get_het_ndx( 'ALKOOH' )
      mekooh_ndx  = get_het_ndx( 'MEKOOH' )
      tolooh_ndx  = get_het_ndx( 'TOLOOH' )
      terpooh_ndx = get_het_ndx( 'TERPOOH' )
      ch3cooh_ndx = get_het_ndx( 'CH3COOH' )
      so2_ndx     = get_het_ndx( 'SO2' )
      soa_ndx     = get_het_ndx( 'SOA' )
      so4_ndx     = get_het_ndx( 'SO4' )
      cb2_ndx     = get_het_ndx( 'CB2' )
      oc2_ndx     = get_het_ndx( 'OC2' )
      nh3_ndx     = get_het_ndx( 'NH3' )
      nh4_ndx     = get_het_ndx( 'NH4' )
      nh4no3_ndx  = get_het_ndx( 'NH4NO3' )
      sa1_ndx     = get_het_ndx( 'SA1' )
      sa2_ndx     = get_het_ndx( 'SA2' )
      sa3_ndx     = get_het_ndx( 'SA3' )
      sa4_ndx     = get_het_ndx( 'SA4' )

      write(*,*) 'sethet_inti: new ndx ',so2_ndx,so4_ndx,soa_ndx,cb2_ndx,oc2_ndx, &
                                         nh3_ndx,nh4_ndx,nh4no3_ndx,sa1_ndx,sa2_ndx,sa3_ndx,sa4_ndx
      write(*,*) ' '
      write(*,*) 'sethet_inti: diagnotics '
      write(*,'(10i5)') h2o2_ndx, hno3_ndx, ch2o_ndx, ch3ooh_ndx, ch3coooh_ndx, &
		 ho2no2_ndx, ch3cocho_ndx, xooh_ndx, onitr_ndx, glyald_ndx, &
		 ch3cho_ndx, mvk_ndx, macr_ndx, pooh_ndx, c2h5ooh_ndx, &
		 c3h7ooh_ndx, rooh_ndx, isopno3_ndx, onit_ndx, Pb_ndx, &
		 macrooh_ndx, isopooh_ndx, ch3oh_ndx, c2h5oh_ndx, hyac_ndx, hydrald_ndx

      end subroutine sethet_inti

      subroutine sethet( het_rates, press, lat, zmid,  phis, &
                         tfld, cmfdqr, nrain, nevapr, delt, &
                         xhnm, qin, plonl )
!-----------------------------------------------------------------------      
!       ... compute rainout loss rates (1/s)
!-----------------------------------------------------------------------      

      use mo_constants, only : phi, gravit, avo => avogadro, boltz
      use chem_mods,    only : hetcnt
      use mo_grid,      only : plev, pcnstm1
      use mo_mpi,       only : base_lat
      use mo_cloud,     only : ktop

      implicit none
!-----------------------------------------------------------------------      
!       ... dummy arguments
!-----------------------------------------------------------------------      
      integer, intent(in)  ::    lat                 ! latitude index
      integer, intent(in)  ::    plonl               ! lon tile dim
      real, intent(in)     ::    delt                ! time step ( s )
      real, intent(in)     ::    press(plonl,plev)   ! pressure in pascals
      real, intent(in)     ::    cmfdqr(plonl,plev)  ! dq/dt for convection
      real, intent(in)     ::    nrain(plonl,plev)   ! stratoform precip
      real, intent(in)     ::    nevapr(plonl,plev)  ! evaporation
      real, intent(in)     ::    qin(plonl,plev,pcnstm1)   ! xported species ( vmr )
      real, intent(in)     ::    zmid(plonl,plev)    ! midpoint geopot (km)
      real, intent(in)     ::    phis(plonl)         ! surf geopot
      real, intent(in)     ::    tfld(plonl,plev)    ! temperature (k)
      real, intent(in)     ::    xhnm(plonl,plev)    ! total atms density ( /cm^3)
      real, intent(out)    ::    het_rates(:,:,:)    ! rainout loss rates

!-----------------------------------------------------------------------      
!       ... local variables
!-----------------------------------------------------------------------      
      real, parameter ::  xrm   = .189               ! mean diameter of rain drop (cm)
      real, parameter ::  xum   = 748.               ! mean rain drop terminal velocity (cm/s)
      real, parameter ::  xvv   = 6.18e-2            ! kinetic viscosity (cm^2/s)
      real, parameter ::  xdg   = .112               ! mass transport coefficient (cm/s)
      real, parameter ::  t0    = 298.               ! reference temperature (k)
      real, parameter ::  xph0  = 1.e-5              ! cloud [h+]
      real, parameter ::  satf_hno3  = .016          ! saturation factor for hno3 in clouds 
      real, parameter ::  satf_h2o2  = .016          ! saturation factor for hno3 in clouds 
      real, parameter ::  satf_ch2o  = .1            ! saturation factor for hno3 in clouds 
      real, parameter ::  const0   = boltz * 1.e-6   ! (atmospheres/deg k/cm^3)
      real, parameter ::  hno3_diss = 15.4           ! hno3 dissociation constant
      real, parameter ::  geo_fac  = 6.              ! geometry factor (surf area/volume = geo_fac/diameter)
      real, parameter ::  mass_air = 29.             ! mass of background atmosphere (amu)
      real, parameter ::  mass_h2o = 18.             ! mass of water vapor (amu)
      real, parameter ::  h2o_mol  = 1.e3/mass_h2o   ! (gm/mol water)
      real, parameter ::  km2cm    = 1.e5            ! convert km to cm
      real, parameter ::  m2km     = 1.e-3           ! convert m to km
      real, parameter ::  cm3_2_m3 = 1.e-6           ! convert cm^3 to m^3
      real, parameter ::  m3_2_cm3 = 1.e6            ! convert m^3 to cm^3
      real, parameter ::  liter_per_gram = 1.e-3
      real, parameter ::  avo2  = avo * liter_per_gram * cm3_2_m3 ! (liter/gm/mol*(m/cm)^3)

      integer    ::      i, m, k, kk                 ! indicies
      real       ::      latitude                    ! latitude in radians
      real       ::      xkgm                        ! mass flux on rain drop
      real       ::      all1, all2                  ! work variables
      real       ::      stay                        ! fraction of layer traversed by falling drop in timestep delt
      real       ::      xeqca1, xeqca2, xca1, xca2, xdtm
      real       ::      xxx1, xxx2, yhno3, yh2o2
      
      real, dimension(plonl)  :: &
                         xk0, work1, work2, work3, zsurf
      real, dimension(plev)  :: &
                         xgas1, xgas2
      real, dimension(plonl)  :: &
                         tmp0_rates, tmp1_rates
      real, dimension(plonl,plev)  :: &
                         delz, &              ! layer depth about interfaces (cm)
                         xhno3, &             ! hno3 concentration (molecules/cm^3)
                         xh2o2, &             ! h2o2 concentration (molecules/cm^3)
                         xliq, &              ! liquid rain water content in a grid cell (gm/m^3)
                         rain                 ! conversion rate of water vapor into rain water (molecules/cm^3/s)
      real, dimension(plonl,plev)  :: &
                         xhen_hno3, xhen_h2o2, xhen_ch2o, xhen_ch3ooh, xhen_ch3co3h, &
                         xhen_ch3cocho, xhen_xooh, xhen_onitr, xhen_ho2no2, xhen_glyald, &
                         xhen_ch3cho, xhen_mvk, xhen_macr
      real, dimension(plonl,plev)  :: &
                         xhen_nh3, xhen_ch3cooh
      real, dimension(plonl,plev,2)  :: &
                         tmp_hetrates
      real, dimension(plonl,plev)  :: precip

!-----------------------------------------------------------------
!        note: the press array is in pascals and must be
!              mutiplied by 10 to yield dynes/cm**2.
!-----------------------------------------------------------------
!       ... set wet deposition for
!           1. h2o2         2. hno3
!           3. ch2o         4. ch3ooh
!           5. pooh         6. ch3coooh
!           7. ho2no2       8. onit
!           9. mvk         10. macr
!          11. c2h5ooh     12. c3h7ooh
!          13. rooh        14. ch3cocho
!          15. pb          16. macrooh
!          17. xooh        18. onitr
!          19. isopooh     20. ch3oh
!          21. c2h5oh      22. glyald
!          23. hyac        24. hydrald
!          25. ch3cho      26. isopno3
!-----------------------------------------------------------------

      if( hetcnt > 0 ) then
         do m = 1,max(1,hetcnt)
	    het_rates(:,:,m) = 0.
         end do
      else
	 return
      end if

!-----------------------------------------------------------------
!	... the 2 and .6 multipliers are from a formula by frossling (1938)
!-----------------------------------------------------------------
      xkgm = xdg/xrm * 2. + xdg/xrm * .6 * sqrt( xrm*xum/xvv ) * (xvv/xdg)**(1./3.) 

      latitude = phi(base_lat+lat)
      do k = ktop+1,plev
         precip(:,k) = cmfdqr(:,k) + nrain(:,k) - nevapr(:,k)
         rain(:,k)   = mass_air*precip(:,k)*xhnm(:,k) / mass_h2o
         xliq(:,k)   = precip(:,k) * delt * xhnm(:,k) / avo*mass_air * m3_2_cm3
      end do 
      if( spc_hno3_ndx > 0 ) then
         xhno3(:,:)  = qin(:,:,spc_hno3_ndx) * xhnm(:,:)
      else
         xhno3(:,:)  = 0.
      end if
      if( spc_h2o2_ndx > 0 ) then
         xh2o2(:,:)  = qin(:,:,spc_h2o2_ndx) * xhnm(:,:)
      else
         xh2o2(:,:)  = 0.
      end if
   
      zsurf(:) = m2km * phis(:) / gravit
      do k = ktop+1,plev-1
         delz(:,k) = abs( (zmid(:,k) - zmid(:,k+1))*km2cm ) 
      end do 
      delz(:,plev) = abs( (zmid(:,plev) - zsurf(:) )*km2cm ) 

!-----------------------------------------------------------------
!       ... part 0b,  for temperature dependent of henrys
!                     xxhe1 = henry con for hno3
!                     xxhe2 = henry con for h2o2
!lwh 10/00 -- take henry''s law constants from brasseur et al. [1999],
!             appendix j. for hno3, also consider dissociation to
!             get effective henry''s law constant; equilibrium
!             constant for dissociation from brasseur et al. [1999],
!             appendix k. assume ph=5 (set as xph0 above).
!             heff = h*k/[h+] for hno3 (complete dissociation)
!             heff = h for h2o2 (no dissociation)
!             heff = h * (1 + k/[h+]) (in general)
!-----------------------------------------------------------------
      do k = ktop+1,plev
          work1(:) = (t0 - tfld(:,k))/(t0*tfld(:,k))
!-----------------------------------------------------------------
! 	... effective henry''s law constants:
!	hno3, h2o2, ch2o, ch3ooh, ch3coooh (brasseur et al., 1999)
!       xooh, onitr, macrooh (j.-f. muller; brocheton, 1999)
!       isopooh (equal to hno3, as for macrooh)
!       ho2no2 (mozart-1)
!       ch3cocho, hoch2cho (betterton and hoffman, environ. sci. technol., 1988)
!       ch3cho (staudinger and roberts, crit. rev. sci. technol., 1996)
!       mvk, macr (allen et al., environ. toxicol. chem., 1998)
!-----------------------------------------------------------------
          xk0(:)             = 2.1e5 *exp( 8700.*work1(:) )
          xhen_hno3(:,k)     = xk0(:) * ( 1. + hno3_diss / xph0 )
          xhen_h2o2(:,k)     = 7.45e4 * exp( 6620. * work1(:) )
          xhen_ch2o(:,k)     = 6.3e3 * exp( 6460. * work1(:) )
          xhen_ch3ooh(:,k)   = 2.27e2 * exp( 5610. * work1(:) )
          xhen_ch3co3h(:,k)  = 4.73e2 * exp( 6170. * work1(:) )
          xhen_ch3cocho(:,k) = 3.70e3 * exp( 7275. * work1(:) )
          xhen_xooh(:,k)     = 90.5 * exp( 5607. * work1(:) )
          xhen_onitr(:,k)    = 7.51e3 * exp( 6485. * work1(:) )
          xhen_ho2no2(:,k)   = 2.e4
          xhen_glyald(:,k)   = 4.1e4 * exp( 4600. * work1(:) )
          xhen_ch3cho(:,k)   = 1.4e1 * exp( 5600. * work1(:) )
          xhen_mvk(:,k)      = 21. * exp( 7800. * work1(:) )
          xhen_macr(:,k)     = 4.3 * exp( 5300. * work1(:) )
          xhen_ch3cooh(:,k)  = 4.1e3 * exp( 6300. * work1(:) )
          xhen_nh3 (:,k)     = 1.e6
          tmp_hetrates(:,k,:) = 0.
      end do

!-----------------------------------------------------------------
!       ... part 1, solve for high henry constant ( hno3, h2o2)
!-----------------------------------------------------------------
long_loop : &
      do i = 1,plonl
         xgas1(:) = xhno3(i,:)                     ! xgas will change during 
         xgas2(:) = xh2o2(i,:)                     ! different levels wash 
level_loop1 : &
         do kk = ktop+1,plev
            stay = 1.
            if( rain(i,kk) /= 0. ) then            ! finding rain cloud           
               all1 = 0.                           ! accumulation to justisfy saturation
               all2 = 0. 
               stay = ((zmid(i,kk) - zsurf(i))*km2cm)/(xum*delt)
               stay = min( stay,1. )
!-----------------------------------------------------------------
!       ... calculate the saturation concentration eqca
!-----------------------------------------------------------------
               do k = kk,plev                      ! cal washout below cloud
                  xeqca1 =  xgas1(k) &
                            / (xliq(i,kk)*avo2 + 1./(xhen_hno3(i,k)*const0*tfld(i,k))) &
                            *  xliq(i,kk)*avo2
                  xeqca2 =  xgas2(k) &
                            / (xliq(i,kk)*avo2 + 1./(xhen_h2o2(i,k)*const0*tfld(i,k))) &
                            *  xliq(i,kk)*avo2
!-----------------------------------------------------------------
!       ... calculate ca; inside cloud concentration in #/cm3(air)
!-----------------------------------------------------------------
                  xca1 = geo_fac*xkgm*xgas1(k)/(xrm*xum)*delz(i,k) * xliq(i,kk) * cm3_2_m3
                  xca2 = geo_fac*xkgm*xgas2(k)/(xrm*xum)*delz(i,k) * xliq(i,kk) * cm3_2_m3
!-----------------------------------------------------------------
!       ... if is not saturated
!               hno3(gas)_new = hno3(gas)_old - hno3(h2o)
!           otherwise
!               hno3(gas)_new = hno3(gas)_old
!-----------------------------------------------------------------
                   all1 = all1 + xca1
                   all2 = all2 + xca2
                   if( all1 < xeqca1 ) then
                      xgas1(k) = max( xgas1(k) - xca1,0. )
                   end if
                   if( all2 < xeqca2 ) then
                      xgas2(k) = max( xgas2(k) - xca2,0. )
                   end if
               end do
            end if
!-----------------------------------------------------------------
!       ... calculate the lifetime of washout (second)
!             after all layers washout 
!             the concentration of hno3 is reduced 
!             then the lifetime xtt is calculated by
!
!                  xtt = (xhno3(ini) - xgas1(new))/(dt*xhno3(ini))
!                  where dt = passing time (s) in vertical
!                             path below the cloud
!                        dt = dz(cm)/um(cm/s)
!-----------------------------------------------------------------
            xdtm = delz(i,kk) / xum                     ! the traveling time in each dz
            xxx1 = (xhno3(i,kk) - xgas1(kk))
            xxx2 = (xh2o2(i,kk) - xgas2(kk))
            if( xxx1 /= 0. ) then                       ! if no washout lifetime = 1.e29
               yhno3  = xhno3(i,kk)/xxx1 * xdtm    
            else
               yhno3  = 1.e29
            end if
            if( xxx2 /= 0. ) then                       ! if no washout lifetime = 1.e29
               yh2o2  = xh2o2(i,kk)/xxx2 * xdtm     
            else
               yh2o2  = 1.e29 
            end if 
            tmp_hetrates(i,kk,1) = max( 1. / yh2o2,0. ) * stay
            tmp_hetrates(i,kk,2) = max( 1. / yhno3,0. ) * stay
         end do level_loop1
      end do long_loop

!-----------------------------------------------------------------
!       ... part 2, in-cloud solve for low henry constant
!                   hno3 and h2o2 have both in and under cloud
!-----------------------------------------------------------------
level_loop2 : &
      do k = ktop+1,plev
         work1(:) = avo2 * xliq(:,k)
         work2(:) = const0 * tfld(:,k)
         work3(:) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_ch2o(:,k)*work2(:)))),0. ) &
                    * satf_ch2o
	 if( ch2o_ndx > 0 ) then
            het_rates(:,k,ch2o_ndx)  = work3(:)
	 end if
	 if( isopno3_ndx > 0 ) then
            het_rates(:,k,isopno3_ndx) = work3(:)
	 end if
	 if( hyac_ndx > 0 ) then
            het_rates(:,k,hyac_ndx) = work3(:)
	 end if
	 if( hydrald_ndx > 0 ) then
            het_rates(:,k,hydrald_ndx) = work3(:)
	 end if

         work3(:) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_ch3ooh(:,k)*work2(:)))),0. )
	 if( ch3ooh_ndx > 0 ) then
            het_rates(:,k,ch3ooh_ndx)  = work3(:)
	 end if
	 if( pooh_ndx > 0 ) then
            het_rates(:,k,pooh_ndx)  = work3(:)
	 end if
	 if( c2h5ooh_ndx > 0 ) then
            het_rates(:,k,c2h5ooh_ndx) = work3(:)
	 end if
	 if( c3h7ooh_ndx > 0 ) then
            het_rates(:,k,c3h7ooh_ndx) = work3(:)
	 end if
	 if( rooh_ndx > 0 ) then
            het_rates(:,k,rooh_ndx) = work3(:)
	 end if
	 if( ch3oh_ndx > 0 ) then
            het_rates(:,k,ch3oh_ndx) = work3(:)
	 end if
	 if( c2h5oh_ndx > 0 ) then
            het_rates(:,k,c2h5oh_ndx) = work3(:)
	 end if

	 if( ch3coooh_ndx > 0 ) then
            het_rates(:,k,ch3coooh_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_ch3co3h(:,k)*work2(:)))),0. )
	 end if
	 if( ho2no2_ndx > 0 ) then
            het_rates(:,k,ho2no2_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_ho2no2(:,k)*work2(:)))),0. )
	 end if
	 if( ch3cocho_ndx > 0 ) then
            het_rates(:,k,ch3cocho_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_ch3cocho(:,k)*work2(:)))),0. )
	 end if
	 if( xooh_ndx > 0 ) then
            het_rates(:,k,xooh_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_xooh(:,k)*work2(:)))),0. )
	 end if
	 if( onitr_ndx > 0 ) then
            het_rates(:,k,onitr_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_onitr(:,k)*work2(:)))),0. )
	 end if
	 if( glyald_ndx > 0 ) then
            het_rates(:,k,glyald_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_glyald(:,k)*work2(:)))),0. )
	 end if
	 if( ch3cho_ndx > 0 ) then
            het_rates(:,k,ch3cho_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_ch3cho(:,k)*work2(:)))),0. )
	 end if
	 if( mvk_ndx > 0 ) then
            het_rates(:,k,mvk_ndx)  = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_mvk(:,k)*work2(:)))),0. )
	 end if
	 if( macr_ndx > 0 ) then
            het_rates(:,k,macr_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_macr(:,k)*work2(:)))),0. )
	 end if
	 if( h2o2_ndx > 0 ) then
            work3(:) = satf_h2o2 * max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_h2o2(:,k)*work2(:)))),0. )    
            het_rates(:,k,h2o2_ndx) =  work3(:) + tmp_hetrates(:,k,1)
	 end if

         work3(:) = tmp_hetrates(:,k,2) &
		    + satf_hno3 * max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_hno3(:,k)*work2(:)))),0. )    
         tmp0_rates(:)   = work3(:)
         tmp1_rates(:)   = .2*work3(:)
	 if( hno3_ndx > 0 ) then
            het_rates(:,k,hno3_ndx) = work3(:)
	 end if
	 if( onit_ndx > 0 ) then
            het_rates(:,k,onit_ndx) = work3(:)
	 end if
	 if( Pb_ndx > 0 ) then
            het_rates(:,k,Pb_ndx) = work3(:)
	 end if
	 if( macrooh_ndx > 0 ) then
            het_rates(:,k,macrooh_ndx) = work3(:)
	 end if
	 if( isopooh_ndx > 0 ) then
            het_rates(:,k,isopooh_ndx) = work3(:)
	 end if
         if( so2_ndx > 0 ) then
	    het_rates(:,k,so2_ndx) = het_rates(:,k,h2o2_ndx)
	 end if
         if( soa_ndx > 0 ) then
	    het_rates(:,k,soa_ndx) = tmp1_rates(:)
	 end if
         if( oc2_ndx > 0 ) then
	    het_rates(:,k,oc2_ndx) = tmp1_rates(:)
	 end if
         if( cb2_ndx > 0 ) then
	    het_rates(:,k,cb2_ndx) = tmp1_rates(:)
	 end if
         if( so4_ndx > 0 ) then
	    het_rates(:,k,so4_ndx) = tmp1_rates(:)
	 end if
         if( sa1_ndx > 0 ) then
	    het_rates(:,k,sa1_ndx) = tmp1_rates(:)
	 end if
         if( sa2_ndx > 0 ) then
	    het_rates(:,k,sa2_ndx) = tmp1_rates(:)
	 end if
         if( sa3_ndx > 0 ) then
	    het_rates(:,k,sa3_ndx) = tmp1_rates(:)
	 end if
         if( sa4_ndx > 0 ) then
	    het_rates(:,k,sa4_ndx) = tmp1_rates(:)
	 end if
         if( nh3_ndx > 0 ) then
            het_rates(:,k,nh3_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_nh3(:,k)*work2(:)))),0. ) * satf_hno3
         end if
         if( nh4_ndx > 0 ) then
	    het_rates(:,k,nh4_ndx) = tmp1_rates(:)
	 end if
         if( nh4no3_ndx > 0 ) then
	    het_rates(:,k,nh4no3_ndx ) = tmp1_rates(:)
	 end if
         if( alkooh_ndx  > 0 ) then
	    het_rates(:,k,alkooh_ndx)  = het_rates(:,k,ch3ooh_ndx)
	 end if
         if( mekooh_ndx  > 0 ) then
	    het_rates(:,k,mekooh_ndx)  = het_rates(:,k,ch3ooh_ndx)
	 end if
         if( tolooh_ndx  > 0 ) then
	    het_rates(:,k,tolooh_ndx)  = het_rates(:,k,ch3ooh_ndx)
	 end if
         if( terpooh_ndx > 0 ) then
	    het_rates(:,k,terpooh_ndx) = het_rates(:,k,ch3ooh_ndx)
	 end if
         if( ch3cooh_ndx > 0 ) then
            het_rates(:,k,ch3cooh_ndx) = max( rain(:,k) / (h2o_mol*(work1(:) + 1./(xhen_ch3cooh(:,k)*work2(:)))),0. )
         end if
      end do level_loop2

      end subroutine sethet

      end module mo_sethet
