
      module mo_setozo

      contains

      subroutine setozo( z, nw, wl, tlay, dto3, &
                         to3, o3, airlev, o3top )
!-----------------------------------------------------------------------------
!   purpose:
!   set up an altitude profile of ozone, and corresponding absorption
!   optical depths.  subroutine includes a shape-conserving scaling method
!   that allows scaling of the entire profile to a given overhead ozone
!   column amount.
!-----------------------------------------------------------------------------
!   parameters:
!   dobnew - real, overhead ozone column amount (du) to which profile     (i)
!            should be scaled.  if dobnew < 0, no scaling is done
!   nz     - integer, number of specified altitude levels in the working  (i)
!            grid
!   z      - real, specified altitude working grid (km)
!   nw     - integer, number of specified intervals + 1 in working        (i)
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in      (i)
!            working wavelength grid
!   xso3   - real, molecular absoprtion cross section (cm^2) of o3 at     (i)
!            each specified wavelength (wmo value at 273)
!   s226   - real, molecular absoprtion cross section (cm^2) of o3 at     (i)
!            each specified wavelength (value from molina and molina at 226k)
!   s263   - real, molecular absoprtion cross section (cm^2) of o3 at     (i)
!            each specified wavelength (value from molina and molina at 263k)
!   s298   - real, molecular absoprtion cross section (cm^2) of o3 at     (i)
!            each specified wavelength (value from molina and molina at 298k)
!   tlay   - real, temperature (k) at each specified altitude layer       (i)
!   dto3   - real, optical depth due to ozone absorption at each          (o)
!            specified altitude at each specified wavelength
!   to3    - real, totol ozone column                                     (o)
!-----------------------------------------------------------------------------

      use mo_params, only : kw
      use mo_waveo3, only : xso3, s226, s263, s298
      use mo_grid,   only : plev, plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: nw
      real, intent(in)    :: o3top
      real, intent(in)    :: wl(kw)
      real, intent(in)    :: z(plevp)
      real, intent(in)    :: tlay(plev) 
      real, intent(in)    :: airlev(plevp) 
      real, intent(out)   :: dto3(plev,nw)     
      real, intent(out)   :: to3(plevp)
      real, intent(inout) :: o3(plevp)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      real, parameter :: wave_lims(2) = (/ 240.5, 350. /)
      real, parameter :: t_lims(3)    = (/ 226., 263., 298. /)
      real, parameter :: tfac1 = 1./(t_lims(2) - t_lims(1))
      real, parameter :: tfac2 = 1./(t_lims(3) - t_lims(2))

      integer :: k, wn
      real    :: so3
      real    :: cz(plevp)

!-----------------------------------------------------------------------------
! 	... compute column increments
!-----------------------------------------------------------------------------
      o3(1:plevp) = o3(1:plevp)*airlev(1:plevp) 
      cz(1:plev)  = .5*(o3(2:plevp) + o3(1:plev)) * 1.e5 * (z(2:plevp) - z(1:plev))
      to3(plevp)  = o3top
      do k = plev,1,-1
         to3(k) = to3(k+1) + cz(k)
      end do

!-----------------------------------------------------------------------------
! 	... include exponential tail integral from infinity to 50 km,
!           fold tail integral into top layer
!           specify scale height near top of data.
!-----------------------------------------------------------------------------

      cz(plev) = cz(plev) + o3top
!-----------------------------------------------------------------------------
! 	... calculate ozone optical depth for each layer, with temperature 
!           correction.  output, dto3(kz,kw)
!-----------------------------------------------------------------------------
      do wn = 1,nw
         if( wl(wn) > wave_lims(1)  .and. wl(wn+1) < wave_lims(2) ) then
            do k = 1,plev
               if( tlay(k) < t_lims(2) ) then
                  so3 = s226(wn) + (s263(wn) - s226(wn)) * tfac1 * (tlay(k) - t_lims(1))
               else
                  so3 = s263(wn) + (s298(wn) - s263(wn)) * tfac2 * (tlay(k) - t_lims(2))
               end if
               dto3(k,wn) = cz(k)*so3
            end do
         else
            dto3(1:plev,wn) = cz(1:plev) * xso3(wn)
         end if
      end do

      end subroutine setozo

      end module mo_setozo

