
      module mo_setz

      private
      public :: setz

      contains

      subroutine setz( cz, tlev, c, zen, adjcoe, pht_tag )
!-----------------------------------------------------------------------------
!   adjcoe - adjust cross section coefficients                        
!-----------------------------------------------------------------------------

      use mo_params,   only : kj
      use mo_calcoe,   only : calcoe
      use mo_grid,     only : plevp
      use chem_mods,   only : phtcnt
      use mo_tuv_inti, only : nlng, ncof

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      real, intent(in)    :: zen                      ! zenith angle (degrees)
      real, intent(in)    :: cz(plevp)
      real, intent(in)    :: tlev(plevp) 
      real, intent(in)    :: c(:,:,:)
      real, intent(inout) :: adjcoe(:,:)
      character(len=32), intent(in) :: pht_tag(:)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer, parameter :: nzen = 4
      real, parameter    :: zen_angles(nzen) = (/ 20.5, 40.5, 60.5, 80. /)
      integer :: astat
      integer :: ndx
      integer :: m, n, nu
      real    :: tt
      real    :: interp_factor
      real    :: c0, c1, c2
      real    :: adj_fac(2)
      real    :: xz(plevp)
      real, allocatable :: wrk(:,:)
      character(len=32) :: jname

!-----------------------------------------------------------------------------
! 1 o2 + hv -> o + o                        
! 2 o3 -> o2 + o(1d)                        
! 3 o3 -> o2 + o(3p)                        
! 4 no2 -> no + o(3p)                       
! 5 no3 -> no + o2                          
! 6 no3 -> no2 + o(3p)                      
! 7 n2o5 -> no3 + no + o(3p)                
! 8 n2o5 -> no3 + no2                       
! 9 n2o + hv -> n2 + o(1d)                  
! 10 ho2 + hv -> oh + o                      
! 11 h2o2 -> 2 oh                            
! 12 hno2 -> oh + no                         
! 13 hno3 -> oh + no2                        
! 14 hno4 -> ho2 + no2                       
! 15 ch2o -> h + hco                         
! 16 ch2o -> h2 + co                         
! 17 ch3cho -> ch3 + hco                     
! 18 ch3cho -> ch4 + co                      
! 19 ch3cho -> ch3co + h                     
! 20 c2h5cho -> c2h5 + hco                   
! 21 chocho -> products                      
! 22 ch3cocho -> products                    
! 23 ch3coch3                                
! 24 ch3ooh -> ch3o + oh                     
! 25 ch3ono2 -> ch3o+no2                     
! 26 pan + hv -> products                    
!-----------------------------------------------------------------------------

      xz(1:plevp) = cz(1:plevp)*1.e-18
      do m = 1,nlng
         adjcoe(1:plevp,m) = 1.
      end do
      if( zen < zen_angles(1) ) then
         ndx = 1
         interp_factor = 0.
      else if( zen >= zen_angles(nzen) ) then
         ndx = nzen
         interp_factor = 0.
      else
         do ndx = 1,nzen-1
            if( zen >= zen_angles(ndx) .and. zen < zen_angles(ndx+1) ) then
               interp_factor = (zen - zen_angles(ndx))/(zen_angles(ndx+1) - zen_angles(ndx))
               exit
            end if
         end do
      end if

      allocate( wrk(plevp,2), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'setz: failed to all wrk; error = ',astat
         call endrun
      end if

      tt = tlev(1)/281.
rate_loop : &
      do m = 1,nlng
         jname = trim(pht_tag(m))
	 if( jname /= 'jo1d' .and. jname /= 'jh2o2' ) then
	    adj_fac(:) = 1.
	 else if( jname == 'jo1d' ) then
!----------------------------------------------------------------------
!    	... temperature modification
!           t0.9 (1.05) t0.95(1.025)  t1.0(1.0)  t1.15(1.02)  t1.1(1.04)  
!----------------------------------------------------------------------
            if( interp_factor /= 0. ) then
               nu = ndx+1
            else
               nu = ndx
            end if
            do n = ndx,nu
               select case( n )
	          case( 1 )
                     c0 = 4.52372 ; c1 = -5.94317 ; c2 = 2.63156
	          case( 2 )
                     c0 = 4.99378 ; c1 = -7.92752 ; c2 = 3.94715
	          case( 3 )
                     c0 = .969867 ; c1 = -.841035 ; c2 = .878835
	          case( 4 )
                     c0 = 1.07801 ; c1 = -2.39580 ; c2 = 2.32632
	       end select
               adj_fac(n-ndx+1) = c0 + tt*(c1 + c2*tt)
            end do
	 else if( jname == 'jh2o2' ) then
!----------------------------------------------------------------------
!      	... temperature modification
!           t0.9 (1.05) t0.95(1.025)  t1.0(1.0)  t1.15(1.02)  t1.1(1.04)  
!----------------------------------------------------------------------
            if( interp_factor /= 0. ) then
               nu = ndx+1
            else
               nu = ndx
            end if
            do n = ndx,nu
               select case( n )
	          case( 1 )
                     c0 = 2.43360 ; c1 = -3.61363 ; c2 = 2.19018
	          case( 2 )
                     c0 = 3.98265 ; c1 = -6.90516 ; c2 = 3.93602
	          case( 3 )
                     c0 = 3.49843 ; c1 = -5.98839 ; c2 = 3.50262
	          case( 4 )
                     c0 = 3.06312 ; c1 = -5.26281 ; c2 = 3.20980
	       end select
               adj_fac(n-ndx+1) = c0 + tt*(c1 + c2*tt)
            end do
	 end if

         call calcoe( c(:,m,ndx), xz, tt, adj_fac(1), wrk(:,1) )
         if( interp_factor /= 0. ) then
            call calcoe( c(:,m,ndx+1), xz, tt, adj_fac(2), wrk(:,2) )
            adjcoe(:,m) = wrk(:,1) + interp_factor * (wrk(:,2) - wrk(:,1))
         else
            adjcoe(:,m) = wrk(:,1)
         end if
      end do rate_loop

      deallocate( wrk )

      end subroutine setz

      end module mo_setz
