
      module mo_solar_parms

      implicit none

      private
      public :: solar_parms_init
      public :: solar_parms_timestep_init
      public :: get_solar_parms
      public :: rebin

      save

      integer               :: ntimes
      integer               :: tim_ndx
      integer,  allocatable :: dates(:)
      real                  :: dels
      real, allocatable     :: times(:)
      real, allocatable     :: f107(:)
      real, allocatable     :: f107a(:)

      contains

      subroutine solar_parms_init( ncdate, ncsec )
!---------------------------------------------------------------
!	... initialize solar parmaters
!---------------------------------------------------------------

      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_comm_comp, mpi_integer, mpi_double_precision
#endif
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : solar_parms_flsp
      use netcdf

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in)          :: ncdate
      integer, intent(in)          :: ncsec

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: ncid
      integer  :: n
      integer  :: dimid
      integer  :: varid
      integer  :: astat
      integer  :: status
      integer  :: wrk_date
      real     :: wrk_time

!-----------------------------------------------------------------------
!	... readin the solar parms dataset
!-----------------------------------------------------------------------
masterproc_only : &
      if( masternode ) then
!-----------------------------------------------------------------------
!	... open solar parms file
!-----------------------------------------------------------------------
         ncid   = open_netcdf_file( solar_parms_flsp%nl_filename, &
                                    solar_parms_flsp%local_path, &
                                    solar_parms_flsp%remote_path, masteronly=.true. )
         status = nf_inq_dimid( ncid, 'time', dimid )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to time id; error = ',status
            call endrun
         else
            write(*,*) 'solar_parms_init: time id = ',dimid
         end if
         status = nf_inq_dimlen( ncid, dimid, ntimes )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to read time dimension; error = ',status
            call endrun
         else
            write(*,*) 'solar_parms_init: time dimension = ',ntimes
         end if
         allocate( dates(ntimes), times(ntimes),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'solar_parms_init: failed to allocate dates, times; error = ',astat
            call endrun
         end if
         status = nf_inq_varid( ncid, 'date', varid )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to get date id; error = ',status
            call endrun
         end if
         status = nf_get_var_int( ncid, varid, dates )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to read dates; error = ',status
            call endrun
         else
            write(*,*) 'solar_parms_init: read dates'
         end if

         do n = 1,ntimes
            call days0( dates(n), 0, times(n) )
         end do

         wrk_date = ncdate
         wrk_date = mod( ncdate,10000)
         wrk_date = 19950000 + wrk_date
         write(*,*) ' '
         write(*,*) '--------------------------------------------------'
         write(*,*) 'solar_parms_init: values for date = ',wrk_date
         call days0( wrk_date, 0, wrk_time )
         if( wrk_time < times(1) .or. wrk_time > times(ntimes) ) then
            write(*,*) 'solar_parms_init: initial model time is out of range of solar parm times'
            write(*,*) 'solar_parms_init: min time = ',dates(1)
            write(*,*) 'solar_parms_init: max time = ',dates(ntimes)
            call endrun
         end if
         do n = 2,ntimes
            if( wrk_time <= times(n) ) then
               exit
            end if
         end do
         tim_ndx = n - 1
         dels    = (wrk_time - times(tim_ndx))/(times(tim_ndx+1) - times(tim_ndx))
         write(*,*) 'solar_parms_init: tim_ndx, dels, times(tim_ndx:tim_ndx+1) = ', &
                    tim_ndx, dels, dates(tim_ndx:tim_ndx+1)
         write(*,*) '--------------------------------------------------'
         write(*,*) ' '
!---------------------------------------------------------------
!	... allocate and read solar parms
!---------------------------------------------------------------
         allocate( f107(ntimes), f107a(ntimes), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'solar_parms_init: failed to allocate f107,f107; error = ',astat
            call endrun
         end if
         status = nf_inq_varid( ncid, 'f107', varid )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to get f107 id; error = ',status
            call endrun
         end if
         status = nf_get_var_double( ncid, varid, f107 )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to read f107; error = ',status
            call endrun
         end if
         status = nf_inq_varid( ncid, 'f107a', varid )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to get f107a id; error = ',status
            call endrun
         end if
         status = nf_get_var_double( ncid, varid, f107a )
         if( status /= 0 ) then
            write(*,*) 'solar_parms_init: failed to read f107a; error = ',status
            call endrun
         end if

         status = nf_close( ncid )
         write(*,*) 'solar_parms_init: closed file'
      end if masterproc_only

#ifdef USE_MPI
      call mpi_bcast( ntimes, 1, mpi_integer, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'solar_parms_init: failed to bcast ntimes; error = ',status
         call endrun
      end if
      call mpi_bcast( tim_ndx, 1, mpi_integer, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'solar_parms_init: failed to bcast time_ndx; error = ',status
         call endrun
      end if
      call mpi_bcast( dels, 1, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'solar_parms_init: failed to bcast dels; error = ',status
         call endrun
      end if
      if( .not. masternode ) then
         allocate( dates(ntimes), times(ntimes),stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'solar_parms_init: failed to allocate dates,times; error = ',astat
            call endrun
         end if
         allocate( f107(ntimes), f107a(ntimes), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'solar_parms_init: failed to allocate f107,f107a; error = ',astat
            call endrun
         end if
      end if
      call mpi_bcast( dates, ntimes, mpi_integer, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'solar_parms_init: failed to bcast dates; error = ',status
         call endrun
      end if
      call mpi_bcast( times, ntimes, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'solar_parms_init: failed to bcast times; error = ',status
         call endrun
      end if
      call mpi_bcast( f107, ntimes, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'solar_parms_init: failed to bcast f107; error = ',status
         call endrun
      end if
      call mpi_bcast( f107a, ntimes, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'solar_parms_init: failed to bcast f107a; error = ',status
         call endrun
      end if
#endif

      end subroutine solar_parms_init

      subroutine solar_parms_timestep_init( ncdate, ncsec )
!---------------------------------------------------------------
!	... set solar parameters timing
!---------------------------------------------------------------

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in) :: ncdate
      integer, intent(in) :: ncsec
      
!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: n
      integer  :: wrk_date
      real     :: wrk_time

      if( ncsec == 0 ) then
         wrk_date = mod( ncdate,10000)
         wrk_date = 19950000 + wrk_date
!        write(*,*) 'solar_parms_timestep_init: values for date = ',ncdate
         write(*,*) 'solar_parms_timestep_init: values for date = ',wrk_date
!        call days0( ncdate, 0, wrk_time )
         call days0( wrk_date, 0, wrk_time )
         if( wrk_time < times(1) .or. wrk_time > times(ntimes) ) then
            write(*,*) 'solar_parms_timestep_init: time is out of range of solar parm times'
            write(*,*) 'solar_parms_timestep: min time = ',dates(1)
            write(*,*) 'solar_parms_timestep: max time = ',dates(ntimes)
            call endrun
         end if
         do n = 2,ntimes
            if( wrk_time <= times(n) ) then
               exit
            end if
         end do
         tim_ndx = n - 1
         dels    = (wrk_time - times(tim_ndx))/(times(tim_ndx+1) - times(tim_ndx))
      end if
      
      end subroutine solar_parms_timestep_init

      subroutine get_solar_parms( f107_s, f107a_s )
!---------------------------------------------------------------
!	... set,retrieve solar parmaters
!---------------------------------------------------------------

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      real, optional, intent(out) :: f107_s                   ! solar euv factor
      real, optional, intent(out) :: f107a_s                  ! averaged solar euv factor

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: tnp

      tnp = tim_ndx + 1
      if( present( f107_s ) ) then
         f107_s  =  f107(tim_ndx) + dels*(f107(tnp) - f107(tim_ndx))
      end if
      if( present( f107a_s ) ) then
         f107a_s  =  f107a(tim_ndx) + dels*(f107a(tnp) - f107a(tim_ndx))
      end if

      end subroutine get_solar_parms

      subroutine rebin( nsrc, ntrg, src_x, trg_x, src, trg )
!---------------------------------------------------------------
!	... rebin src to trg
!---------------------------------------------------------------

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      integer, intent(in)   :: nsrc                  ! dimension source array
      integer, intent(in)   :: ntrg                  ! dimension target array
      real, intent(in)      :: src_x(nsrc+1)         ! source coordinates
      real, intent(in)      :: trg_x(ntrg+1)         ! target coordinates
      real, intent(in)      :: src(nsrc)             ! source array
      real, intent(out)     :: trg(ntrg)             ! target array

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: i, l
      integer  :: si, si1
      integer  :: sil, siu
      real     :: y
      real     :: sl, su
      real     :: tl, tu

!---------------------------------------------------------------
!	... check interval overlap
!---------------------------------------------------------------
      if( trg_x(1) < src_x(1) .or. trg_x(ntrg+1) > src_x(nsrc+1) ) then
         write(*,*) 'rebin: target grid is outside source grid'
         write(*,*) '       target grid from ',trg_x(1),' to ',trg_x(ntrg+1)
         write(*,*) '       source grid from ',src_x(1),' to ',src_x(nsrc+1)
         call endrun
      end if
#ifdef REBIN_DIAGS
      write(*,*) ' '
      write(*,*) '------------------------------------------------------'
      write(*,*) 'rebin: diagnostics'
      write(*,*) 'rebin: nsrc, ntrg = ',nsrc,ntrg
      write(*,*) 'rebin: src_x'
      write(*,'(1p,5g15.7)') src_x(:)
      write(*,*) 'rebin: trg_x'
      write(*,'(1p,5g15.7)') trg_x(:)
      write(*,*) '------------------------------------------------------'
      write(*,*) ' '
      write(*,*) ' '
      write(*,*) 'rebin: individual bins'
#endif
      do i = 1,ntrg
        tl     = trg_x(i)
        do sil = 1,nsrc+1
           if( tl <= src_x(sil) ) then
              exit
           end if
        end do
        tu     = trg_x(i+1)
        do siu = 1,nsrc+1
           if( tu <= src_x(siu) ) then
              exit
           end if
        end do
        y   = 0.
        sil = max( sil,2 )
        siu = min( siu,nsrc+1 )
        do si = sil,siu
           si1 = si - 1
           sl  = max( tl,src_x(si1) )
           su  = min( tu,src_x(si) )
           y   = y + (su - sl)*src(si1)
        end do
        trg(i) = y/(trg_x(i+1) - trg_x(i))
      end do

      end subroutine rebin

      subroutine days0( ncdate, ncsec, days )
!----------------------------------------------------------------------- 
! Purpose: Convert date and seconds of day to floating point days since
!          00/01/01
! 
! Method: Use table of days per month to do conversion
! 
! Author: CCM Core Group
!-----------------------------------------------------------------------


      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)   :: ncdate      ! Current date as yymmdd or yyyymmdd
      integer, intent(in)   :: ncsec       ! Seconds of day for current date
      real, intent(out) :: days        ! Day of year

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer  :: year        ! year
      integer  :: mnth        ! Month number
      integer  :: mday        ! Day number of month
      integer  :: jdcon(12) & ! Starting day number for each month
                       = (/ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 /)
      real :: doy

!-----------------------------------------------------------------------
! Decode month and day
!-----------------------------------------------------------------------
      year = ncdate/10000
      mnth = mod( ncdate,10000 )/100
      if( mnth < 1 .or. mnth > 12) then
         write(*,*) 'days0: Bad month index=', mnth
         call endrun
      end if
      mday = mod(ncdate,100)
      doy  = jdcon(mnth) + mday + ncsec/86400.

      if( doy < 1. .or. doy > 366.) then
         write(*,*) 'days0: bad day of year = ',doy
         call endrun
      end if

      days = 365.*year + doy

      end subroutine days0

      end module mo_solar_parms
