
      module mo_sphers

      contains

      subroutine sphers( z, zen, dsdh, nid )
!-----------------------------------------------------------------------------
!   purpose:
!   calculate slant path over vertical depth ds/dh in spherical geometry.
!   calculation is based on:  a.dahlback, and k.stamnes, a new spheric model
!   for computing the radiation field available for photolysis and heating
!   at twilight, planet.space sci., v39, n5, pp. 671-683, 1991 (appendix b)
!-----------------------------------------------------------------------------
!   parameters:
!   nz      - integer, number of specified altitude levels in the working (i)
!             grid
!   z       - real, specified altitude working grid (km)                  (i)
!   zen     - real, solar zenith angle (degrees)                          (i)
!   dsdh    - real, slant path of direct beam through each layer crossed  (o)
!             when travelling from the top of the atmosphere to layer i;
!             dsdh(i,j), i = 0..nz-1, j = 1..nz-1
!   nid     - integer, number of layers crossed by the direct beam when   (o)
!             travelling from the top of the atmosphere to layer i;
!             nid(i), i = 0..nz-1
!-----------------------------------------------------------------------------

      use mo_params,    only : radius
      use mo_constants, only : d2r
      use mo_grid,      only : plev, plevp

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      real, intent(in)     :: zen
      real, intent(in)     :: z(plevp)
      integer, intent(out) :: nid(0:plev)
      real, intent(out)    :: dsdh(0:plev,plev)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: i, j, k
      integer :: id
      real    :: re, ze(plevp)
      real    :: zd(0:plev)
      real    :: zenrad, rpsinz, rj, rjp1, dsj, dhj, ga, gb, sm

      zenrad = zen*d2r
!-----------------------------------------------------------------------------
! 	... include the elevation above sea level to the radius of the earth:
!-----------------------------------------------------------------------------
      re = radius + z(1)
!-----------------------------------------------------------------------------
! 	... correspondingly z changed to the elevation above earth surface:
!-----------------------------------------------------------------------------
      ze(1:plevp) = z(1:plevp) - z(1)

!-----------------------------------------------------------------------------
! 	... inverse coordinate of z
!-----------------------------------------------------------------------------
      zd(0) = ze(plevp)
      do k = 1,plev
        zd(k) = ze(plevp - k)
      end do

!-----------------------------------------------------------------------------
! 	... initialize dsdh(i,j), nid(i)
!-----------------------------------------------------------------------------
      dsdh(0:plev,1:plev) = 0.
      nid(0:plev)         = 0

!-----------------------------------------------------------------------------
! 	... calculate ds/dh of every layer
!-----------------------------------------------------------------------------
      do i = 0,plev
        rpsinz = (re + zd(i)) * sin(zenrad)
        if( zen > 90. .and. rpsinz < re ) then
           nid(i) = -1
        else
!-----------------------------------------------------------------------------
! 	... find index of layer in which the screening height lies
!-----------------------------------------------------------------------------
           id = i 
           if( zen > 90. ) then
              do j = 1,plev
                 if( (rpsinz < ( zd(j-1) + re ) ) .and. (rpsinz >= ( zd(j) + re )) ) then
		    id = j
		 end if
              end do
           end if
 
           do j = 1,id
             if( j == id .and. id == i .and. zen > 90.) then
                sm = -1.
	     else
                sm = 1.
             end if 
             rj = re + zd(j-1)
             rjp1 = re + zd(j)
             dhj = zd(j-1) - zd(j)
             ga = max( 0.,rj*rj - rpsinz*rpsinz )
             gb = max( 0.,rjp1*rjp1 - rpsinz*rpsinz )
             if( id > i .and. j == id ) then
                dsj = sqrt( ga )
             else
                dsj = sqrt( ga ) - sm*sqrt( gb )
             end if
             dsdh(i,j) = dsj / dhj
           end do
           nid(i) = id
        end if
      end do

      end subroutine sphers

      end module mo_sphers
