
      module mo_srf_emis
!---------------------------------------------------------------
! 	... surface emissions module
!---------------------------------------------------------------

      use mo_grid,   only : plon, plat
      use m_types,   only : time_ramp
      use chem_mods, only : pcnstm1

      implicit none

      type :: emission
         integer           :: spc_ndx
         integer           :: gndx
         integer           :: ntimes
         integer           :: nlons
         integer           :: tim_ndx(2)
         integer           :: jlim(2)
         integer, pointer  :: dates(:)
         real              :: mw
         character(len=80) :: filename
         real, pointer     :: times(:)
         real, pointer     :: flux(:,:,:,:)
         integer           :: xactive_type_cnt
         character(len=32) :: xactive_type(5)
         character(len=8)  :: species
         character(len=8)  :: units
         logical           :: from_file
      end type emission

      private
      public  :: baseflux, landflux, npp
      public  :: srf_emis_inti, srf_emis_set, srf_emis_chk

      save

      integer, parameter :: time_span = 12
      real, parameter :: amufac = 1.65979e-23         ! 1.e4* kg / amu

      real, allocatable :: &
               npp(:,:,:)
      real ::  baseflux = 0.
      real ::  landflux = 0.
      real    :: sf(plat)
      real    :: factor
      integer :: ntimes
      integer :: emis_cnt
      integer :: isop_ndx, c10h16_ndx, Rn_ndx
      logical :: has_emis(pcnstm1)
      logical :: Rn_emis
      character(len=80) ::   filename, lpath, mspath
      character(len=30) ::   emires                     ! emission file resolution

      type(emission), allocatable :: emissions(:)
      type(time_ramp)             :: emis_timing

      contains

      subroutine srf_emis_inti( plonl, platl, pplon, emis_timing_in, ncdate, ncsec )
!-----------------------------------------------------------------------
! 	... initialize the surface emissions
!-----------------------------------------------------------------------

      use m_tracname,    only : tracnam
      use mo_chem_utls,  only : get_spc_ndx, has_srfems, has_megan_srfems
      use mo_chem_utls,  only : has_fixed_lbc, has_xactive_srfems
      use chem_mods,     only : adv_mass
      use mo_constants,  only : d2r, pi, rearth, latwts
      use mo_control,    only : emis_flsp, xactive_emissions
      use mo_file_utils, only : open_netcdf_file
      use mo_charutl,    only : upcase
      use netcdf

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate
      integer, intent(in) :: ncsec
      type(time_ramp), intent(in) :: emis_timing_in

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      integer :: astat
      integer :: j, l, m, n                           ! Indices
      integer :: t1, t2
      integer :: ncid
      integer :: dimid
      integer :: varid
      integer :: yr, wrk_date, wrk_sec
      real    :: seq
      real    :: wrk_time
      real    :: mfactor
      character(len=8)  :: species
      character(len=8)  :: spc_name
      character(len=8)  :: time_type

!-----------------------------------------------------------------------
! 	... check emissions timing
!-----------------------------------------------------------------------
      emis_timing = emis_timing_in
      call upcase( emis_timing%type, time_type )
      emis_timing%type = time_type
      if( time_type /= 'SERIAL' .and. time_type /= 'CYCLICAL' &
                                .and. time_type /= 'FIXED' ) then
         write(*,*) 'srf_emis_inti: time type ',trim(time_type),' is not SERIAL,CYCLICAL, or FIXED'
         call endrun
      end if

      wrk_sec  = ncsec
      if( time_type == 'SERIAL' ) then
	 wrk_date = ncdate + emis_timing%yr_offset*10000
      else if( time_type == 'CYCLICAL' ) then
         wrk_date = (emis_timing%date/10000)*10000 + mod(ncdate,10000)
      else
         wrk_date = emis_timing%date
         wrk_sec  = 0
      end if
      wrk_time = days0( wrk_date, wrk_sec )
      write(*,*) 'srf_emis_inti: wrk_date,wrk_sec,wrk_time = ',wrk_date,wrk_sec,wrk_time

!-----------------------------------------------------------------------
! 	... species has surface emission ?
!-----------------------------------------------------------------------
      do m = 1,pcnstm1
         has_emis(m) = has_srfems( trim(tracnam(m)) ) .and. (.not. has_fixed_lbc( trim(tracnam(m)) ))
      end do
      isop_ndx   = get_spc_ndx( 'ISOP' )
      c10h16_ndx = get_spc_ndx( 'C10H16' )
      Rn_ndx     = get_spc_ndx( 'Rn' )
!-----------------------------------------------------------------------
! 	... Radon has an emission set in code
!-----------------------------------------------------------------------
      if( Rn_ndx > 0 ) then
         has_emis(Rn_ndx) = .false.
         Rn_emis          = has_srfems( 'Rn' ) .and. (.not. has_fixed_lbc( 'Rn' ))
      end if

      write(*,*) ' '
      emis_cnt = count( has_emis(:) )
      if( emis_cnt > 0 .or. Rn_emis ) then
         write(*,*) 'Species with surface emissions'
         do m = 1,pcnstm1
            if( has_emis(m) ) then
               write(*,*) trim(tracnam(m))
            end if
         end do
         if( Rn_emis ) then
            write(*,*) trim(tracnam(Rn_ndx))
         end if
      else
         write(*,*) 'There are no species with surface emissions'
      end if
      write(*,*) ' '
!-----------------------------------------------------------------------
! 	... for npp
!-----------------------------------------------------------------------
      emis_cnt = emis_cnt + 1

!-----------------------------------------------------------------------
! 	... allocate emissions type array
!-----------------------------------------------------------------------
      allocate( emissions(emis_cnt), stat=astat )
      if( astat/= 0 ) then
        write(*,*) 'srf_emis_inti: failed to allocate emissions array; error = ',astat
        call endrun
      end if

!-----------------------------------------------------------------------
! 	... diagnostics setup
!-----------------------------------------------------------------------
      seq = 2.*pi*1.e4*rearth**2/REAL(plon)
      sf(:plat) = seq*latwts(:plat)
      factor = 86400. * 365. &   ! sec / year
             / 6.022e23 &        ! molec / mole 
             * 1.e-12            ! Tg / g

      emires = emis_flsp%hor_res
      if( emires(1:1) /= '.' ) then
         emires = '.' // emires
      end if
      lpath  = emis_flsp%local_path
      mspath = emis_flsp%remote_path

!-----------------------------------------------------------------------
! 	... setup the emission type array
!-----------------------------------------------------------------------
      n = 0
species_loop : &
      do m = 1,pcnstm1
has_emission : &
         if( has_emis(m) ) then
            spc_name = trim( tracnam(m) )
            n        = n + 1
!-----------------------------------------------------------------------
! 	... default settings
!-----------------------------------------------------------------------
            emissions(n)%spc_ndx          = m
            if( .not. has_xactive_srfems( trim( spc_name ) ) ) then
               emissions(n)%xactive_type_cnt = 0
            else
               emissions(n)%xactive_type_cnt = 1
               if( has_megan_srfems( trim( spc_name ) ) ) then
                  emissions(n)%xactive_type(1)  = 'biogenic'
               end if
            end if
            emissions(n)%units            = 'Tg/y'
            emissions(n)%species          = spc_name
            emissions(n)%mw               = adv_mass(m)                     ! g / mole
            emissions(n)%filename         = 'emissions.' // trim(spc_name) // '.surface' // trim(emires) // '.nc'
            emissions(n)%from_file        = .true.
!-----------------------------------------------------------------------
! 	... species specific modifications
!-----------------------------------------------------------------------
            select case( spc_name )
               case( 'NO' )
!-----------------------------------------------------------------------
! 	... no
!-----------------------------------------------------------------------
                  emissions(n)%mw       = 14.00674                   ! g N / mole
                  emissions(n)%units    = 'TgN/y'
                  if( xactive_emissions ) then
                     emissions(n)%xactive_type(1)  = 'soil'
                  end if
!-----------------------------------------------------------------------
! 	... c2h4
!-----------------------------------------------------------------------
               case( 'C2H4' )
                  emissions(n)%mw    = 2.*12.011                       ! g C / mole
                  emissions(n)%units = 'TgC/y'
!-----------------------------------------------------------------------
! 	... c2h6
!-----------------------------------------------------------------------
               case( 'C2H6' )
                  emissions(n)%mw    = 2.*12.011                       ! g C / mole
                  emissions(n)%units = 'TgC/y'
!-----------------------------------------------------------------------
! 	... c3h6
!-----------------------------------------------------------------------
               case( 'C3H6' )
                  emissions(n)%mw    = 3.*12.011                       ! g C / mole
                  emissions(n)%units = 'TgC/y'
!-----------------------------------------------------------------------
! 	... c3h8
!-----------------------------------------------------------------------
               case( 'C3H8' )
                  emissions(n)%mw = 3.*12.011                       ! g C / mole
!-----------------------------------------------------------------------
! 	... isoprene
!-----------------------------------------------------------------------
               case( 'ISOP' )
                  emissions(n)%mw       = 5.*12.011                      ! g C / mole
                  emissions(n)%units    = 'TgC/y'
!-----------------------------------------------------------------------
! 	... terpene
!-----------------------------------------------------------------------
               case( 'C10H16' )
                  emissions(n)%mw       = 10.*12.011                     ! g C / mole
                  emissions(n)%units    = 'TgC/y'
!-----------------------------------------------------------------------
! 	... oc1
!-----------------------------------------------------------------------
               case( 'OC1' )
                  emissions(n)%mw       = 12.011                         ! g C / mole
                  emissions(n)%units    = 'TgC/y'
!-----------------------------------------------------------------------
! 	... oc2
!-----------------------------------------------------------------------
               case( 'OC2' )
                  emissions(n)%mw       = 12.011                         ! g C / mole
                  emissions(n)%filename = 'emissions.OC1.surface' // trim(emires) // '.nc'
                  emissions(n)%units    = 'TgC/y'
!-----------------------------------------------------------------------
! 	... cb1
!-----------------------------------------------------------------------
               case( 'CB1' )
                  emissions(n)%mw       = 12.011                         ! g C / mole
                  emissions(n)%units    = 'TgC/y'
!-----------------------------------------------------------------------
! 	... cb2
!-----------------------------------------------------------------------
               case( 'CB2' )
                  emissions(n)%mw       = 12.011                         ! g C / mole
                  emissions(n)%filename = 'emissions.CB1.surface' // trim(emires) // '.nc'
                  emissions(n)%units    = 'TgC/y'
            end select
         end if has_emission
      end do species_loop

!-----------------------------------------------------------------------
! 	... npp
!-----------------------------------------------------------------------
      n = emis_cnt
      emissions(n)%species  = 'NPP'
      emissions(n)%mw       = 1.               ! g / mole
      emissions(n)%units    = ' '
      emissions(n)%filename = 'emissions.npp.surface' // trim(emires) // '.nc'
      emissions(n)%xactive_type_cnt = 0
      emissions(n)%from_file        = .true.

!-----------------------------------------------------------------------
! 	... diagnostics
!-----------------------------------------------------------------------
      write(*,*) ' '
      write(*,*) 'srf_emis_inti: diagnostics'
      write(*,*) ' '
      write(*,*) 'emission timing specs'
      write(*,*) 'type = ',emis_timing%type
      if( time_type == 'SERIAL' ) then
         write(*,*) 'year offset = ',emis_timing%yr_offset
      else if( time_type == 'CYCLICAL' ) then
         write(*,*) 'year = ',emis_timing%date/10000
      else
         write(*,*) 'date = ',emis_timing%date
      end if
      write(*,*) ' '
      write(*,*) 'there are ',emis_cnt,' species with surface emission files'
      do m = 1,emis_cnt
         write(*,*) ' '
         write(*,*) 'emission type ',m
         write(*,*) 'species = ',trim(emissions(m)%species)
         write(*,*) 'spc ndx = ',emissions(m)%spc_ndx
         write(*,*) 'mw      = ',emissions(m)%mw
         write(*,*) 'units   = ',trim(emissions(m)%units)
         write(*,*) 'filename= ',trim(emissions(m)%filename)
      end do
      write(*,*) ' '

!-----------------------------------------------------------------------
! 	... get timing information, allocate arrays, and read in dates
!-----------------------------------------------------------------------
emis_loop : &
      do m = 1,emis_cnt
         ncid = open_netcdf_file( emissions(m)%filename, lpath, mspath, abort=.false. )
         if( ncid < 0 ) then
            emissions(m)%from_file = .false.
            cycle emis_loop
         end if
         call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid ), &
                            'srf_emis_inti: Failed to find time dimension for species ' // trim(emissions(m)%species) )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, emissions(m)%ntimes ), &
                            'srf_emis_inti: Failed to get length of dimension date for species ' // &
                            trim(emissions(m)%species) )
         allocate( emissions(m)%dates(emissions(m)%ntimes),stat=astat )
         if( astat/= 0 ) then
	    write(*,*) 'srf_emis_inti: failed to allocate dates array for species ',trim(emissions(m)%species),'; error = ',astat
	    call endrun
         end if
         allocate( emissions(m)%times(emissions(m)%ntimes),stat=astat )
         if( astat/= 0 ) then
	    write(*,*) 'srf_emis_inti: failed to allocate times array for species ',trim(emissions(m)%species),'; error = ',astat
	    call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'date', varid ), &
                            'srf_emis_inti: Failed to find dat lat for species ' // trim(emissions(m)%species) )
         call handle_ncerr( nf_get_var_int( ncid, varid, emissions(m)%dates ), &
                            'srf_emis_inti: Failed to read variable dates for species ' // trim(emissions(m)%species) )
         do n = 1,emissions(m)%ntimes
            emissions(m)%times(n) = days0( emissions(m)%dates(n), 0 )
         end do
	 if( time_type /= 'CYCLICAL' ) then
            if( wrk_time < emissions(m)%times(1) .or. wrk_time > emissions(m)%times(emissions(m)%ntimes) ) then
               write(*,*) 'srf_emis_inti: emission time out of bounds for dataset = ',trim(emissions(m)%filename)
               call endrun
            end if
            do n = 2,emissions(m)%ntimes
               if( wrk_time <= emissions(m)%times(n) ) then
                  exit
               end if
            end do
            emissions(m)%tim_ndx(1) = n - 1
         else
	    yr = emis_timing%date/10000
            do n = 1,emissions(m)%ntimes
               if( yr == emissions(m)%dates(n)/10000 ) then
                  exit
               end if
            end do
	    if( n >= emissions(m)%ntimes ) then
               write(*,*) 'srf_emis_inti: emission time out of bounds for dataset = ',trim(emissions(m)%filename)
               call endrun
	    end if
            emissions(m)%tim_ndx(1) = n
         end if
         select case( time_type )
            case( 'FIXED' )
               emissions(m)%tim_ndx(2) = n
            case( 'CYCLICAL' )
               do n = emissions(m)%tim_ndx(1),emissions(m)%ntimes
                  if( yr /= emissions(m)%dates(n)/10000 ) then
                     exit
                  end if
               end do
               emissions(m)%tim_ndx(2) = n - 1
	       if( (emissions(m)%tim_ndx(2) - emissions(m)%tim_ndx(1)) < 2 ) then
                  write(*,*) 'srf_emis_inti: cyclical emissions require at least two time points'
                  call endrun
	       end if
            case( 'SERIAL' )
               emissions(m)%tim_ndx(2) = min( emissions(m)%ntimes,emissions(m)%tim_ndx(1) + time_span )
         end select
         t1 = emissions(m)%tim_ndx(1)
         t2 = emissions(m)%tim_ndx(2)
!-----------------------------------------------------------------------
! 	... allocate emission array
!-----------------------------------------------------------------------
         allocate( emissions(m)%flux(plonl,platl,pplon,t1:t2),stat=astat )
         if( astat/= 0 ) then
	    write(*,*) 'srf_emis_inti: failed to allocate emissions flux; error = ',astat
	    call endrun
         end if
	 write(*,*) ' '
	 write(*,*) 'emission time cnt = ',emissions(m)%ntimes
	 write(*,*) 'emission times'
	 write(*,'(10i10)') emissions(m)%dates(:)
	 write(*,'(1p,5g15.7)') emissions(m)%times(:)
	 write(*,*) 'emission time indicies = ',emissions(m)%tim_ndx(:)
	 write(*,'(10i10)') emissions(m)%dates(t1:t2)
	 write(*,*) ' '
!-----------------------------------------------------------------------
! 	... readin the surface emissions
!-----------------------------------------------------------------------
         call srf_emis_get( plonl, platl, pplon, ncid, emissions(m), .true. )
!-----------------------------------------------------------------------
! 	... xform emissions from molecules/cm^2/s to kg/m^2/s
!-----------------------------------------------------------------------
         if( m < emis_cnt ) then
            mfactor = amufac * adv_mass(emissions(m)%spc_ndx)
            emissions(m)%flux(:,:,:,:) = mfactor * emissions(m)%flux(:,:,:,:)
         end if
!-----------------------------------------------------------------------
! 	... close the file
!-----------------------------------------------------------------------
         call handle_ncerr( nf_close( ncid ), &
                            'srf_emis_inti: Failed to close NetCDF file =' // trim(emissions(m)%filename) )
      end do emis_loop

!-----------------------------------------------------------------------
! 	... allocate npp array
!-----------------------------------------------------------------------
      allocate( npp(plonl,platl,pplon),stat=astat )
      if( astat/= 0 ) then
         write(*,*) 'srf_emis_inti: failed to allocate npp array; error = ',astat
	 call endrun
      end if

      end subroutine srf_emis_inti

      subroutine srf_emis_chk( plonl, platl, pplon, ncdate, ncsec )
!-----------------------------------------------------------------------
!       ... check serial case for time span
!-----------------------------------------------------------------------

      use netcdf
      use mo_file_utils, only : open_netcdf_file
      use chem_mods,     only : adv_mass

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      integer, intent(in) :: ncdate              ! present date (yyyymmdd)
      integer, intent(in) :: ncsec               ! seconds in present date

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer                     :: m
      integer                     :: t1, t2
      integer                     :: astat
      integer                     :: ncid
      real                        :: wrk_time
      real                        :: mfactor

      if( emis_cnt < 1 .or. emis_timing%type /= 'SERIAL' ) then
	 return
      end if
      wrk_time = days0( ncdate + emis_timing%yr_offset*10000, ncsec )

emis_loop : &
      do m = 1,emis_cnt
         if( emissions(m)%from_file .and. &
             wrk_time > emissions(m)%times(emissions(m)%tim_ndx(2)) ) then
            emissions(m)%tim_ndx(1) = emissions(m)%tim_ndx(2)
            emissions(m)%tim_ndx(2) = min( emissions(m)%ntimes,emissions(m)%tim_ndx(1) + time_span )
            t1 = emissions(m)%tim_ndx(1)
            t2 = emissions(m)%tim_ndx(2)
!-----------------------------------------------------------------------
! 	... allocate emission array
!-----------------------------------------------------------------------
            if( associated( emissions(m)%flux ) ) then
               deallocate( emissions(m)%flux,stat=astat )
               if( astat/= 0 ) then
	          write(*,*) 'srf_emis_chk: failed to deallocate emissions flux; error = ',astat
	          call endrun
               end if
	    end if
            allocate( emissions(m)%flux(plonl,platl,pplon,t1:t2),stat=astat )
            if( astat/= 0 ) then
	       write(*,*) 'srf_emis_chk: failed to allocate emissions flux; error = ',astat
	       call endrun
	    end if

            ncid = open_netcdf_file( emissions(m)%filename, lpath, mspath )
!-----------------------------------------------------------------------
! 	... readin the surface emissions
!-----------------------------------------------------------------------
            call srf_emis_get( plonl, platl, pplon, ncid, emissions(m), .false. )
!-----------------------------------------------------------------------
! 	... xform emissions from molecules/cm^2/s to kg/m^2/s
!-----------------------------------------------------------------------
            if( m < emis_cnt ) then
               mfactor = amufac * adv_mass(emissions(m)%spc_ndx)
               emissions(m)%flux(:,:,:,:) = mfactor * emissions(m)%flux(:,:,:,:)
            end if
!-----------------------------------------------------------------------
! 	... close the file
!-----------------------------------------------------------------------
            call handle_ncerr( nf_close( ncid ), &
                               'srf_emis_chk: Failed to close NetCDF file =' // trim(emissions(m)%filename) )
         end if
      end do emis_loop

      end subroutine srf_emis_chk

      subroutine srf_emis_get( plonl, platl, pplon, ncid, emissions, initial )
!-----------------------------------------------------------------------
!       ... read surface emissions from NetCDF file
!-----------------------------------------------------------------------

      use mo_mpi
      use netcdf
      use mo_constants,  only : phi, lam, d2r
      use mo_file_utils, only : open_netcdf_file
      use mo_regrider,   only : regrid_inti, regrid_2d, regrid_lat_limits

      implicit none

!-----------------------------------------------------------------------
!       ... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)           :: plonl
      integer, intent(in)           :: platl
      integer, intent(in)           :: pplon
      integer, intent(in)           :: ncid
      logical, intent(in)           :: initial
      type(emission), intent(inout) :: emissions

!-----------------------------------------------------------------------
!       ... local variables
!-----------------------------------------------------------------------
      integer                     :: j, jl, ju, m               ! Indices
      integer                     :: t1, t2, tcnt
      integer                     :: ierr
      integer                     :: vid, nlat, nlon, nvars, ndims, attlen
      integer                     :: dimid_lat, dimid_lon, dimid_time
      integer                     :: gndx
      integer                     :: jlim(2)
      integer, allocatable        :: dimid(:)
      real                        :: total, total_wrk, scale_factor
      real, allocatable           :: lat(:)
      real, allocatable           :: lon(:)
      real, allocatable           :: wrk(:,:,:)
      real                        :: wrk2d(plon,platl)
      character(len=nf_max_name)  :: varname
      character(len=100)          :: var_longname
      character(len=8)            :: species

      species = emissions%species
      if( initial ) then
!-----------------------------------------------------------------------
!       ... get grid dimensions from file
!-----------------------------------------------------------------------
!           latitudes
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                            'srf_emis_get: Failed to find dimension lat for species ' // trim(species) )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                            'srf_emis_get: Failed to get length of dimension lat for species ' // &
                            trim(species) )
         allocate( lat(nlat), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'srf_emis_get: lat allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                            'srf_emis_get: Failed to find variable lat for species ' // trim(species) )
         call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                            'srf_emis_get: Failed to read variable lat for species ' // trim(species) )
         lat(:nlat) = lat(:nlat) * d2r
 
!-----------------------------------------------------------------------
!           longitudes
!-----------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                            'srf_emis_get: Failed to find dimension lon for species ' // trim(species) )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                            'srf_emis_get: Failed to find dimension lon for species ' // trim(species) )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                            'srf_emis_get: Failed to get length of dimension lon for species ' // &
                            trim(species) )
         allocate( lon(nlon), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'srf_emis_get: lon allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                            'srf_emis_get: Failed to find variable lon for species ' // trim(species) )
         call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                            'srf_emis_get: Failed to read variable lon for species ' // trim(species) )
         lon(:nlon) = lon(:nlon) * d2r
!-----------------------------------------------------------------------
!       ... set up regridding
!-----------------------------------------------------------------------
         gndx = regrid_inti( nlat, plat, &
                             nlon, plon, &
                             lon,  lam, &
                             lat,  phi, &
                             0, platl, &
                             do_lons=.true., do_lats=.true. )
         deallocate( lat, lon, stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'srf_emis_get: Failed to deallocate lat,lon; ierr = ',ierr
            call endrun
         end if
         if( gndx /= 0 ) then
            jlim = regrid_lat_limits( gndx )
         else
            jlim = (/ base_lat+1,base_lat+platl /)
         end if
         write(*,'(1x,''srf_emis_get: gndx='',i2,'', grid limits = '',2i4,'', jl,ju='',2i4)') gndx,jlim,base_lat+1,base_lat+platl
	 emissions%gndx    = gndx
	 emissions%nlons   = nlon
	 emissions%jlim(:) = jlim(:)
      else
	 gndx    = emissions%gndx
	 nlon    = emissions%nlons
	 jlim(:) = emissions%jlim(:)
      end if

      call handle_ncerr( nf_inq_dimid( ncid, 'time', dimid_time ), &
                         'srf_emis_get: Failed to find dimension time for species ' // trim(species) )

      call handle_ncerr( nf_inq_nvars( ncid, nvars ), &
                         'srf_emis_get: Failed to get number of variables for species ' // &
                         trim(species) )

!-----------------------------------------------------------------------
!       ... initialize total emissions to zero
!-----------------------------------------------------------------------
      emissions%flux(:,:,:,:) = 0.
!-----------------------------------------------------------------------
!       ... loop over file variables
!-----------------------------------------------------------------------
      if( masternode ) then
         write(*,*) 'Annual average ',trim(emissions%species),' emissions ',trim(emissions%units)
      end if

      t1 = emissions%tim_ndx(1)
      t2 = emissions%tim_ndx(2)
types_loop : &
      do vid = 1,nvars
         ierr = nf_inq_varname( ncid, vid, varname )
         if( ierr /= 0 ) then
            write(*,*) 'srf_emis_get: Failed to get name of variable # ',vid, &
                '; species=' // trim(species)
            call endrun
         end if
!-----------------------------------------------------------------------
!       ... check for interactive emission
!-----------------------------------------------------------------------
         if( emissions%xactive_type_cnt > 0 ) then
            do m = 1,emissions%xactive_type_cnt
               if( trim(varname) == trim(emissions%xactive_type(m)) ) then
                  cycle types_loop
               end if
            end do
         end if
         call handle_ncerr( nf_inq_varndims( ncid, vid, ndims ), &
                            'srf_emis_get: Failed to get number of dimensions for ' // &
                            'variable ' // trim(varname) // ', species=' // trim(species) )
         if( ndims < 3 ) then
            cycle types_loop
         else if( ndims > 3 ) then
            write(*,*) 'srf_emis_get: Skipping variable ', trim(varname),', ndims = ',ndims, &
                       ' , species=',trim(species)
            cycle types_loop
         end if
         if( initial ) then
!-----------------------------------------------------------------------
!       ... check order of dimensions; must be (lon, lat, date).
!-----------------------------------------------------------------------
            allocate( dimid(ndims),stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'srf_emis_get: dimid allocation error = ',ierr
               call endrun
            end if
            call handle_ncerr( nf_inq_vardimid( ncid, vid, dimid ), &
                               'srf_emis_get: Failed to get dimension IDs for variable ' // &
                               trim(varname)  // ', species=' // trim(emissions%species) )
            if( dimid(1) /= dimid_lon .or. dimid(2) /= dimid_lat .or. dimid(3) /= dimid_time ) then
               write(*,*) 'srf_emis_get: Dimensions in wrong order for variable ',trim(varname)
               write(*,*) '...      Expecting (lon, lat, date)'
               call endrun
            end if
            deallocate( dimid,stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'srf_emis_get: Failed to deallocate dimid, ierr = ',ierr
               call endrun
            end if
         end if
!-----------------------------------------------------------------------
!       ... check for long_name of variable
!-----------------------------------------------------------------------
         var_longname = ' '
         ierr = nf_get_att_text( ncid, vid, 'long_name', var_longname )
         if( ierr /= 0 ) then
            var_longname = trim( varname )
         end if
!-----------------------------------------------------------------------
!       ... read data from this emission type
!-----------------------------------------------------------------------
         tcnt = t2 - t1 + 1
         allocate( wrk(nlon,jlim(1):jlim(2),tcnt), stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'srf_emis_get: wrk allocation error = ',ierr
            call endrun
         end if
         call handle_ncerr( nf_get_vara_double( ncid, vid, &
                                                (/ 1, jlim(1), t1/), &                     ! start
                                                (/ nlon, jlim(2)-jlim(1)+1, tcnt /), &  ! count
                                                wrk ), &
                            'srf_emis_get: Failed to read variable ' // trim( varname ) // &
                            ', species=' // trim(species) )
!-----------------------------------------------------------------------
!       ... check for scale_factor
!-----------------------------------------------------------------------
         ierr = nf_get_att_double( ncid, vid, 'scale_factor', scale_factor )
         if( ierr /= NF_NOERR  ) then
            scale_factor = 1.
         end if

         total = 0.
         jl    = base_lat + 1
         ju    = base_lat + platl
         do m = t1,t2
            call regrid_2d( wrk(:,jlim(1):jlim(2),m-t1+1), wrk2d, gndx, jl, ju, &
                            do_poles=.true., scaling=scale_factor )
            do j = 1,platl
               total = total + sf(base_lat+j) * sum( wrk2d(:plon,j) )
            end do
            emissions%flux(:,:,:,m) = emissions%flux(:,:,:,m) &
                          + reshape( wrk2d, (/plonl,platl,pplon/), order = (/1,3,2/) )
         end do
         deallocate( wrk,stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'srf_emis_get: Failed to deallocate wrk, ierr = ',ierr
            call endrun
         end if
!-----------------------------------------------------------------------
!       ... output global emission from this source
!-----------------------------------------------------------------------
         if( emis_timing%type /= 'FIXED' ) then
#ifdef USE_MPI
            call mpi_reduce( total, total_wrk, 1, mpi_double_precision, mpi_sum, 0, mpi_comm_comp, ierr )
#endif
            if( masternode ) then
#ifdef USE_MPI
               if( ierr /= mpi_success ) then
                  write(*,*) 'srf_emis_get: mpi_allreduce for total failed; error = ',ierr
                  call endrun
               end if
               total = total_wrk
#endif
	       if( vid == 1 ) then
                  write(*,*) ' '
	       end if
               scale_factor = 1./real( emissions%tim_ndx(2) - emissions%tim_ndx(1) + 1 )
               total = total * factor * emissions%mw * scale_factor  ! convert from molec/s to Tg/y
               write(*,'(2a10,1x,a50,1x,g10.5)') trim(species), trim(varname), trim(var_longname), total
            end if
         end if
      end do types_loop

!-----------------------------------------------------------------------
!       ... aerosol emission scaling
!-----------------------------------------------------------------------
      select case( emissions%species )
         case( 'CB1' )
            emissions%flux(:,:,:,:) = emissions%flux(:,:,:,:) *.8
         case( 'CB2' )
            emissions%flux(:,:,:,:) = emissions%flux(:,:,:,:) *.2
         case( 'OC1' )
            emissions%flux(:,:,:,:) = emissions%flux(:,:,:,:) *.5
         case( 'OC2' )
            emissions%flux(:,:,:,:) = emissions%flux(:,:,:,:) *.5
      end select

!-----------------------------------------------------------------------
!       ... output total global emission for this species
!-----------------------------------------------------------------------
      if( emis_timing%type /= 'FIXED' ) then
         total = 0.
         do j = 1,platl
            total = total + sf(base_lat+j) * sum( emissions%flux(:plonl,j,:pplon,t1:t2) )
         end do
#ifdef USE_MPI
         call mpi_reduce( total, total_wrk, 1, mpi_double_precision, mpi_sum, 0, mpi_comm_comp, ierr )
#endif
         if( masternode ) then
#ifdef USE_MPI
            if( ierr /= mpi_success ) then
               write(*,*) 'srf_emis_get: mpi_allreduce for total failed; error = ',ierr
               call endrun
            end if
            total = total_wrk
#endif
            scale_factor = 1./real( emissions%tim_ndx(2) - emissions%tim_ndx(1) + 1 )
            total = total * factor * emissions%mw * scale_factor ! convert from molec/s to Tg/y
            write(*,'(2a10,1x,a50,1x,g10.5)') trim(species), 'TOTAL', 'TOTAL', total
         end if
      end if

      end subroutine srf_emis_get

      subroutine srf_emis_set( lat, ip, ncdate, ncsec, sflx, &
                               oro, local_angle, polar_night, polar_day, sunon, &
                               sunoff, plonl )
!--------------------------------------------------------
!	... form the surface fluxes for this latitude slice
!--------------------------------------------------------

      use mo_constants, only : pi, phi, d2r, lat60, lat70
      use mo_grid,      only : pcnst
      use mo_mpi,       only : base_lat
      use mo_calendar,  only : caldayr
      use m_tracname,   only : tracnam
      use mo_chem_utls, only : has_srfems

      implicit none

!--------------------------------------------------------
!	... dummy arguments
!--------------------------------------------------------
      integer, intent(in) ::   plonl
      integer, intent(in) ::   lat                 ! latitude index
      integer, intent(in) ::   ip                  ! longitude tile index
      integer, intent(in) ::   ncdate              ! present date (yyyymmdd)
      integer, intent(in) ::   ncsec               ! seconds in present date
      integer, intent(in) ::   oro(plonl)          ! land surface type
      real, intent(in)    ::   sunon               ! sunrise angle in radians
      real, intent(in)    ::   sunoff              ! sundown angle in radians
      real, intent(in)    ::   local_angle(plonl)  ! local "time" angle
      real, intent(inout) ::   sflx(plonl,pcnst)   ! surface emissions ( kg/m^2/s )
      logical, intent(in) ::   polar_night         ! polar night flag
      logical, intent(in) ::   polar_day           ! polar day flag

!--------------------------------------------------------
!	... local variables
!--------------------------------------------------------
      integer, parameter :: land  = 1
      integer, parameter :: ocean = 0
      integer  ::  surf_type
      integer  ::  i, m, n
      integer  ::  last, next
      integer  ::  wrk_date, wrk_sec
      integer  ::  tcnt
      integer  ::  astat
      real     ::  factor, dels, phir
      real     ::  wrk_time
      real     ::  dayfrac            ! fration of day in light
      real     ::  iso_off            ! time iso flux turns off
      real     ::  iso_on             ! time iso flux turns on

      phir = phi(base_lat+lat)
!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
      wrk_sec  = ncsec
      select case( emis_timing%type )
         case( 'SERIAL' )
            wrk_date = ncdate + emis_timing%yr_offset*10000
         case( 'CYCLICAL' )
            wrk_date = (emis_timing%date/10000)*10000 + mod( ncdate,10000 )
         case( 'FIXED' )
            wrk_date = emis_timing%date
            wrk_sec  = 0
      end select
      wrk_time = days0( wrk_date, wrk_sec )

!--------------------------------------------------------
!	... set non-zero emissions
!--------------------------------------------------------
emis_loop : &
      do m = 1,emis_cnt
!--------------------------------------------------------
!	... set time interpolation factor
!--------------------------------------------------------
emis_from_file : &
         if( emissions(m)%from_file ) then
            if( emis_timing%type /= 'CYCLICAL' ) then
               do n = emissions(m)%tim_ndx(1)+1,emissions(m)%tim_ndx(2)
                  if( wrk_time <= emissions(m)%times(n) ) then
                     last = n - 1
                     next = n
                     exit
                  end if
               end do
               if( n > emissions(m)%ntimes ) then
                  write(*,*) 'srf_emis_set: interp time is out of bounds'
                  call endrun
               end if
               dels = (wrk_time - emissions(m)%times(last)) &
                      /(emissions(m)%times(next) - emissions(m)%times(last))
            else
               tcnt = emissions(m)%tim_ndx(2) - emissions(m)%tim_ndx(1) + 1
               call moz_findplb( emissions(m)%times(emissions(m)%tim_ndx(1)), tcnt, wrk_time, n )
               if( n < tcnt ) then
                  last = emissions(m)%tim_ndx(1) + n - 1
                  next = last + 1
                  dels = (wrk_time - emissions(m)%times(last)) &
                         /(emissions(m)%times(next) - emissions(m)%times(last))
               else
                  next = emissions(m)%tim_ndx(1)
                  last = emissions(m)%tim_ndx(2)
                  dels = wrk_time - emissions(m)%times(last)
                  if( dels < 0. ) then
                     dels = 365. + dels
                  end if
                  dels = dels/(365. + emissions(m)%times(next) - emissions(m)%times(last))
               end if
            end if

            dels = max( min( 1.,dels ),0. )
            if( m < emis_cnt ) then
               n = emissions(m)%spc_ndx
               sflx(:,n) = emissions(m)%flux(:,lat,ip,last) &
                           + dels * (emissions(m)%flux(:,lat,ip,next) - emissions(m)%flux(:,lat,ip,last))
            else
               npp(:,lat,ip) = emissions(m)%flux(:,lat,ip,last) &
                               + dels * (emissions(m)%flux(:,lat,ip,next) - emissions(m)%flux(:,lat,ip,last))
            end if
         end if emis_from_file
      end do emis_loop

!--------------------------------------------------------
!	... adjust alpha-pinene for diurnal variation
!--------------------------------------------------------
      if( c10h16_ndx > 0 ) then
         if( has_emis(c10h16_ndx) ) then
            if( .not. polar_night .and. .not. polar_day ) then
	       dayfrac = sunoff / pi
	       sflx(:,c10h16_ndx) = sflx(:,c10h16_ndx) / (.7 + .3*dayfrac)
	       where( local_angle >= sunoff .and. local_angle <= sunon )
	          sflx(:,c10h16_ndx) = sflx(:,c10h16_ndx) * .7
               endwhere
            end if
         end if
      end if

!--------------------------------------------------------
!	... adjust isoprene for diurnal variation
!--------------------------------------------------------
      if( isop_ndx > 0 ) then
         if( has_emis(isop_ndx) ) then
            if( .not. polar_night ) then
	       if( polar_day ) then
	          iso_off = .8 * pi
	          iso_on  = 1.2 * pi
	       else
	          iso_off = .8 * sunoff
	          iso_on  = 2. * pi - iso_off
	       end if
               do i = 1,plonl
	          if( local_angle(i) >= iso_off .and. local_angle(i) <= iso_on ) then
	             sflx(i,isop_ndx) = 0.
	          else
	             factor = local_angle(i) - iso_on
	             if( factor <= 0. ) then
	                factor = factor + 2.*pi
                     end if
	             factor = factor / (2.*iso_off + 1.e-6)
                     sflx(i,isop_ndx) = sflx(i,isop_ndx) * 2. / iso_off * pi * (sin(pi*factor))**2
	          end if
               end do
            else
	       sflx(:,isop_ndx) = 0.
            end if
         end if
      end if

!--------------------------------------------------------
!       ... set Radon flux 
!           if lat > 70N or lat < 60S then zero the flux
!--------------------------------------------------------
      if( Rn_emis ) then
         if( phir >= lat70 .or. phir <= -lat60 ) then
            sflx(:,Rn_ndx) = baseflux
         else
            do i = 1,plonl
               surf_type = oro(i)
               if( surf_type == land ) then
                  if( phir >= lat60 ) then
                     sflx(i,Rn_ndx) = .5*landflux
                  else
                     sflx(i,Rn_ndx) = landflux
                  end if
               else if( surf_type == ocean ) then
                  sflx(i,Rn_ndx)  = baseflux
               else
                  sflx(i,Rn_ndx) = baseflux
               end if
            end do
         end if
      end if

      end subroutine srf_emis_set

      real function days0( ncdate, ncsec )
!----------------------------------------------------------------------- 
! Purpose: Convert date and seconds of day to floating point days since
!          00/01/01
! 
! Method: Use table of days per month to do conversion
! 
! Author: CCM Core Group
!-----------------------------------------------------------------------


      implicit none

!-----------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in)   :: ncdate      ! Current date as yymmdd or yyyymmdd
      integer, intent(in)   :: ncsec       ! Seconds of day for current date

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer  :: year        ! year
      integer  :: mnth        ! Month number
      integer  :: mday        ! Day number of month
      integer  :: jdcon(12) & ! Starting day number for each month
                       = (/ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 /)
      real     :: doy

!-----------------------------------------------------------------------
! 	... decode month and day
!-----------------------------------------------------------------------
      year = ncdate/10000
      mnth = mod( ncdate,10000 )/100
      if( mnth < 1 .or. mnth > 12) then
         write(*,*) 'days0: Bad month index = ', mnth
         call endrun
      end if
      mday = mod(ncdate,100)
      doy  = jdcon(mnth) + mday + ncsec/86400.

      if( doy < 1. .or. doy > 366. ) then
         write(*,*) 'days0: bad day of year = ',doy
         call endrun
      end if

      days0 = 365.*year + doy

      end function days0

      end module mo_srf_emis
