
      module mo_sulf
!---------------------------------------------------------------
!	... annual cycle for sulfur
!---------------------------------------------------------------

      implicit none

      private
      public  :: sulf_inti
      public  :: read_sulf
      public  :: sulf_interp

      save

      integer :: gndx     = 0
      integer :: ncid     = 0
      integer :: time_cnt = 0
      integer :: lotim    = 0
      integer :: hitim    = 0
      integer :: loin     = 1
      integer :: hiin     = 2
      integer :: sulf_nlevs
      integer :: nlon
      integer :: nlat
      integer :: jlim_in(2)
      real    :: sulf_P0
      real, allocatable :: times(:)
      real, allocatable :: psi(:,:,:,:)
      real, allocatable :: sulfatei(:,:,:,:,:)
      real, allocatable :: sulf_hyam(:)
      real, allocatable :: sulf_hybm(:)
      real, allocatable :: psi_in(:,:)
      real, allocatable :: sulfatei_in(:,:,:)
      real, allocatable :: wrk_in(:,:)
      logical :: cyclical = .true.

      contains 

      subroutine sulf_inti( plonl, platl, pplon, ncfile, lpath, mspath )
!-----------------------------------------------------------------------
! 	... Open netCDF file containing annual sulfur data.  Initialize
!           arrays with the data to be interpolated to the current time.
!
!           It is assumed that the time coordinate is increasing
!           and represents calendar days; range = [1.,366.).
!-----------------------------------------------------------------------

      use netcdf
      use mo_regrider,   only : regrid_inti, regrid_lat_limits, regrid_diagnostics
      use mo_constants,  only : phi, lam, d2r
      use mo_file_utils, only : open_netcdf_file
      use mo_mpi,        only : base_lat, masternode
      use mo_grid,       only : platg=>plat, plong=>plon

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, &
	platl, &
	pplon
      character(len=*), intent(in) :: &
        ncfile, &            ! file name of netCDF file containing data
        lpath, &             ! local pathname to ncfile
        mspath               ! mass store pathname to ncfile

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: &
        ierr, &     ! return code
        times_id, &
        vid
      integer :: dimid_lat, dimid_lon, dimid_lev
      real, allocatable :: lat(:),lon(:)
      character(len=128) ::  recname
      logical :: there

!-----------------------------------------------------------------------
!     	... Open netcdf file
!-----------------------------------------------------------------------
      ncid = open_netcdf_file( ncfile, lpath, mspath )

!-----------------------------------------------------------------------
!     	... Inquire about file
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_unlimdim( ncid, times_id ), &
                         'sulf_inti: File inquiry' )
      call handle_ncerr( nf_inq_dimlen( ncid, times_id, time_cnt ), &
                         'sulf_inti: Getting time point count' )

!-----------------------------------------------------------------------
!     	... Allocate space for time coordinate data
!-----------------------------------------------------------------------
      allocate( times(time_cnt), stat = ierr )
      if( ierr /= 0 ) then
	 write(*,*) ' sulf_inti : Failed to allocate times array; error = ',ierr
	 call endrun
      end if
!-----------------------------------------------------------------------
!     	... Get time coordinate
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_varid( ncid, 'time', vid ), 'sulf_inti: Getting time id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, times ), 'sulf_inti: Getting times' )
      if( masternode ) then
         write(*,*) 'sulf_inti: times' 
         write(*,'(1p,5g20.13)') times
      end if

!-----------------------------------------------------------------------
!     	... Get vertical coordinate
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lev', dimid_lev ), 'sulf_inti: Failed to find dimension lev' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lev, sulf_nlevs ), &
                         'sulf_inti: Failed to get length of dimension lev' )
      allocate( sulf_hyam(sulf_nlevs), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: hyam allocation error = ',ierr
         call endrun
      end if
      allocate( sulf_hybm(sulf_nlevs), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: hybm allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'hyam', vid ), 'sulf_inti: Getting hyam id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, sulf_hyam ), 'sulf_inti: Getting hyam' )
      call handle_ncerr( nf_inq_varid( ncid, 'hybm', vid ), 'sulf_inti: Getting hybm id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, sulf_hybm ), 'sulf_inti: Getting hybm' )
      call handle_ncerr( nf_inq_varid( ncid, 'P0', vid ), 'sulf_inti: Getting P0 id' )
      call handle_ncerr( nf_get_var_double( ncid, vid, sulf_P0 ), 'sulf_inti: Getting P0' )
      if( masternode ) then
         write(*,*) 'sulf_inti: P0 = ',sulf_P0
      end if

!-----------------------------------------------------------------------
!     	... Make sure the time coordinate looks like calendar day, and is increasing.
!-----------------------------------------------------------------------
      call moz_chktime( times, time_cnt )

!-----------------------------------------------------------------------
!     	... Get latitude and longitude
!-----------------------------------------------------------------------
      call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid_lat ), &
                         'sulf_inti: Failed to find dimension lat' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lat, nlat ), &
                         'sulf_inti: Failed to get length of dimension lat' )
      allocate( lat(nlat), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: lat allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lat', vid ), &
                         'sulf_inti: Failed to find variable lat' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lat ), &
                         'sulf_inti: Failed to read variable lat' )
      lat(:nlat) = lat(:nlat) * d2r
 
      call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid_lon ), &
                         'sulf_inti: Failed to find dimension lon' )
      call handle_ncerr( nf_inq_dimlen( ncid, dimid_lon, nlon ), &
                         'sulf_inti: Failed to get length of dimension lon' )
      allocate( lon(nlon), stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: lon allocation error = ',ierr
         call endrun
      end if
      call handle_ncerr( nf_inq_varid( ncid, 'lon', vid ), &
                         'sulf_inti: Failed to find variable lon' )
      call handle_ncerr( nf_get_var_double( ncid, vid, lon ), &
                         'sulf_inti: Failed to read variable lon' )
      lon(:nlon) = lon(:nlon) * d2r

!-----------------------------------------------------------------------
!     	... Get grid interp limits
!-----------------------------------------------------------------------
      gndx = regrid_inti( nlat, platg, &
                          nlon, plong, &
                          lon,  lam, &
                          lat,  phi, &
                          0, platl )
      deallocate( lat, lon, stat=ierr )
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: Failed to deallocate lat,lon; ierr = ',ierr
         call endrun
      end if
      if( gndx /= 0 ) then
	 write(*,*) 'sulf_inti: non-xparent transform'
	 call regrid_diagnostics( gndx )
         jlim_in = regrid_lat_limits( gndx )
      else
	 write(*,*) 'sulf_inti: xparent transform'
         jlim_in = (/ base_lat+1, base_lat+platl /)
      end if

      write(*,'(''sulf_inti: gndx='',i2,'', grid limits = '',2i4,'', jl,ju='',2i4)') &
         gndx, jlim_in, base_lat+1, base_lat+platl

      allocate( psi_in( nlon,jlim_in(1):jlim_in(2)), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: psi_in allocation error = ',ierr
         call endrun
      end if
      allocate( sulfatei_in( nlon,sulf_nlevs,jlim_in(1):jlim_in(2)), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: sulfatei_in allocation error = ',ierr
         call endrun
      end if
      allocate( wrk_in( nlon,jlim_in(1):jlim_in(2)), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: wrk_in allocation error = ',ierr
         call endrun
      end if

      allocate( psi( plonl,platl,pplon,2 ), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: psi allocation error = ',ierr
         call endrun
      end if
      allocate( sulfatei( plonl,sulf_nlevs,platl,pplon,2 ), stat=ierr)
      if( ierr /= 0 ) then
         write(*,*) 'sulf_inti: sulfatei allocation error = ',ierr
         call endrun
      end if

      end subroutine sulf_inti

      subroutine read_sulf( plonl, platl, pplon, calday )
!-----------------------------------------------------------------------
! 	... Interpolate annual cycle sulfur data to the current time.
!           Read in new time samples of the input data as necessary.
!-----------------------------------------------------------------------

      use netcdf
      use mo_regrider, only : regrid_2d
      use mo_mpi,      only : base_lat, masternode
      use mo_grid,     only : plon

      implicit none

!-----------------------------------------------------------------------
!	... dummy args
!-----------------------------------------------------------------------
      integer, intent(in) :: &
	plonl, &
	platl, &
	pplon
      real, intent(in) :: &
        calday  ! current time in calendar days + fraction.

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      integer :: &
        i, ip, k, jl, ju, &
        oldlotim, oldhitim, &
        vid, ret
      integer, dimension(4) :: start, count
      real :: &
        cday
      real :: wrk(plon,platl)

      cday = calday  ! convert calday to single precision if necessary

!-----------------------------------------------------------------------
!     	... check to see if model time is still bounded by dataset times.
!-----------------------------------------------------------------------
      oldlotim = lotim
      oldhitim = hitim
      call moz_findplb( times, time_cnt, cday, lotim )
      if( cyclical ) then
         hitim = mod( lotim,time_cnt ) + 1
      else
         hitim = lotim + 1
      end if

      if( hitim /= oldhitim ) then
!-----------------------------------------------------------------------
! 	... read in new hitim data; replace old lotim data
!-----------------------------------------------------------------------
	 jl   = base_lat + 1
	 ju   = base_lat + platl
         loin = hiin
         hiin = mod( loin,2 ) + 1
         start(:3) = (/ 1, jlim_in(1), hitim /)
         count(:3) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, 1 /)
         call handle_ncerr( nf_inq_varid( ncid, 'PS', vid ), &
                            'read_sulf: Getting PS id' )
         call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, psi_in ), &
                            'read_sulf: Getting PS' )
!-----------------------------------------------------------------------
! 	... regrid maccm input surface presssure
!-----------------------------------------------------------------------
         call regrid_2d( psi_in, wrk, gndx, jl, ju, do_poles=.true. )
	 psi(:,:,:,hiin) = reshape( wrk, (/ plonl,platl,pplon/), order = (/1,3,2/) )
         start(:4) = (/ 1, 1, jlim_in(1), hitim /)
         count(:4) = (/ nlon, sulf_nlevs, jlim_in(2) - jlim_in(1) + 1, 1 /)
         call handle_ncerr( nf_inq_varid( ncid, 'SULFATE', vid ), &
                            'read_sulf: Getting SULFATE id' )
         call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, sulfatei_in ), &
                            'read_sulf: Getting SULFATE' )
!-----------------------------------------------------------------------
! 	... regrid maccm input sulfate
!-----------------------------------------------------------------------
	 do k = 1,sulf_nlevs
	    wrk_in(:nlon,jlim_in(1):jlim_in(2)) = sulfatei_in(:nlon,k,jlim_in(1):jlim_in(2))
            call regrid_2d( wrk_in, wrk, gndx, jl, ju, do_poles=.true. )
#ifdef DEBUG
            if( k == sulf_nlevs ) then
               write(*,*) 'read_sulf: wrk_in :'
               write(*,'(1p,10e8.1)') wrk_in(1,jlim_in(1):jlim_in(2))
               write(*,*) 'read_sulf: wrk_out :'
               write(*,'(1p,10e8.1)') wrk(1,1:platl)
            end if
#endif
	    do ip = 1,pplon
	       sulfatei(:plonl,k,:platl,ip,hiin) = wrk((ip-1)*plonl+1:ip*plonl,:platl)
	    end do
	 end do
         if( masternode ) then
            write(*,*) ' '
            write(*,*) 'read_sulf: read data for day ',times(hitim)
         end if
         if( lotim /= oldhitim ) then
!-----------------------------------------------------------------------
!   	... read in new lotim data.  Replace old hitim data.
!-----------------------------------------------------------------------
            start(:3) = (/ 1, jlim_in(1), lotim /)
            count(:3) = (/ nlon, jlim_in(2) - jlim_in(1) + 1, 1 /)
            call handle_ncerr( nf_inq_varid( ncid, 'PS', vid ), 'read_sulf: Getting PS id' )
            call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, psi_in ), 'read_sulf: Getting PS' )
!-----------------------------------------------------------------------
! 	... regrid maccm input surface presssure
!-----------------------------------------------------------------------
            call regrid_2d( psi_in, wrk, gndx, jl, ju, do_poles=.true. )
	    psi(:,:,:,loin) = reshape( wrk, (/ plonl,platl,pplon/), order = (/1,3,2/) )
            start(:4) = (/ 1, 1, jlim_in(1), lotim /)
            count(:4) = (/ nlon, sulf_nlevs, jlim_in(2) - jlim_in(1) + 1, 1 /)
            call handle_ncerr( nf_inq_varid( ncid, 'SULFATE', vid ), &
                               'read_sulf: Getting SULFATE id' )
            call handle_ncerr( nf_get_vara_double( ncid, vid, start, count, sulfatei_in ), &
                               'read_sulf: Getting SULFATE' )
!-----------------------------------------------------------------------
! 	... regrid maccm input sulfate
!-----------------------------------------------------------------------
	    do k = 1,sulf_nlevs
	       wrk_in(:nlon,jlim_in(1):jlim_in(2)) = sulfatei_in(:nlon,k,jlim_in(1):jlim_in(2))
               call regrid_2d( wrk_in, wrk, gndx, jl, ju, do_poles=.true. )
	       do ip = 1,pplon
	          sulfatei(:plonl,k,:platl,ip,loin) = wrk((ip-1)*plonl+1:ip*plonl,:platl)
	       end do
	    end do
            if( masternode ) then
               write(*,*) 'read_sulf: read data for day ',times(lotim)
            end if
         end if
      end if

      end subroutine read_sulf

      subroutine sulf_interp( lat, ip, pmid, calday, ccm_sulf, plonl )
!-----------------------------------------------------------------------
! 	... Time interpolate sulfatei to current time
!-----------------------------------------------------------------------

      use mo_grid, only : plev

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) :: lat                 ! latitude index
      integer, intent(in) :: ip                  ! longitude tile index
      integer, intent(in) :: plonl
      real, intent(in)    :: calday              ! day of year
      real, intent(in)    :: pmid(plonl,plev)    ! midpoint pressure ( pascals )
      real, intent(out)   :: ccm_sulf(plonl,plev)    ! output sulfate

!-----------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------
      real    ::  tint, dt, dt1      ! interpolation time
      real    ::  delp               ! pressure delta
      real    ::  pinterp            ! interpolation pressure
      real    ::  sulf_press(plonl,sulf_nlevs,2)
      real    ::  zint_sulf(plonl,plev,2)
      integer ::  i, k, ku, kl      ! working indicies

!-----------------------------------------------------------------------
! 	... form MOZ1 pressures at enclosing months
!-----------------------------------------------------------------------
      do k = 1,sulf_nlevs
	 sulf_press(:,k,loin) = sulf_hyam(k)*sulf_P0 + sulf_hybm(k)*psi(:,lat,ip,loin)
	 sulf_press(:,k,hiin) = sulf_hyam(k)*sulf_P0 + sulf_hybm(k)*psi(:,lat,ip,hiin)
      end do

!-----------------------------------------------------------------------
! 	... vertical interpolation of images data to model levels
!	    Note: images data only up to 50mb
!-----------------------------------------------------------------------
      do i = 1,plonl
	 do k = 1,plev
	    pinterp = pmid(i,k)
	    if( pinterp < maxval( sulf_press(i,1,:2) ) ) then
	       zint_sulf(i,k,loin) = 0.
	       zint_sulf(i,k,hiin) = 0.
	    else
	       if( pinterp > sulf_press(i,sulf_nlevs,loin) ) then
		  zint_sulf(i,k,loin) = sulfatei(i,sulf_nlevs,lat,ip,loin)
	       else
	          do ku = 2,sulf_nlevs
		     if( pinterp <= sulf_press(i,ku,loin) ) then
			kl = ku - 1
			delp =  log( pinterp/sulf_press(i,kl,loin) ) &
                             /  log( sulf_press(i,ku,loin)/sulf_press(i,kl,loin) )
			zint_sulf(i,k,loin) = sulfatei(i,kl,lat,ip,loin) &
                             + delp * (sulfatei(i,ku,lat,ip,loin) - sulfatei(i,kl,lat,ip,loin))
		        exit
		     end if
	          end do
	       end if
	       if( pinterp > sulf_press(i,sulf_nlevs,hiin) ) then
		  zint_sulf(i,k,hiin) = sulfatei(i,sulf_nlevs,lat,ip,hiin)
		  cycle
	       else
	          do ku = 2,sulf_nlevs
		     if( pinterp <= sulf_press(i,ku,hiin) ) then
			kl = ku - 1
			delp = log( pinterp/sulf_press(i,kl,hiin) ) &
                             / log( sulf_press(i,ku,hiin)/sulf_press(i,kl,hiin) )
			zint_sulf(i,k,hiin) = sulfatei(i,kl,lat,ip,hiin) &
                             + delp * (sulfatei(i,ku,lat,ip,hiin) - sulfatei(i,kl,lat,ip,hiin))
		        exit
		     end if
	          end do
	       end if
	    end if
	 end do
      end do

!-----------------------------------------------------------------------
!     	... Linear interpolation     
!           Start by computing the number of days between
!           the lower and upper bounds, and days between
!           the model time and lower bound.
!-----------------------------------------------------------------------
      if( times(hitim) < times(lotim) ) then
         dt = 365. - times(lotim) + times(hitim)
         if( calday <= times(hitim) )then
            dt1 = 365. - times(lotim) + calday
         else
            dt1 = calday - times(lotim)
         end if
      else
         dt = times(hitim) - times(lotim)
         dt1 = calday - times(lotim)
      end if

      tint = dt1 / dt
      call moz_linintp( plonl*plev, 0., 1., tint, &
                        zint_sulf(1,1,loin), &
                        zint_sulf(1,1,hiin), &
                        ccm_sulf )

      end subroutine sulf_interp

      end module mo_sulf
