
      module mo_synoz
!--------------------------------------------------------------------
!	... stratospheric ozone emission source
!--------------------------------------------------------------------

      implicit none

      save 

      real, allocatable :: po3(:,:,:,:)

      private
      public :: synoz_inti
      public :: po3

      contains

      subroutine synoz_inti( plonl, platl, pplon )
!-----------------------------------------------------------------------
! 	... initialize synoz emissions
!	    note: the emissions are in in units of molecules/cm**3/s
!-----------------------------------------------------------------------

      use mo_mpi
      use mo_constants,  only : phi, lam, r2d, pi, rearth, latwts
      use mo_grid,       only : plon, plat, plev, plevp
      use plevs,         only : ps0, hyam, hybm,hyai,hybi
      use chem_mods,     only : adv_mass
      use mo_chem_utls,  only : get_spc_ndx

      implicit none

!-----------------------------------------------------------------------
! 	... dummy arguments
!-----------------------------------------------------------------------
      integer, intent(in) ::   plonl              ! lon tile dim
      integer, intent(in) ::   platl              ! lat tile dim
      integer, intent(in) ::   pplon              ! lon tile count

!-----------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------
      real, parameter :: latmin                = -30.
      real, parameter :: latmax                = 30.
      real, parameter :: prsmin                = 1000.
      real, parameter :: prsmax                = 7000.
      real, parameter :: ps                    = 100000.
      real, parameter :: grav                  = 9.81
      real, parameter :: ozone_source_per_year = 500.     ! global/annual average of ozone source (Tg/yr)
      real, parameter :: dry_mass              = 28.966
      real, parameter :: tg2kg                 = 1.e9
      real, parameter :: seconds_in_day        = 86400.
      real, parameter :: days_per_year         = 365.

      integer :: i, j, k, jl, ju
      integer :: jmin, jmax, kmin, kmax
      integer :: synoz_ndx
      integer :: astat
      real    :: diff, diff_min, diff_max
      real    :: total_mass = 0.
      real    :: seq
      real    :: sf(plat)
      real, allocatable :: lat(:)
      real, allocatable :: prs(:)
      real, allocatable :: dp(:)
      real, allocatable :: wk(:,:,:)

!-----------------------------------------------------------------------
! 	... allocate memory
!-----------------------------------------------------------------------
      allocate( po3(plonl,plev,platl,pplon), &
                lat(plat), &
                prs(plev), &
                dp(plev), &
                wk(plon,plat,plev), stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'synoz_inti: failed to allocate po3 ... wk; error = ',astat
         call endrun
      end if

!-----------------------------------------------------------------------
! 	... change units of lat to degrees
!-----------------------------------------------------------------------
      lat = phi * r2d

!-----------------------------------------------------------------------
! 	... find indices of the latitudinal box
!-----------------------------------------------------------------------
      jmin = -99
      jmax = -99
      diff_min = 1.e20
      diff_max = 1.e20
      do j = 1,plat
        diff = abs(lat(j)-latmin)
        if( diff < diff_min ) then
          diff_min = diff
          jmin = j
        end if
        diff = abs(lat(j)-latmax)
        if( diff < diff_max ) then
          diff_max = diff
          jmax = j
        end if
      end do

!-----------------------------------------------------------------------
! 	... make sure we found them
!-----------------------------------------------------------------------
      if( jmin < 0 .or. jmax < 0 ) then
        write(*,*) 'synoz_inti: problem finding the min/max lat in synoz_inti',jmin,jmax
        call endrun
      end if
!-----------------------------------------------------------------------
! 	... define pressure arrays, assuming surface pressure = 1000 hPa
!-----------------------------------------------------------------------
      prs(:) = hyam(:) * ps0 + hybm(:) * ps
      dp(:)  = (hyai(2:plevp) - hyai(1:plev)) * ps0 + (hybi(2:plevp) - hybi(1:plev)) * ps

!-----------------------------------------------------------------------
! 	... find indices of the pressure box
!-----------------------------------------------------------------------
      kmin = -99
      kmax = -99
      diff_min = 1.e20
      diff_max = 1.e20
      do k = 1,plev
        diff = abs( prs(k) - prsmin )
        if( diff < diff_min ) then
          diff_min = diff
          kmin = k
        end if
        diff = abs( prs(k) - prsmax )
        if( diff < diff_max ) then
          diff_max = diff
          kmax = k
        end if
      end do

!-----------------------------------------------------------------------
! 	... make sure we found them
!-----------------------------------------------------------------------
      if( kmin < 0 .or. kmax < 0 ) then
        write(*,*) 'synoz_inti: problem finding the min/max prs in synoz_inti',kmin,kmax
        call endrun
      end if

!-----------------------------------------------------------------------
! 	... define geometric factors (in SI)
!-----------------------------------------------------------------------
      seq = 2.*pi*rearth**2/real(plon)
      do j = 1,plat
         sf(j) = seq*latwts(j)
      end do

!-----------------------------------------------------------------------
! 	... find index of synoz
!-----------------------------------------------------------------------
      synoz_ndx = get_spc_ndx('SYNOZ')
has_synoz : &
      if( synoz_ndx > 0 ) then
!-----------------------------------------------------------------------
! 	... compute total mass (in kg) over the domain for which
!           the ozone source will be defined
!-----------------------------------------------------------------------
         total_mass = 0.
         do k = kmin,kmax
           do j = jmin,jmax
             total_mass = total_mass + dp(k)/grav * sf(j)
           end do
         end do
         total_mass = total_mass * plon
#ifdef DEBUG
         write(*,*)'synoz_inti: total mass = ',total_mass
#endif
!-----------------------------------------------------------------------
! 	... define the location of the ozone source
!-----------------------------------------------------------------------
         wk(:,:,:) = 0.
         do k = kmin,kmax
           do j = jmin,jmax
             wk(1:plon,j,k) = 1.
           end do
         end do
!-----------------------------------------------------------------------
! 	... define the ozone source as vmr/s (what is needed for the chemistry solver)
!           note : a change in chemdr is made to avoid the division by invariants
!-----------------------------------------------------------------------
         wk(:,:,:) = wk(:,:,:) * (ozone_source_per_year*tg2kg/total_mass) &
                               / (seconds_in_day*days_per_year) &
                               * (dry_mass/adv_mass(synoz_ndx))
         write(*,*) 'synoz_inti: max wk = ',maxval( wk(:,:,:) )
!-----------------------------------------------------------------------
! 	... reshape the wk array according to lat-lon blocks
!-----------------------------------------------------------------------
         jl = base_lat + 1
         ju = base_lat + platl
         do k = 1,plev
           po3(:,k,:,:) = reshape( wk(:,jl:ju,k), (/plonl,platl,pplon/), order=(/1,3,2/) )
         end do
      end if has_synoz

!-----------------------------------------------------------------------
! 	... deallocate memory
!-----------------------------------------------------------------------
      deallocate( lat, prs, dp, wk )

      end subroutine synoz_inti

      end module mo_synoz
