
      module mo_tuv_inti

      implicit none

      private
      public :: nlng, nzen, ncof
      public :: tuv_inti

      save

      integer :: nlng
      integer :: nzen, ncof

      contains

      subroutine tuv_inti( nz, lng_indexer )
!-----------------------------------------------------------------------------
!   purpose:
!   read 17 bins data outputed from tuv
!-----------------------------------------------------------------------------
!   parameters:
!   nw     - integer, number of specified intervals + 1 in working
!            wavelength grid
!   wl     - real, vector of lower limits of wavelength intervals in
!            working wavelength grid
!   wc     - real, vector of center  of wavelength intervals in
!            working wavelength grid
!   wu     - real, vector of upper limits of wavelength intervals in
!            working wavelength grid
!   f      - real, spectral irradiance at the top of the atmosphere at
!            each specified wavelength
!-----------------------------------------------------------------------------
!   edit history:        
!   10/2000  similified by xuexi
!-----------------------------------------------------------------------------

      use netcdf
      use mo_file_utils, only : open_netcdf_file
      use mo_mpi
      use mo_params,     only : kj, kw, smallest, largest
      use mo_waveall,    only : r01g1, r01g2, r01g3, r01g4, &
                                r04g, r08g, r06g1, r06g2, &
                                r10g1, r10g2, r10g3, r10g4, r10g5, &
                                r11g, r11g1, r11g2, r11g3, r11g4, &
                                r14g, r14g1, r14g2, &
                                r15g, r15g1, r15g2, r15g3, &
                                r17g, r17g1, &
                                r18g, r18g2
      use mo_wavelab,    only : sj
      use mo_wavelen,    only : nw, deltaw, delw_bin, sflx, wc, wl, wu
      use mo_waveo3,     only : xso3, s226, s263, s298
      use mo_zadj,       only : adj_coeffs
      use mo_schu,       only : schu_inti
      use mo_xsections,  only : r44_inti, r08_inti
      use chem_mods,     only : phtcnt, pht_alias_lst, rxt_tag_lst
      use mo_control,    only : photo_xs_short_flsp

      implicit none
     
!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in)    :: nz
      integer, intent(inout) :: lng_indexer(phtcnt)

!-----------------------------------------------------------------------------
!	... local variables
!-----------------------------------------------------------------------------
      integer :: ncid
      integer :: ndx
      integer :: dimid, vid
      integer :: iw, ios, iret
      integer :: k, m
      integer :: ind_wrk(4)
      integer :: wrk_ndx(phtcnt)
      real, allocatable  :: coeff_adj(:,:)
      character(len=40)  :: note
      character(len=20)  :: coeff_tag

!------------------------------------------------------------------------
!     for wl(iw) .lt. 150.01                                susim_hi.flx
!     for wl(iw) .ge. 150.01 and wl(iw) .le. 400            atlas3.flx 
!     for wl(iw) .gt. 400                                   neckel & labs 
!------------------------------------------------------------------------

!------------------------------------------------------------------------
!     input data files including:
!           (0) wavelength nw,wl,wc,wu
!           (1) solar flux
!           (2) o3 cross sections
!           (3) other cross 
!           (4) t dependence parameter of cross section
!------------------------------------------------------------------------

!---------------------------------------------------------------------------
!       ... open netcdf file
!---------------------------------------------------------------------------
master_only : &
      if( masternode ) then
         ncid = open_netcdf_file( photo_xs_short_flsp%nl_filename, &
                                  photo_xs_short_flsp%local_path, &
                                  photo_xs_short_flsp%remote_path, masteronly=.true. )
!---------------------------------------------------------------------------
! 	... get the dimensions
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'nw', dimid ), 'tuv_inti: dimension nw not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nw ), 'tuv_inti: failed to read nw' )
         call handle_ncerr( nf_inq_dimid( ncid, 'nzen', dimid ), 'tuv_inti: dimension nzen not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nzen ), 'tuv_inti: failed to read nzen' )
         call handle_ncerr( nf_inq_dimid( ncid, 'ncof', dimid ), 'tuv_inti: dimension ncof not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, ncof ), 'tuv_inti: failed to read ncof' )
         write(*,*) ' '
         write(*,*) 'tuv_inti: nw, nzen, ncof = ',nw,nzen,ncof
!---------------------------------------------------------------------------
! 	... read the wave bin coordinates
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'wl', vid ), 'tuv_inti: wl not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, wl(1:nw) ), 'tuv_inti: getting wl' )
         call handle_ncerr( nf_inq_varid( ncid, 'wc', vid ), 'tuv_inti: wc not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, wc(1:nw) ), 'tuv_inti: getting wc' )
         call handle_ncerr( nf_inq_varid( ncid, 'wu', vid), 'tuv_inti: wu not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, wu(1:nw) ), 'tuv_inti: getting wu' )
         wl(nw+1) = wu(nw)
         write(*,*) ' '
         write(*,*) 'tuv_inti: wl(nw+1) = ',wl(nw+1)
!------------------------------------------------------------------------
!  	... solar flux
!------------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'sflx', vid ), 'tuv_inti: sflx not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, sflx(1:nw) ), 'tuv_inti: reading sflx' )
!------------------------------------------------------------------------
!    	... o3 cross (t dependence)
!------------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'xso3', vid), 'tuv_inti: xso3 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, xso3(1:nw) ), 'tuv_inti: reading xso3' )
         call handle_ncerr( nf_inq_varid( ncid, 's226', vid), 'tuv_inti: s226 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, s226(1:nw) ), 'tuv_inti: reading s226' )
         call handle_ncerr( nf_inq_varid( ncid, 's263', vid), 'tuv_inti: s263 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, s263(1:nw) ), 'tuv_inti: reading s263' )
         call handle_ncerr( nf_inq_varid( ncid, 's298', vid), 'tuv_inti: s298 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, s298(1:nw) ), 'tuv_inti: reading s298' )
!---------------------------------------------------------------------------
!   	... temperature dependent cross section parameters
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'r01g1', vid), 'tuv_inti: r01g1 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r01g1(1:nw) ), 'tuv_inti: reading r01g1' )
         call handle_ncerr( nf_inq_varid( ncid, 'r01g2', vid), 'tuv_inti: r01g2 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r01g2(1:nw) ), 'tuv_inti: reading r01g2' )
         call handle_ncerr( nf_inq_varid( ncid, 'r01g3', vid), 'tuv_inti: r01g3 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r01g3(1:nw) ), 'tuv_inti: reading r01g3' )
         call handle_ncerr( nf_inq_varid( ncid, 'r01g4', vid), 'tuv_inti: r01g4 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r01g4(1:nw) ), 'tuv_inti: reading r01g4' )

         call handle_ncerr( nf_inq_varid( ncid, 'r04g', vid), 'tuv_inti: r04g not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r04g(1:nw) ), 'tuv_inti: reading r04g' )

         call handle_ncerr( nf_inq_varid( ncid, 'r08g', vid), 'tuv_inti: r08g not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r08g(1:nw) ), 'tuv_inti: reading r08g' )

         call handle_ncerr( nf_inq_varid( ncid, 'r06g1', vid), 'tuv_inti: r06g1 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r06g1(1:nw) ), 'tuv_inti: reading r06g1' )
         call handle_ncerr( nf_inq_varid( ncid, 'r06g2', vid), 'tuv_inti: r06g2 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r06g2(1:nw) ), 'tuv_inti: reading r06g2' )

         call handle_ncerr( nf_inq_varid( ncid, 'r10g1', vid), 'tuv_inti: r10g1 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r10g1(1:nw) ), 'tuv_inti: reading r10g1' )
         call handle_ncerr( nf_inq_varid( ncid, 'r10g2', vid), 'tuv_inti: r10g2 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r10g2(1:nw) ), 'tuv_inti: reading r10g2' )
         call handle_ncerr( nf_inq_varid( ncid, 'r10g3', vid), 'tuv_inti: r10g3 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r10g3(1:nw) ), 'tuv_inti: reading r10g3' )
         call handle_ncerr( nf_inq_varid( ncid, 'r10g4', vid), 'tuv_inti: r10g4 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r10g4(1:nw) ), 'tuv_inti: reading r10g4' )
         call handle_ncerr( nf_inq_varid( ncid, 'r10g5', vid), 'tuv_inti: r10g5 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r10g5(1:nw) ), 'tuv_inti: reading r10g5' )

         call handle_ncerr( nf_inq_varid( ncid, 'r11g', vid), 'tuv_inti: r11g not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r11g(1:nw) ), 'tuv_inti: reading r11g' )
         call handle_ncerr( nf_inq_varid( ncid, 'r11g1', vid), 'tuv_inti: r11g1 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r11g1(1:nw) ), 'tuv_inti: reading r11g1' )
         call handle_ncerr( nf_inq_varid( ncid, 'r11g2', vid), 'tuv_inti: r11g2 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r11g2(1:nw) ), 'tuv_inti: reading r11g2' )
         call handle_ncerr( nf_inq_varid( ncid, 'r11g3', vid), 'tuv_inti: r11g3 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r11g3(1:nw) ), 'tuv_inti: reading r11g3' )
         call handle_ncerr( nf_inq_varid( ncid, 'r11g4', vid), 'tuv_inti: r11g4 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r11g4(1:nw) ), 'tuv_inti: reading r11g4' )

         call handle_ncerr( nf_inq_varid( ncid, 'r14g', vid), 'tuv_inti: r14g not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r14g(1:nw) ), 'tuv_inti: reading r14g' )
         call handle_ncerr( nf_inq_varid( ncid, 'r14g1', vid), 'tuv_inti: r14g1 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r14g1(1:nw) ), 'tuv_inti: reading r14g1' )
         call handle_ncerr( nf_inq_varid( ncid, 'r14g2', vid), 'tuv_inti: r14g2 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r14g2(1:nw) ), 'tuv_inti: reading r14g2' )

         call handle_ncerr( nf_inq_varid( ncid, 'r15g', vid), 'tuv_inti: r15g not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r15g(1:nw) ), 'tuv_inti: reading r15g' )
         call handle_ncerr( nf_inq_varid( ncid, 'r15g1', vid), 'tuv_inti: r15g1 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r15g1(1:nw) ), 'tuv_inti: reading r15g1' )
         call handle_ncerr( nf_inq_varid( ncid, 'r15g2', vid), 'tuv_inti: r15g2 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r15g2(1:nw) ), 'tuv_inti: reading r15g2' )
         call handle_ncerr( nf_inq_varid( ncid, 'r15g3', vid), 'tuv_inti: r15g3 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r15g3(1:nw) ), 'tuv_inti: reading r15g3' )

         call handle_ncerr( nf_inq_varid( ncid, 'r17g', vid), 'tuv_inti: r17g not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r17g(1:nw) ), 'tuv_inti: reading r17g' )
         call handle_ncerr( nf_inq_varid( ncid, 'r17g1', vid), 'tuv_inti: r17g1 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r17g1(1:nw) ), 'tuv_inti: reading r17g1' )

         call handle_ncerr( nf_inq_varid( ncid, 'r18g', vid), 'tuv_inti: r18g not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r18g(1:nw) ), 'tuv_inti: reading r18g' )
         call handle_ncerr( nf_inq_varid( ncid, 'r18g2', vid), 'tuv_inti: r18g2 not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, r18g2(1:nw) ), 'tuv_inti: reading r18g2' )
!------------------------------------------------------------------------------
!       ... check for cross section in dataset
!------------------------------------------------------------------------------
         do m = 1,phtcnt
            if( pht_alias_lst(m,2) == ' ' ) then
               iret = nf_inq_varid( ncid, rxt_tag_lst(m), vid )
               if( iret == nf_noerr ) then 
                  lng_indexer(m) = vid
               end if
            else if( pht_alias_lst(m,2) == 'userdefined' ) then
               lng_indexer(m) = -1
            else
               iret = nf_inq_varid( ncid, pht_alias_lst(m,2), vid )
               if( iret == nf_noerr ) then 
                  lng_indexer(m) = vid
               else
	          write(*,*) 'tuv_inti : ',rxt_tag_lst(m)(:len_trim(rxt_tag_lst(m))),' alias ', &
                             pht_alias_lst(m,2)(:len_trim(pht_alias_lst(m,2))),' not in dataset'            
                  call endrun
               end if
            end if
         end do
         nlng = 0
         do m = 1,phtcnt
            if( lng_indexer(m) > 0 ) then
               if( any( lng_indexer(:m-1) == lng_indexer(m) ) ) then
                  cycle
               end if
               nlng = nlng + 1
            end if 
         end do
!---------------------------------------------------------------------------
!	... allocate the cross section array
!---------------------------------------------------------------------------
         allocate( sj(nw,nz,nlng), adj_coeffs(ncof,nlng,nzen), coeff_adj(ncof,nzen), stat=ios )
         if( ios /= 0 ) then
            write(*,*) 'tuv_inti: failed to allocate sj ... coeff_adj; error = ',ios
            call endrun
         end if
         sj(:,:,:)         = 0.
         adj_coeffs(:,:,:) = 0.
         write(*,*) 'tuv_inti: nlng = ',nlng
         write(*,*) 'tuv_inti: lng_indexer'
         write(*,'(10i5)') lng_indexer(:)
         if( nlng > 0 ) then
            write(*,*) ' '
            write(*,*) 'tuv_inti: photo xsect analysis'
            do m = 1,phtcnt
               if( lng_indexer(m) > 0 ) then
                  write(*,*) trim(rxt_tag_lst(m)),lng_indexer(m)
               end if
            end do
         end if
!---------------------------------------------------------------------------
!   	... read the raw cross sections
!   	... and adjustment coefficients
!---------------------------------------------------------------------------
!        jnames(1:4) = (/ 'jo2             ', 'jo1d', 'jo3p', 'jno2' /)
!        jnames(6)   = 'jno3            '
!        jnames(8:9) = (/ 'jn2o5           ', 'jn2o' /)
!        jnames(11)  = 'jh2o2'
!        jnames(13:16) = (/ 'jhno3           ', 'jho2no2', 'jch2o_a', 'jch2o_b' /)
!        jnames(17:19) = (/ 'jch3cho_a       ', 'jch3cho_b', 'jch3cho_c' /)
!        jnames(22:24) = (/ 'jmgly           ', 'jacet',   'jch3ooh' /)
!        jnames(26:29) = (/ 'jpan            ',  'jmvk',    'jmacr',   'jglyald' /)
!        jnames(31)    = 'jhyac              '
         ndx = 0
         do m = 1,phtcnt
            if( lng_indexer(m) > 0 ) then
               if( any( lng_indexer(:m-1) == lng_indexer(m) ) ) then
                  cycle
               end if
               ndx = ndx + 1
               note = 'tuv_inti: failed to read ' // trim(rxt_tag_lst(m))
               call handle_ncerr( nf_get_var_double( ncid, lng_indexer(m), sj(1:nw,1,ndx) ), trim(note) )
               do k = 2,nz
	          sj(:,k,ndx) = sj(:,1,ndx)
               end do
               coeff_tag = trim(rxt_tag_lst(m)) // '_adj'
               note = 'tuv_inti: ' // trim(coeff_tag) // ' not found in input file'
               call handle_ncerr( nf_inq_varid( ncid, trim(coeff_tag), vid), trim(note) )
               note = 'tuv_inti: failed to read ' // trim(coeff_tag)
               call handle_ncerr( nf_get_var_double( ncid, vid, coeff_adj ), trim(note) )
               adj_coeffs(:,ndx,1:nzen) = coeff_adj(:,1:nzen)
            end if
         end do
         if( ndx /= nlng ) then
	    write(*,*) 'tuv_inti : ndx count /= cross section count'
            call endrun
         end if
!------------------------------------------------------------------------------
!       ... setup final lng_indexer
!------------------------------------------------------------------------------
         ndx = 0
         wrk_ndx(:) = lng_indexer(:)
         do m = 1,phtcnt
            if( wrk_ndx(m) > 0 ) then
               ndx = ndx + 1
               k = wrk_ndx(m)
               where( wrk_ndx(:) == k )
                  lng_indexer(:) = ndx
                  wrk_ndx(:)     = -100000
               end where
            end if
         end do
         if( nlng > 0 ) then
            write(*,*) ' '
            write(*,*) 'tuv_inti: photo xsect analysis'
            do m = 1,phtcnt
               if( lng_indexer(m) > 0 ) then
                  write(*,*) trim(rxt_tag_lst(m)),lng_indexer(m)
               end if
            end do
         end if
!---------------------------------------------------------------------------
! 	... close netcdf file
!---------------------------------------------------------------------------
         note = 'tuv_inti: error closing ' // trim(photo_xs_short_flsp%nl_filename)
         call handle_ncerr( nf_close( ncid ), trim(note) )
         deallocate( coeff_adj )
#ifdef USE_MPI
         ind_wrk(:4) = (/ nw, nlng, ncof, nzen /)
#endif
      end if master_only

#ifdef USE_MPI
!------------------------------------------------------------------------------
!       ... bcast indicies
!------------------------------------------------------------------------------
      call mpi_bcast( ind_wrk, 4, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast nw ... nzen; error = ',iret
         call endrun
      end if
      call mpi_bcast( lng_indexer, phtcnt, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast lng_indexer; error = ',iret
         call endrun
      end if
      if( .not. masternode ) then
         nw   = ind_wrk(1)
         nlng = ind_wrk(2)
         ncof = ind_wrk(3)
         nzen = ind_wrk(4)
         allocate( sj(nw,nz,nlng), adj_coeffs(ncof,nlng,nzen), stat=ios )
         if( ios /= 0 ) then
            write(*,*) 'tuv_inti: failed to allocate sj,adj_coeffs; error = ',ios
            call endrun
         end if
      end if
!------------------------------------------------------------------------------
!       ... bcast arrays
!------------------------------------------------------------------------------
      call mpi_bcast( wl, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast wl; error = ',iret
         call endrun
      end if
      call mpi_bcast( wc, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast wc; error = ',iret
         call endrun
      end if
      call mpi_bcast( wu, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast wu; error = ',iret
         call endrun
      end if
      call mpi_bcast( sflx, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast sflx; error = ',iret
         call endrun
      end if
      call mpi_bcast( xso3, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast xso3; error = ',iret
         call endrun
      end if
      call mpi_bcast( s226, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast s226; error = ',iret
         call endrun
      end if
      call mpi_bcast( s263, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast s263; error = ',iret
         call endrun
      end if
      call mpi_bcast( s298, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast s298; error = ',iret
         call endrun
      end if
      call mpi_bcast( r01g1, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r01g1; error = ',iret
         call endrun
      end if
      call mpi_bcast( r01g2, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r01g2; error = ',iret
         call endrun
      end if
      call mpi_bcast( r01g3, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r01g3; error = ',iret
         call endrun
      end if
      call mpi_bcast( r01g4, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r01g4; error = ',iret
         call endrun
      end if
      call mpi_bcast( r04g, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r04g; error = ',iret
         call endrun
      end if
      call mpi_bcast( r08g, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r08g; error = ',iret
         call endrun
      end if
      call mpi_bcast( r06g1, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r06g1; error = ',iret
         call endrun
      end if
      call mpi_bcast( r06g2, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r06g2; error = ',iret
         call endrun
      end if
      call mpi_bcast( r10g1, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r10g1; error = ',iret
         call endrun
      end if
      call mpi_bcast( r10g2, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r10g2; error = ',iret
         call endrun
      end if
      call mpi_bcast( r10g3, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r10g3; error = ',iret
         call endrun
      end if
      call mpi_bcast( r10g4, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r10g4; error = ',iret
         call endrun
      end if
      call mpi_bcast( r10g5, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r10g5; error = ',iret
         call endrun
      end if
      call mpi_bcast( r11g, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r11g; error = ',iret
         call endrun
      end if
      call mpi_bcast( r11g1, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r11g1; error = ',iret
         call endrun
      end if
      call mpi_bcast( r11g2, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r11g2; error = ',iret
         call endrun
      end if
      call mpi_bcast( r11g3, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r11g3; error = ',iret
         call endrun
      end if
      call mpi_bcast( r11g4, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r11g4; error = ',iret
         call endrun
      end if
      call mpi_bcast( r14g, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r14g; error = ',iret
         call endrun
      end if
      call mpi_bcast( r14g1, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r14g1; error = ',iret
         call endrun
      end if
      call mpi_bcast( r14g2, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r14g2; error = ',iret
         call endrun
      end if
      call mpi_bcast( r15g, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r15g; error = ',iret
         call endrun
      end if
      call mpi_bcast( r15g1, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r15g1; error = ',iret
         call endrun
      end if
      call mpi_bcast( r15g2, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r15g2; error = ',iret
         call endrun
      end if
      call mpi_bcast( r15g3, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r15g3; error = ',iret
         call endrun
      end if
      call mpi_bcast( r17g, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r17g; error = ',iret
         call endrun
      end if
      call mpi_bcast( r17g1, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r17g1; error = ',iret
         call endrun
      end if
      call mpi_bcast( r18g, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r18g; error = ',iret
         call endrun
      end if
      call mpi_bcast( r18g2, kw , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast r18g2; error = ',iret
         call endrun
      end if
      call mpi_bcast( sj, nw*nz*nlng , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast sj; error = ',iret
         call endrun
      end if
      call mpi_bcast( adj_coeffs, ncof*nlng*nzen , mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'tuv_inti: failed to bcast adj_coeffs; error = ',iret
         call endrun
      end if
#endif

      delw_bin(:nw) = wu(:nw) - wl(:nw)
      deltaw(:nw)   = delw_bin(:nw) * wc(:nw) * 5.039e11
      delw_bin(:nw) = 1./delw_bin(:nw)
      largest  = huge( largest )
      smallest = tiny( largest )

      write(*,'(''tuv_inti: smallest,largest = '',1p,2e21.13)') smallest,largest

      call schu_inti
      call r44_inti( nw, wc )
      call r08_inti( nw, wl, wc )

      end subroutine tuv_inti

      end module mo_tuv_inti
