
      module mo_xemis
!---------------------------------------------------------------------
!       ... initialize interactive isoprene and soil no srf emissions
!---------------------------------------------------------------------

      implicit none

      private
      public :: xemis_inti
      public :: xemis_timestep_inti
      public :: has_xemis_isop
      public :: has_xemis_mterps
      public :: has_xemis_no
      public :: isop_ndx
      public :: mterps_ndx
      public :: no_ndx

      save

      integer :: no_ndx
      integer :: isop_ndx
      integer :: mterps_ndx
      integer :: nlon_veg
      integer :: nlat_veg
      integer :: npft_veg
      real, allocatable :: vegetation_map (:,:,:)
      real, allocatable :: monthly_lai_clm(:,:,:,:)
      real, allocatable :: landmask(:,:)
      logical :: has_xemis_isop
      logical :: has_xemis_mterps
      logical :: has_xemis_no
      logical :: has_xemis

      contains

      subroutine xemis_inti( ncdate, ts_avg, fsds_avg, plonl, platl, pplon )
!-------------------------------------------------------------------------------------
! 	... intialize interactive emissions
!-------------------------------------------------------------------------------------

      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_integer, mpi_double_precision, mpi_comm_comp
#endif
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : dvel_flsp
      use mo_chem_utls,  only : get_spc_ndx, has_fixed_lbc, has_xactive_srfems
      use mo_bvoc,       only : interp_map_inti
      use mo_bvoc,       only : bvoc_inti
      use mo_bvoc,       only : megan_cnt
      use mo_soil_no,    only : soil_no_inti
      use chem_mods,     only : megan_map, srfems_cnt, srfems_lst, xactive_srf_flx_map
      use netcdf

      implicit none

!-------------------------------------------------------------------------------------
! 	... dummy arguments
!-------------------------------------------------------------------------------------
      integer, intent(in) :: ncdate               ! date in yyyymmdd format
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      real, intent(in)    :: ts_avg(plonl,platl,pplon)
      real, intent(in)    :: fsds_avg(plonl,platl,pplon)

!-------------------------------------------------------------------------------------
! 	... local variables
!-------------------------------------------------------------------------------------
      integer :: i, j, ii
      integer :: m, n
      integer :: ncid, vid, dimid
      integer :: astat
      integer :: iret
      real    :: dels
      real, allocatable :: lon_veg (:)
      real, allocatable :: lon_veg_edge (:)
      real, allocatable :: lat_veg (:)
      real, allocatable :: lat_veg_edge (:)
      real, allocatable :: work(:,:)

      character(len=80) :: filename, lpath, rpath

!-------------------------------------------------------------------------------------
! 	... check for isoprene in simulation
!-------------------------------------------------------------------------------------
      isop_ndx       = get_spc_ndx( 'ISOP' )
      has_xemis_isop = has_xactive_srfems( 'ISOP' )
      if( has_xemis_isop ) then
         write(*,*) 'xemis_inti: simulation has interactive ISOP biogenic srf emission'
      end if
!-------------------------------------------------------------------------------------
! 	... check for monoterpenes in simulation
!-------------------------------------------------------------------------------------
      mterps_ndx       = get_spc_ndx( 'C10H16' )
      has_xemis_mterps = has_xactive_srfems( 'C10H16' )
      if( has_xemis_mterps ) then
        write(*,*) 'xemis_inti: simulation has interactive C10H16 biogenic srf emission'
      end if
!---------------------------------------------------------------------------
! 	... check for NO in simulation
!---------------------------------------------------------------------------
      no_ndx = get_spc_ndx( 'NO' )
      has_xemis_no = has_xactive_srfems( 'NO' )
      if( has_xemis_no ) then
        write(*,*) 'xemis_inti: simulation has interactive NO soil srf emission'
      end if
      has_xemis = has_xemis_isop .or. has_xemis_no .or. has_xemis_mterps
      if( .not. has_xemis ) then
         return
      end if

!---------------------------------------------------------------------------
! 	... open netcdf file and read landuse map
!---------------------------------------------------------------------------
master_only : &
      if( masternode ) then
          filename = 'mksrf_pft.060929.nc'
          lpath    = dvel_flsp%local_path
!lke - new PFT map
!        filename= 'mksrf_pft.081008.nc'
!        lpath = '/ptmp/pfister/mz4.5_CARB/'
         rpath    = dvel_flsp%remote_path
         ncid     = open_netcdf_file( filename, lpath, rpath, masteronly=.true. )
         write(*,*) 'xemis_inti: Opened netcdf file ',trim(lpath) // trim(filename)
!---------------------------------------------------------------------------
! 	... get the dimensions
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_dimid( ncid, 'lon', dimid ), 'xemis_inti: dimension nlon not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlon_veg ), 'xemis_inti: failed to read nlon' )
         call handle_ncerr( nf_inq_dimid( ncid, 'lat', dimid ), 'xemis_inti: dimension nlat not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, nlat_veg ), 'xemis_inti: failed to read nlat' )
         call handle_ncerr( nf_inq_dimid( ncid, 'pft', dimid ), 'xemis_inti: dimension nlat not found' )
         call handle_ncerr( nf_inq_dimlen( ncid, dimid, npft_veg ), 'xemis_inti: failed to read nlat' )
!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
         allocate( lon_veg(nlon_veg), lon_veg_edge(nlon_veg+1), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate lon_veg array; error = ',astat
	    call endrun
         end if
         allocate( lat_veg(nlat_veg), lat_veg_edge(nlat_veg+1), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate lat_veg array; error = ',astat
	    call endrun
         end if
         allocate( vegetation_map(nlon_veg,nlat_veg,npft_veg), work(nlon_veg,nlat_veg), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate vegation_map array; error = ',astat
	    call endrun
         end if
         allocate( landmask(nlon_veg,nlat_veg), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate landmask array; error = ',astat
            call endrun
         end if
!---------------------------------------------------------------------------
! 	... read the vegetation map and landmask
!---------------------------------------------------------------------------
         call handle_ncerr( nf_inq_varid( ncid, 'LON', vid), 'xemis_inti: LON not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, lon_veg ), 'xemis_inti: failed to read LON' )
         call handle_ncerr( nf_inq_varid( ncid, 'LAT', vid), 'xemis_inti: LAT not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, lat_veg ), 'xemis_inti: failed to read LAT' )
         call handle_ncerr( nf_inq_varid( ncid, 'EDGEW', vid), 'xemis_inti: EDGEW not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, lon_veg_edge(1) ), 'xemis_inti: failed to read EDGEW' )
         call handle_ncerr( nf_inq_varid( ncid, 'EDGES', vid), 'xemis_inti: EDGES not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, lat_veg_edge(1) ), 'xemis_inti: failed to read EDGES' )
         call handle_ncerr( nf_inq_varid( ncid, 'PCT_PFT', vid), 'xemis_inti: PCT_PFT not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, vegetation_map ), 'xemis_inti: getting vegetation_map' )
         call handle_ncerr( nf_inq_varid( ncid, 'LANDMASK', vid), 'xemis_inti: LANDMASK not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, landmask ), 'xemis_inti: getting landmask' )
!---------------------------------------------------------------------------
! 	... close netcdf file
!---------------------------------------------------------------------------
         call handle_ncerr( nf_close( ncid ), 'xemis_inti: error closing vegetation file' )
      end if master_only
#ifdef USE_MPI
!------------------------------------------------------------------------------
!       ... bcast dimensions
!------------------------------------------------------------------------------
      call mpi_bcast( nlon_veg, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast nlon_veg; error = ',iret
         call endrun
      end if
      call mpi_bcast( nlat_veg, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast nlat_veg; error = ',iret
         call endrun
      end if
      call mpi_bcast( npft_veg, 1, mpi_integer, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast npft_veg; error = ',iret
         call endrun
      end if
!------------------------------------------------------------------------------
!       ... allocate arrays on non-master,computational mpi processes
!------------------------------------------------------------------------------
      if( .not. masternode ) then
         allocate( lon_veg(nlon_veg), lon_veg_edge(nlon_veg+1), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate lon_veg array; error = ',astat
	    call endrun
         end if
         allocate( lat_veg(nlat_veg), lat_veg_edge(nlat_veg+1), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate lat_veg array; error = ',astat
	    call endrun
         end if
         allocate( vegetation_map(nlon_veg,nlat_veg,npft_veg), work(nlon_veg,nlat_veg), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate vegation_map array; error = ',astat
	    call endrun
         end if
         allocate( landmask(nlon_veg,nlat_veg), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'xemis_inti: failed to allocate landmask array; error = ',astat
            call endrun
         end if
      end if
!------------------------------------------------------------------------------
!       ... bcast arrays
!------------------------------------------------------------------------------
      call mpi_bcast( lon_veg, nlon_veg, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast lon_veg; error = ',iret
         call endrun
      end if
      call mpi_bcast( lat_veg, nlat_veg, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast lat_veg; error = ',iret
         call endrun
      end if
      call mpi_bcast( lon_veg_edge, 1, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast lon_veg_edge; error = ',iret
         call endrun
      end if
      call mpi_bcast( lat_veg_edge, 1, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast lat_veg_edge; error = ',iret
         call endrun
      end if
      call mpi_bcast( vegetation_map, nlon_veg*nlat_veg*npft_veg, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast vegetation_map; error = ',iret
         call endrun
      end if
      call mpi_bcast( landmask, nlon_veg*nlat_veg, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast landmask; error = ',iret
         call endrun
      end if
#endif

      dels = lon_veg(1) - lon_veg_edge(1)
!---------------------------------------------------------------------------
! 	... adjust lon coordinates
!---------------------------------------------------------------------------
      lon_veg(:) = 180. + lon_veg(:)
!---------------------------------------------------------------------------
! 	... form edge coordinates
!---------------------------------------------------------------------------
      lon_veg_edge(1)            = lon_veg_edge(1) + 180.
      lon_veg_edge(2:nlon_veg+1) = lon_veg(1:nlon_veg) + dels
      dels = lat_veg(1) - lat_veg_edge(1)
      lat_veg_edge(2:nlat_veg+1) = lat_veg(1:nlat_veg) + dels

      write(*,*) ' '
      write(*,*) 'xemis_inti: lon_veg'
      write(*,'(1p,5g12.5)') lon_veg(:)
      write(*,*) 'xemis_inti: lon_veg_edge'
      write(*,'(1p,5g12.5)') lon_veg_edge(:)
      write(*,*) 'xemis_inti: lat_veg'
      write(*,'(1p,5g12.5)') lat_veg(:)
      write(*,*) 'xemis_inti: lat_veg_edge'
      write(*,'(1p,5g12.5)') lat_veg_edge(:)
!---------------------------------------------------------------------------
! 	... allocate arrays
!---------------------------------------------------------------------------
      allocate( monthly_lai_clm(nlon_veg,nlat_veg,npft_veg,12),stat=astat )
      if( astat /= 0 ) then
         write(*,*) 'xemis_inti: failed to allocate monthly_lai_clm array; error = ',astat
         call endrun
      end if
master_only1 : &
      if( masternode ) then
!-------------------------------------------------------------------------------------
! 	... read monthly lai from CLM dataset
!-------------------------------------------------------------------------------------
          filename = 'mksrf_lai.060929.nc'
          lpath    = dvel_flsp%local_path
!        filename='mksrf_lai.081008.nc'
!        lpath = '/ptmp/pfister/mz4.5_CARB/'
!        rpath    = dvel_flsp%remote_path
         ncid     = open_netcdf_file( filename, lpath, rpath, masteronly=.true. )
         write(*,*) 'xemis_inti: Opened netcdf file ',trim(lpath) // trim(filename)
         call handle_ncerr( nf_inq_varid ( ncid, 'MONTHLY_LAI', vid), 'xemis_inti: MONTHLY_LAI not found in input file' )
         call handle_ncerr( nf_get_var_double( ncid, vid, monthly_lai_clm ), 'xemis_inti: getting monthly_lai_clm' )
!-------------------------------------------------------------------------------------
! 	... close netcdf file
!-------------------------------------------------------------------------------------
         call handle_ncerr( nf_close( ncid ), 'xemis_inti: error closing vegetation file' )
      end if master_only1
#ifdef USE_MPI
      call mpi_bcast( monthly_lai_clm, nlon_veg*nlat_veg*npft_veg*12, mpi_double_precision, 0, mpi_comm_comp, iret )
      if( iret /= 0 ) then
         write(*,*) 'xemis_inti: failed to bcast monthly_lai_clm; error = ',iret
         call endrun
      end if
#endif
!-------------------------------------------------------------------------------------
! 	... flip longitudes for vegetation_map, landmask and lai arrays
!-------------------------------------------------------------------------------------
      do n = 1,npft_veg
         work(:,:) = vegetation_map(:,:,n)
         do j = 1,nlat_veg
            do i = 1,nlon_veg/2
               ii = i + nlon_veg/2
               vegetation_map(i,j,n) = work(ii,j)
            end do
            do i = nlon_veg/2+1,nlon_veg
               ii = i - nlon_veg/2
               vegetation_map(i,j,n) = work(ii,j)
            end do
         end do
      end do

      work(:,:) = landmask(:,:)
      do j = 1,nlat_veg
         do i = 1,nlon_veg/2
            ii = i + nlon_veg/2
            landmask(i,j) = work(ii,j)
         end do
         do i = nlon_veg/2+1,nlon_veg
            ii = i - nlon_veg/2
            landmask(i,j) = work(ii,j)
         end do
      end do

      do m = 1,12
         do n = 1,npft_veg
            work = monthly_lai_clm(:,:,n,m)
            do j = 1,nlat_veg
               do i = 1,nlon_veg/2
                  ii = i + nlon_veg/2
                  monthly_lai_clm(i,j,n,m) = work(ii,j)
               end do
               do i = nlon_veg/2+1,nlon_veg
                  ii = i - nlon_veg/2
                  monthly_lai_clm(i,j,n,m) = work(ii,j)
               end do 
            end do
         end do
      end do
!---------------------------------------------------------------------------
! 	... scale vegetation map
!---------------------------------------------------------------------------
      vegetation_map(:,:,:) = .01 * vegetation_map(:,:,:)

!---------------------------------------------------------------------------
! 	... intialize megan mapping
!---------------------------------------------------------------------------
      megan_cnt = count( megan_map(:) )
      if( megan_cnt > 0 ) then
         call interp_map_inti( nlon_veg, nlat_veg, npft_veg, lat_veg, lat_veg_edge, &
                               lon_veg, lon_veg_edge, plonl, platl, pplon )
         write(*,*) 'xemis_inti: finished interp_map_inti'
         call bvoc_inti( plonl, platl, pplon, nlon_veg, nlat_veg, &
                         npft_veg, lon_veg, lon_veg_edge, lat_veg, lat_veg_edge, &
                         vegetation_map, landmask, monthly_lai_clm, ts_avg, fsds_avg, ncdate )
         write(*,*) 'xemis_inti: finished bvoc_inti'
      end if

!---------------------------------------------------------------------------
! 	... intialize soil no
!---------------------------------------------------------------------------
      if( has_xemis_no ) then
         write(*,*) 'xemis_inti: calling soil_no_inti'
         call soil_no_inti( plonl, platl, pplon, nlon_veg, nlat_veg, &
                            npft_veg, lon_veg, lon_veg_edge, lat_veg, lat_veg_edge, &
                            vegetation_map, landmask, monthly_lai_clm, no_ndx )
         write(*,*) 'xemis_inti: finished soil_no_inti'
      end if

      deallocate( work, stat=astat )
      deallocate( lon_veg, lon_veg_edge, lat_veg, lat_veg_edge, stat=astat )

      end subroutine xemis_inti

      subroutine xemis_timestep_inti( month, ts_avg, fsds_avg, plonl, platl, pplon )
!---------------------------------------------------------------------------
! 	... initialize interactive emissions
!---------------------------------------------------------------------------

      use mo_bvoc,       only : interp_map_iso
      use mo_bvoc,       only : interp_map_mterps
      use mo_bvoc,       only : megan_cnt, megan_species

      implicit none

!---------------------------------------------------------------------------
! 	... dummy arguments
!---------------------------------------------------------------------------
      integer, intent(in) :: month
      integer, intent(in) :: plonl
      integer, intent(in) :: platl
      integer, intent(in) :: pplon
      real, intent(in)    :: ts_avg(plonl,platl,pplon)
      real, intent(in)    :: fsds_avg(plonl,platl,pplon)

!---------------------------------------------------------------------------
! 	... local variables
!---------------------------------------------------------------------------
      integer :: m

      if( megan_cnt > 0 ) then
species_loop : &
         do m = 1,megan_cnt
            select case( megan_species(m) )
            case( 'ISOP' )
               call interp_map_iso( nlon_veg, nlat_veg, npft_veg, landmask, vegetation_map, &
                                    monthly_lai_clm, ts_avg, fsds_avg, month, plonl, &
                                    platl, pplon, m )
            case( 'C10H16' )
               call interp_map_mterps( nlon_veg, nlat_veg, npft_veg, landmask, vegetation_map, &
                                       monthly_lai_clm, month, plonl, platl, pplon, m )
            end select
         end do species_loop
      end if

      end subroutine xemis_timestep_inti

      end module mo_xemis
