
      module mo_zenith

      contains

      subroutine zenith( lat, long, idate, ut, zen )
!-----------------------------------------------------------------------------
!   purpose:
!   calculate solar zenith angle for a given time and location.
!   calculation is based on equations given in:  paltridge and platt, radia-
!   tive processes in meteorology and climatology, elsevier, pp. 62,63, 1976.
!   fourier coefficients originally from:  spencer, j.w., 1971, fourier
!   series representation of the position of the sun, search, 2:172.
!   note:  this approximate program does not account fro changes from year
!   to year.
!-----------------------------------------------------------------------------
!   parameters:
!   lat   - real, latitude of location (degrees)                          (i)
!   long  - real, longitude of location (degrees)                         (i)
!   idate - integer, date in the form yymmdd                              (i)
!   ut    - real, local time in decimal ut (e.g., 16.25 means 15 minutes  (i)
!           after 4 pm)
!   zen   - real, solar zenith angle (degrees)                            (o)
!-----------------------------------------------------------------------------

      use mo_constants, only  : d2r, r2d, pi

      implicit none

!-----------------------------------------------------------------------------
!	... dummy arguments
!-----------------------------------------------------------------------------
      integer, intent(in) :: idate
      real, intent(in)    :: lat,long
      real, intent(in)    :: ut
      real, intent(out)   :: zen

!-----------------------------------------------------------------------------
! 	... local variables
!-----------------------------------------------------------------------------
      integer :: i
      integer :: iiyear, imth, iday, ijd
      integer :: imn(12) = (/ 31,28,31,30,31,30,31,31,30,31,30,31 /)

      real :: lbut,lzut
      real :: rlt
      real :: d, tz, rdecl, eqr, eqh, zpt
      real :: csz, zr, caz, raz 
      real :: sintz, costz, sin2tz, cos2tz, sin3tz, cos3tz

!-----------------------------------------------------------------------------
! 	... convert to radians
!-----------------------------------------------------------------------------
      rlt = lat*d2r

!-----------------------------------------------------------------------------
! 	... parse date
!-----------------------------------------------------------------------------
      iiyear = idate/10000
      imth   = (idate - iiyear*10000)/100
      iday   = idate - iiyear*10000 - imth*100

!-----------------------------------------------------------------------------
! 	... identify and correct leap years
!-----------------------------------------------------------------------------
      if( mod(iiyear,4) == 0 ) then
         imn(2) = 29
      else
         imn(2) = 28
      end if

!-----------------------------------------------------------------------------
! 	... compute current (julian) day of year ijd = 1 to 365
!-----------------------------------------------------------------------------
      ijd = 0
      do i = 1,imth-1
         ijd = ijd + imn(i)
      end do
      ijd = ijd + iday

!-----------------------------------------------------------------------------
! 	... calculate decimal julian day from start of year:
!-----------------------------------------------------------------------------
      d = real(ijd-1) + ut/24.

!-----------------------------------------------------------------------------
! 	... equation 3.8 for "day-angle"
!-----------------------------------------------------------------------------
      tz = 2.*pi*d/365.

!-----------------------------------------------------------------------------
! 	... calculate sine and cosine from addition theoremes for 
!           better performance;  the computation of sin2tz,
!           sin3tz, cos2tz and cos3tz is about 5-6 times faster
!           than the evaluation of the intrinsic functions 
!
!           it is sin(x+y) = sin(x)*cos(y)+cos(x)*sin(y)
!           and   cos(x+y) = cos(x)*cos(y)-sin(x)*sin(y)
!
!           sintz  = sin(tz)      costz  = cos(tz)
!           sin2tz = sin(2.*tz)   cos2tz = sin(2.*tz)
!           sin3tz = sin(3.*tz)   cos3tz = cos(3.*tz)
!-----------------------------------------------------------------------------
      sintz  = sin( tz )
      costz  = cos( tz )
      sin2tz = 2.*sintz*costz
      cos2tz = costz*costz - sintz*sintz
      sin3tz = sintz*cos2tz + costz*sin2tz
      cos3tz = costz*cos2tz - sintz*sin2tz

!-----------------------------------------------------------------------------
! 	... equation 3.7 for declination in radians
!-----------------------------------------------------------------------------
      rdecl = 0.006918 - 0.399912*costz  + 0.070257*sintz  &
                       - 0.006758*cos2tz + 0.000907*sin2tz &    
                       - 0.002697*cos3tz + 0.001480*sin3tz

!-----------------------------------------------------------------------------
! 	... equation 3.11 for equation of time  in radians
!-----------------------------------------------------------------------------
      eqr   = 0.000075 + 0.001868*costz  - 0.032077*sintz &
      		       - 0.014615*cos2tz - 0.040849*sin2tz

!-----------------------------------------------------------------------------
! 	... convert equation of time to hours
!-----------------------------------------------------------------------------
      eqh = eqr*24./(2.*pi) 

!-----------------------------------------------------------------------------
! 	... calculate local hour angle (hours):
!-----------------------------------------------------------------------------
      lbut = 12. - eqh - long*24./360 

!-----------------------------------------------------------------------------
! 	... convert to angle from ut
!-----------------------------------------------------------------------------
      lzut = 15.*(ut - lbut)
      zpt  = lzut*d2r

!-----------------------------------------------------------------------------
! 	... equation 2.4 for cosine of zenith angle 
!-----------------------------------------------------------------------------
      csz = sin(rlt)*sin(rdecl) + cos(rlt)*cos(rdecl)*cos(zpt)
      zr  = acos(csz)
      zen = zr*r2d

      end subroutine zenith

      end module mo_zenith
