
      module woods

      implicit none

      private
      public :: woods_init
      public :: woods_set_etf
      public :: nbins, woods_etf, we

      save

      integer           :: nstruct
      integer           :: nbins
      real, allocatable :: wc(:)                ! wave interval center (nm)
      real, allocatable :: we(:)                ! wave interval edges (nm)
      real, allocatable :: wlintv(:)            ! wave interval (nm)
      real, allocatable :: wlintvi(:)           ! inverse wave interval (nm)
      real, allocatable :: refmin(:)
      real, allocatable :: rt27day(:)
      real, allocatable :: rt11yr(:)
      real, allocatable :: woods_etf(:)

      contains

      subroutine woods_init
!---------------------------------------------------------------
!	... initialize woods etf module
!---------------------------------------------------------------

      use mo_mpi,        only : masternode
#ifdef USE_MPI
      use mo_mpi,        only : mpi_comm_comp, mpi_integer, mpi_double_precision
#endif
      use mo_file_utils, only : open_netcdf_file
      use mo_control,    only : woods_flsp
      use netcdf

      implicit none

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      integer  :: ncid
      integer  :: n
      integer  :: dimid
      integer  :: varid
      integer  :: astat
      integer  :: status

!-----------------------------------------------------------------------
!	... readin the etf data
!-----------------------------------------------------------------------
masterproc_only : &
      if( masternode ) then
         ncid = open_netcdf_file( woods_flsp%nl_filename, &
                                  woods_flsp%local_path, &
                                  woods_flsp%remote_path, masteronly=.true. )
!-----------------------------------------------------------------------
!	... check primary dimension consistency
!-----------------------------------------------------------------------
         status = nf_inq_dimid( ncid, 'dim1_WC', dimid )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to get dim1_WC id; error = ',status
            call endrun
         end if
         status = nf_inq_dimlen( ncid, dimid, nbins )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to read dim1_WC dimension; error = ',status
            call endrun
         end if
         status = nf_inq_dimid( ncid, 'dim1_WLINT', dimid )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to get dim1_WLINT id; error = ',status
            call endrun
         end if
         status = nf_inq_dimlen( ncid, dimid, n )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to read dim1_WLINT dimension; error = ',status
            call endrun
         end if
         if( n /= nbins ) then
            write(*,*) 'woods_init: WLINT dimension(',n,') does not match bin count ',nbins
            call endrun
         end if
         status = nf_inq_dimid( ncid, 'dim1_REFMIN', dimid )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to get dim1_REFMIN id; error = ',status
            call endrun
         end if
         status = nf_inq_dimlen( ncid, dimid, n )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to read dim1_REFMIN dimension; error = ',status
            call endrun
         end if
         if( n /= nbins ) then
            write(*,*) 'woods_init: REFMIN dimension(',n,') does not match bin count ',nbins
            call endrun
         end if
         status = nf_inq_dimid( ncid, 'dim1_RT27DAY', dimid )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to get dim1_RT27DAY id; error = ',status
            call endrun
         end if
         status = nf_inq_dimlen( ncid, dimid, n )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to read dim1_RT27DAY dimension; error = ',status
            call endrun
         end if
         if( n /= nbins ) then
            write(*,*) 'woods_init: RT27DAY dimension(',n,') does not match bin count ',nbins
            call endrun
         end if
         status = nf_inq_dimid( ncid, 'dim1_RT11YR', dimid )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to get dim1_RT11YR id; error = ',status
            call endrun
         end if
         status = nf_inq_dimlen( ncid, dimid, n )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to read dim1_RT11YR dimension; error = ',status
            call endrun
         end if
         if( n /= nbins ) then
            write(*,*) 'woods_init: RT11YR dimension(',n,') does not match bin count ',nbins
            call endrun
         end if

!-----------------------------------------------------------------------
!	... allocate primary arrays
!-----------------------------------------------------------------------
         allocate( wc(nbins), we(nbins+1), wlintv(nbins), wlintvi(nbins), &
                   refmin(nbins), rt27day(nbins), rt11yr(nbins), woods_etf(nbins), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'woods_init: failed to allocate wc ... woods_etf; error = ',astat
            call endrun
         end if
!-----------------------------------------------------------------------
!	... read primary arrays
!-----------------------------------------------------------------------
         status = nf_inq_varid( ncid, 'WC', varid )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to get WC id; error = ',status
            call endrun
         end if
         status = nf_get_var_double( ncid, varid, wc )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to read wc; error = ',status
            call endrun
         end if
         status = nf_inq_varid( ncid, 'WLINT', varid )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to get WLINT id; error = ',status
            call endrun
         end if
         status = nf_get_var_double( ncid, varid, wlintv )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to read wlintv; error = ',status
            call endrun
         end if
         status = nf_inq_varid( ncid, 'REFMIN', varid )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to get REFMIN id; error = ',status
            call endrun
         end if
         status = nf_get_var_double( ncid, varid, refmin )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to read refmin; error = ',status
            call endrun
         end if
         status = nf_inq_varid( ncid, 'RT27DAY', varid )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to get RT27DAY id; error = ',status
            call endrun
         end if
         status = nf_get_var_double( ncid, varid, rt27day )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to read rt27day; error = ',status
            call endrun
         end if
         status = nf_inq_varid( ncid, 'RT11YR', varid )
         if( status /= 0 ) then
            write(*,*) 'woofs_init: failed to get RT11YR id; error = ',status
            call endrun
         end if
         status = nf_get_var_double( ncid, varid, rt11yr )
         if( status /= 0 ) then
            write(*,*) 'woods_init: failed to read rt11yr; error = ',status
            call endrun
         end if

         status = nf_close( ncid )
      end if masterproc_only

#ifdef USE_MPI
      call mpi_bcast( nbins, 1, mpi_integer, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'woods_init: failed to bcast nbins; error = ',status
         call endrun
      end if
      if( .not. masternode ) then
         allocate( wc(nbins), we(nbins+1), wlintv(nbins), wlintvi(nbins), &
                   refmin(nbins), rt27day(nbins), rt11yr(nbins), woods_etf(nbins), stat=astat )
         if( astat /= 0 ) then
            write(*,*) 'woods_init: failed to allocate wc ... woods_etf; error = ',astat
            call endrun
         end if
      end if
      call mpi_bcast( wc, nbins, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'woods_init: failed to bcast wc; error = ',status
         call endrun
      end if
      call mpi_bcast( wlintv, nbins, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'woods_init: failed to bcast wlintv; error = ',status
         call endrun
      end if
      call mpi_bcast( refmin, nbins, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'woods_init: failed to bcast refmin; error = ',status
         call endrun
      end if
      call mpi_bcast( rt27day, nbins, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'woods_init: failed to bcast rt27day; error = ',status
         call endrun
      end if
      call mpi_bcast( rt11yr, nbins, mpi_double_precision, 0, mpi_comm_comp, status )
      if( status /= 0 ) then
         write(*,*) 'woods_init: failed to bcast rt11yr; error = ',status
         call endrun
      end if
#endif

      wlintvi(:)  = 1./wlintv(:)
      we(:nbins)  = wc(:nbins) - .5*wlintv(:nbins)
      we(nbins+1) = wc(nbins) + .5*wlintv(nbins)

      end subroutine woods_init

      subroutine woods_set_etf( f107, f107a , etf_temp )
!---------------------------------------------------------------
!	... set woods etf
!---------------------------------------------------------------

      implicit none

!---------------------------------------------------------------
!	... dummy arguments
!---------------------------------------------------------------
      real, intent(in) :: f107
      real, intent(in) :: f107a
      real, optional, intent(out) :: etf_temp(nbins)

!---------------------------------------------------------------
!	... local variables
!---------------------------------------------------------------
      real, parameter :: f7min  = 71.
      real, parameter :: f7max  = 212.
      real, parameter :: f7deli = 1./(f7max - f7min)
      real, parameter :: f7roti = 1./73.

      real :: ref_27day(nbins)
      real :: ref_11yr(nbins)

      ref_27day(:) = refmin(:) * (rt27day(:) - 1.)
      ref_11yr(:)  = refmin(:) * (rt11yr(:) - 1.)

      if ( present( etf_temp ) ) then
         etf_temp(:) = refmin(:) + ref_27day(:) * (f107 - f107a)*f7roti &
              + ref_11yr(:) * (f107a - f7min)*f7deli
      else
         woods_etf(:) = refmin(:) + ref_27day(:) * (f107 - f107a)*f7roti &
              + ref_11yr(:) * (f107a - f7min)*f7deli
      end if

#ifdef WOODS_DIAGS
      write(*,*) ' '
      write(*,*) '----------------------------------------------'
      write(*,*) 'woods_set_etf: diagnostics for woods_etf'
      write(*,*) 'woods_set_etf: nbins = ',nbins
      write(*,'(1p,5g15.7)') woods_etf(:)
      write(*,*) '----------------------------------------------'
      write(*,*) ' '
#endif

      end subroutine woods_set_etf

      end module woods
