// ckopus
// read & interrogate OPUS files

// Aug 2016
// added RIK, TKB, MOS, SYO
// made correction to collection date read  to check for yyyy/mm/dd format

// Oct 2013
// bug fix with call to filterid

// July 2013
// added check for magic number return code for incorrect value = 3

// June 2013
// - added -U switch to apply the UTC offset
// - bug fix: lat lon alt arguments were not applied
// - changed -e to -a altitude switch
// - added -T flag to write a t15asc file - caveat whole file only and no snr calculation (-999.9)

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>

// Local functions
#include"constant.h"
//#include"meeus.h" called in bnr.h
#include"opus.h"
#include"bnr.h"

char version[] = "ckopus.c V1.0.6 20160803";

#define VERBOSE FALSE

// variables used in nearly all functions
FILE *opus;

char specblock[] ="NONE\0";

char fltr[]="X\0";

int datablocktype=0, paramblocktype=0;

float  aperture=-1., offs=0.;
int    applyUTC=0;
double field_of_view=-1.;

struct utc_date collection_date = {0,0,0,0},         utc_coll_date = {0,0,0,0};;
struct utc_time collection_time = {0.0,0,0,0,0,0.0}, utc_coll_time = {0.0,0,0,0,0,0.0};
struct coord res;
struct loc site;

// Phase, Interferogram, Emission, Single Channel, Interferogram #2, Single Chan #2 block
struct SPPB_rcd IFPB, EMPB, SPPB, PHPB, SCPB, I2PB, S2PB, TRAN;
struct AQPB_rcd AQPB;                     // Acquisition block
struct FTPB_rcd FTPB;                     // Fourier Transform block
struct OPPB_rcd OPPB;                     // Optic block
struct INPB_rcd INPB;                     // Instrument block

int starttime( double *bruker_time_adjustment ) {

   // this routine accounts for Bruker idiosyncratic timing only,
   //    not pc clock errors or UTC offsets (see initialize())

   // this routine tries to calculate the optimal time for the SZA calculation given:
   //    a) the strt_flag given in the location structure
   //    b) then number of scans, velocity, or scan time of a scan
   //    c) whether forward or backward or both type scans are in the file

   // optimal time for the solar zenith angle flag may be determined by
   //    known empirical offset from one of the time stamps in the opus file
   //    calculation given the length of scan, number of scans, whether forward fast return or
   //    forward/backward, velocity and resolution

   // some older notes - not implemented verbatim here
   // site.strt_flg (sflag) :
   //    1 = Empirical Adjust Value (MLOA and Oklahoma)
   //    2 = Oklahoma Calculated (Oklahoma files starting with 's')
   //    3 = Time recorded at end of scan (Denver and Alaska)
   //    4 = Time recorded before scan
   //    5 = Steve/Nicholas/Bruker ZPT calc

   // here are choices of site.strt_flg (sflag) :
   //    0 - make no adjustment
   //    1 - empirical : fixed number currently given via site.offs_secs
   //    2 - forward & backward scans in two files (based on opus v3.04 ifs120 circa 1996)
   //    3 - forward scans (based on opus v3.04 120 circa 1996) uses inteferogram points
   //    4 - forward / backward one file (IFS125) plus -t time
   //    5 - forward scans using scantime duration (IFS125) plus -t time
   //    6 - forward scans fast return use scan duration value, time is end of last scan (eg PHAS in OPUS v1.41)

// Veocity values in Hz by Bruker index
   double velocity[] = { 0., 2200., 3000., 4000., 5000., 7500., 10000., 20000., 40000., 60000., 80000. };

   int time_adjust = 0, ns = 0, i, verb=FALSE;
   double ts=0.0, ti=0.0, offs1=0.0, offs2=0.0, rvel=80000.0, svel=0.0;

   int clock       = site.sec_offs;    // fixed offset either clock or known Bruker offset
   int sflag       = site.strt_flg;    // time calculation flag
   int nsf         = INPB.ASS;         // # of scans in this file whether fwd or bwd file
   int nss         = AQPB.NSS;         // # of scans taken in experiment = gfw+gbw for type 2
   int gfw         = INPB.GFW;         // # of good forward scans in file
   int gbw         = INPB.GBW;         // # of good backward scans in file
   int pkl         = INPB.PKL;         // interferogram peak location in data points
   int ipt         = IFPB.NPT;         // interferogram # of data points
   double dur      = INPB.DUR;         // scan time in seconds
   double lwn      = INPB.LWN;         // laser wavenumber
   double res      = AQPB.RES;         // spectral resolution
   double phr      = FTPB.PHR;         // phase resolution

   // Determine if its a forward only or forward/backward pair in a 125 file
   // sflag needs to be 4 or 5 ! (was 3, 4 before 10Feb2016)
   // This accounts for designated F & B but file is F only and designated F only but file is F & B
   if(( nsf == nss ) && ( nsf == gfw ) && ( sflag == 4 ))sflag = 5;
   if(( nsf == nss ) && ( nsf >  gfw ) && ( nsf > gbw ) && ( sflag == 5 ) && ( gfw+gbw > 0 ))sflag = 4;

   // Determine the scan velocity
   //    on the 125 the value of optical param block velocity is the velocity in kHz
   //    on the 120 it is the index (see velocity array defined above)
   // Try to determine if the value is a integer (index) or real number (kHz)
   i = 0;
   while( !ispunct( OPPB.VEL[i] )) {
      i++;
      if( i == ENUMSZ )break;
   }

   //printf("i= %i\n", i );
   if( i < ENUMSZ ) {
      svel = atof(OPPB.VEL) * 1000.;      // found a '.'
   } else {
      i = atoi(OPPB.VEL);
      if( i < 1 || i > 10 ) {
         printf(" velocity index out of range (1-10) %s %i\n", OPPB.VEL, i );
        exit (1);
      }
      svel = velocity[i];               // must be an integer
   }

   if( verb ) {
      fprintf(stderr,"     Determined scan velocity = %lf\n", svel );
      fprintf(stderr,"     sflg:%i gfw:%i gbw:%i nss:%i nsf:%i pkl:%i ipt:%i\n", sflag, gfw, gbw, nss, nsf, pkl, ipt );
      fprintf(stderr,"     dur:%lf lwn:%lf res:%lf phr:%lf\n", dur, lwn, res, phr );
   }

	// calculate time_adjust if needed
	switch( sflag ) {

	   case 0:        // no time adjustment done here
	      *bruker_time_adjustment = 0.0;
	      break;

		case 1:        // empirical currently done with site.offs_secs on command line or in site struct
	      *bruker_time_adjustment = clock;
		   break;

		case 2:        // forward / backward two file : timestamp is time of first zpd, correction is half the time to the last zpd
		   if( gfw == 1 ) {                    // block time is zpd time
		      *bruker_time_adjustment = 0.0 + clock;
		   } else if( gbw == 1 ) {		         // block time is start of bkw scan, zpd at the end, dur is time of one scan
            *bruker_time_adjustment = dur + clock;
         } else if( gfw > 1 ) {              // fwd & bwd scans take the same amount of time
            ns = nss - 1;                    // dur is time from start of 1st scan (zpd) to end of last fwd scan (mpd)
            ts = dur / ((double) ns);        // time of one scan
            *bruker_time_adjustment = (dur - ts) / 2.0 + clock;            // mid point of fwd zpd's
         } else if( gbw > 1 ) {              // fwd & bwd scans take the same amount of time
            ns = nss - 1;                    // dur is time from start of 1st bwd scan (mpd) to end of last bwd scan (zpd)
            ts = dur / ((double) ns);
   		   *bruker_time_adjustment = ts + (dur - ts) / 2.0 + clock;       // shift to bwd zpd, add midpoint
   		}
   		break;

		case 3:        // forward scan fast return calculating scantime between first and last zpd
		   if( nsf > 1 ) {
            ts = .95 * lwn*( 1./res + 1./phr) / svel;              		// time for one scan (ts)
            ti = ( dur - (double)nsf * ts ) / (double)( nsf - 1 );      // inactive time (ti), during backward scan
            offs1 = (double)( pkl / ipt ) * ts;			                  // zpdtime offset for scan 1
                                                                        // zpdtime offset for last scan (nss)
            offs2 = (double)( nsf - 1 + pkl / ipt ) * ts + ( nsf - 1 ) * ti;
            *bruker_time_adjustment = ( offs1 + offs2 ) / 2. + clock;
         } else {
            offs1 = (double)( pkl / ipt ) * ts;			                  // zpdtime offset for scan 1
            *bruker_time_adjustment = offs1 + clock;
         }
			break;

		case 4:        // forward & backward in 1 file (125)
		   if( dur == 0.0 ) dur = nss * .95 * lwn*( 1./res + 1./phr) / svel;
		   *bruker_time_adjustment = dur / 2.0 + clock;
			break;

		case 5:        // forward scan fast return calculating scantime using duration
		   if( nsf > 1 ) {
            ts = .95 * lwn*( 1./res + 1./phr) / svel;              		// time for one scan (ts)
            ti = ( dur - (double)nsf * ts ) / (double)( nsf - 1 );      // inactive time (ti), during backward scan
            offs1 = dur - ts;
            offs2 = nsf*ts + (nsf-1)*ti;                                // recalculate dur
            *bruker_time_adjustment = (dur - ts) / 2.0 + clock;         // mid point of fwd zpd's
         } else {
            *bruker_time_adjustment = clock;
         }
			break;

		case 6:        // forward scan fast return calculating scantime using duration
		               // file time is end of scan not counting last fast return
		               // eg PHAS time of early MLO v1.41 OPUS/120HR
		   if( nsf > 1 ) {
            ts = .95 * lwn*( 1./res + 1./phr) / svel;              		// time for one scan (ts)
            ti = ( dur - (double)nsf * ts ) / (double)( nsf - 1 );      // inactive time (ti), during backward scan
            offs1 = (dur - ts) / 2.0;                                   // mid point of fwd zpd's
            offs2 = nsf*ts + (nsf-1)*ti;                                // recalculate dur
            *bruker_time_adjustment = -ts - offs1 + clock;               // this is added to end of scan time (PHAS)
         } else {
            *bruker_time_adjustment = -ts + clock;
         }
			break;

		default:
			break;
	}

   if( verb ) {
      fprintf(stderr,"     sflg:%i\n     ts:%lf ti:%lf ofs1:%lf ofs2:%lf dur:%lf adj:%lf\n", sflag, ts, ti, offs1, offs2, dur, *bruker_time_adjustment );
      fprintf(stderr,"     ts:%lf ti:%lf ofs1:%lf ofs2:%lf dur:%lf adj:%lf\n", ts/60., ti/60., offs1/60., offs2/60., dur/60., *bruker_time_adjustment/60. );
   }

      fprintf(stderr,"     ZPD flag, time adjust. :  %16d %16.4lf\n", sflag, *bruker_time_adjustment );

   return(0);


			// calc time adjust assuming time recorded at beginning
			//time_adjust=int (3*scan_time*(num_scans-1) / (2*(3*num_scans-1))+.5);

			// calc time adjust assuming time recorded at end
			//time_adjust=-int(scan_time/2*(4+3*(num_scans-1)) / (3*num_scans-1)+.5);

/*

*/

   return(0);

}


int getSPblock( char *filename, char *blocktype, int SWAP ) {

   // choose data blocktype, if none is given we are not accessing data but see if we have one
   if( ! strncmp( blocktype, "NONE", 4 )) {
      datablocktype = SNGCSPCBLK;
      paramblocktype = SNGCPARBLK;
      if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
         fprintf( stderr,"  Single Channel Param Block Read OK.\n" );
         strncpy( specblock, "SNGC", 4 );
         strncpy( blocktype, "SNGC", 4 );
      } else {
         // no sample spectra so check for emission block
         fprintf( stderr, "*Could not read Single Channel Param Block...\n");
         datablocktype = EMISSPCBLK;
         paramblocktype = EMISPARBLK;
         if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
            fprintf( stderr,"  Emission Param Block Read OK.\n" );
            strncpy( specblock, "EMIS", 4 );
            strncpy( blocktype, "EMIS", 4 );
         } else {
            // no emission spectra either so check for transmittance block
            fprintf( stderr, "*Could not read Single Chan or Emission Param Block...\n");
            datablocktype = TRANSPCBLK;
            paramblocktype = TRANPARBLK;
            if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
               fprintf( stderr,"  Transmittance Param Block Read OK.\n" );
               strncpy( specblock, "TRAN", 4 );
               strncpy( blocktype, "TRAN", 4 );
            } else {
               report( "*Error No Emission or Single Channel or Transmittance Spectra block in file", opus_error, filename );
               fclose( opus );
               return(8);
            }
         }
      }

   // if there is a desired blocktype find it
   //    we use the time from this as well!!!
   } else {

      if( ! strncmp( blocktype, "IFGM", 4 )) {
         datablocktype = IFGMSPCBLK;
         paramblocktype = IFGMPARBLK;
         strncpy( specblock, "IFGM", 4 );
      }
      if( ! strncmp( blocktype, "PHAS", 4 )) {
         datablocktype = PHASSPCBLK;
         paramblocktype = PHASPARBLK;
         strncpy( specblock, "PHAS", 4 );
      }
      if( ! strncmp( blocktype, "EMIS", 4 )) {
         datablocktype = EMISSPCBLK;
         paramblocktype = EMISPARBLK;
         strncpy( specblock, "EMIS", 4 );
      }
      if( ! strncmp( blocktype, "SNGC", 4 )) {
         datablocktype = SNGCSPCBLK;
         paramblocktype = SNGCPARBLK;
         strncpy( specblock, "SNGC", 4 );
      }
      if( ! strncmp( blocktype, "TRAN", 4 )) {
         datablocktype = TRANSPCBLK;
         paramblocktype = TRANPARBLK;
         strncpy( specblock, "TRAN", 4 );
      }
      if( ! strncmp( blocktype, "SNG2", 4 )) {
         datablocktype = SNG2SPCBLK;
         paramblocktype = SNG2PARBLK;
         strncpy( specblock, "SNG2", 4 );
      }
      if( ! strncmp( blocktype, "IFG2", 4 )) {
         datablocktype = IFG2SPCBLK;
         paramblocktype = IFG2PARBLK;
         strncpy( specblock, "IFG2", 4 );
      }
      if( paramblocktype != 0 ) {
          if( read_SPPAR_block( opus, paramblocktype, &SPPB, SWAP )) {
            fprintf( stderr,"  %s Param Block Read OK.\n", blocktype );
         } else {
            fprintf( stderr,"  *%s Param Block Read NOT OK!!\n", blocktype );
            fclose( opus );
            return(9);
         }
      } else {
         fprintf( stderr,"** %s Block type invalid.\n", blocktype );
         fclose( opus );
         return(10);
      }
   }

   return(0);
}


void filterid( void ) {

// find filter id as a 1 char
// use site information
// sketchy the string is not standard : likely need info for all sites

   char flt1[]="X\0";
   char fword[7], fposition[]="x\0", buf[3];

   //printf( "<%s>\n", fltr );
   //printf( "<%s>\n", site.tag );

   // default scan for filter id number
   //sscanf( OPPB.OPF, "%s %1s", fword, fltr );
   //printf( "\n OPF: %s\t %s\t %s\n", OPPB.OPF, fword, fltr );
   //sscanf( OPPB.OF1, "%s %1s", fword, fltr );
   //printf( " OF1: %s\t %s\t %s\n", OPPB.OF1, fword, fltr );

   // TAB
   if( !strncmp( site.tag, "TAB", 3 )) {
      // 120M 1999 through 2014
      if( utc_coll_date.year < 2015 ) {
         sscanf( OPPB.OPF, "%s %1s", fword, fltr );          // string should be "Filter n"
         if( !strncmp( fword, "OPEN", 4 ) && utc_coll_date.year > 2011 ) strcpy( fltr, "9");
      } else {
         // 125HR installed May 2015 fixed filter 6 at MCT
         sscanf( OPPB.OF1, "%1s%1s %s %1s", buf, fposition, fword, fltr );          // string should be "Pn Filter n
         //printf( "<%s> %s    %s   %s   %s\n", OPPB.OF1, buf, fposition, fword, fltr );          // string should be "Pn Filter n
         if( !strncmp( fword, "Open", 4 ) && !strncmp( fposition, "8", 1 )) strcpy( fltr, "6");
      }

   // MLO
   } else if ( !strncmp( site.tag, "MLO", 3 )) {
      // MLO after 2010 has 2 optical filters
      // OF1 has filters 1, 2, 3, 4, 5, 6, mesh(7) & Open(8) in those positions
      // OPF has filters 9, A, B in position 2, 1, 3
      // when using one the other is Open
      sscanf( OPPB.OF1, "%s %1s", fword, fltr );          // get fword
      //printf("%i %s %s >%s< %1s\n", utc_coll_date.year, flt1, OPPB.OF1, fword, fltr );

      if( utc_coll_date.year > 2010 ) {
         if( !strncmp( fword, "Open", 4 )) {
            sscanf( OPPB.OPF, "%s %1s", fword, flt1 );
            //printf("%s %s >%s< %1s\n", flt1, OPPB.OF1, fword, flt1 );
            if(!strncmp( flt1, "2", 1 )) strcpy( fltr, "9");
            if(!strncmp( flt1, "1", 1 )) strcpy( fltr, "A");
            if(!strncmp( flt1, "3", 1 )) strcpy( fltr, "B");
         } else {
            sscanf( OPPB.OF1, "%s %s", fword, fltr );
            //printf("%s %s #%s# %1s\n", flt1, OPPB.OF1, fword, fltr );
         }
      }

   // FL0
   } else if ( !strncmp( site.tag, "FL0", 3 )) {
      // FL0 after 2017 has 2 optical filters
      // OPF is in det. compartment, OP1 is in beamsplitter compartment
      // no data in 2018 until October when inst. delivered
      // OF1 has filters 1, 2, 3, 4, 5, 6, in those positions, NDSC 9 in pos. 7 & pos. 0 is Open
      // OPF has filters A, B, 0 in position 1, 2, 3
      // when using one the other is "Open"
      // Currently (Nov 2018) the labels are "Filter x" for both

      sscanf( OPPB.OF1, "%s %1s", fword, fltr );          // get fword
      //printf("%i %s %s >%s< %1s\n", utc_coll_date.year, flt1, OPPB.OF1, fword, fltr );

      if(!strncmp( fltr, "7", 1 )) strcpy( fltr, "9");

      if( utc_coll_date.year > 2017 ) {
         if( !strncmp( fword, "Open", 4 )) {
            sscanf( OPPB.OPF, "%s %1s", fword, flt1 );
            //printf("%s %s >%s< %1s\n", flt1, OPPB.OF1, fword, flt1 );
            if(!strncmp( flt1, "1", 1 )) strcpy( fltr, "A");
            if(!strncmp( flt1, "2", 1 )) strcpy( fltr, "B");
            if(!strncmp( flt1, "3", 1 )) strcpy( fltr, "0");
         } else {
            sscanf( OPPB.OF1, "%s %s", fword, fltr );
            //printf("%s %s #%s# %1s\n", flt1, OPPB.OF1, fword, fltr );
         }
      }

   // EUR
   } else if ( !strncmp( site.tag, "EUR", 3 )) {
      sscanf( OPPB.OF1, "%s %s", fword, fltr );

   // LDR
   } else if ( !strncmp( site.tag, "LDR", 3 )) {
      sscanf( OPPB.OF1, "%s %s", fword, fltr );
   }

   //printf( "\n of1: %s\t opf: %s\t %s\t fltrID : %s\n", OPPB.OF1, OPPB.OPF, fword, fltr );

}


void skipComment( FILE *input) {
    char buffer[81];
    int i=0;
    memset( buffer, '\0', strlen( buffer ) );
    //fprintf( stderr, "%i  %s\n", -1, buffer );
    while( buffer[0] != '$' )  {
       //fprintf( stderr, "%i  %s\n", i, buffer );
       fscanf( input, "%80s", buffer );
       //fprintf( stderr, "%i  %s\n", i++, buffer );
   }
}


int write_t15( char *blocktype, int SWAP ) {

   FILE *bnr, *txt;
   char title[80], t15name[]="t15asc.4", cdum[ENUMSZ];
   int fnum_points, length, pointer, i, rc;
   float *ydata;
   double fstartw, fstopw, spacing;

   strncpy( title, bnr_makeheader( utc_coll_date, utc_coll_time, res,
                        INPB.DUR, AQPB.RES, FTPB.APF, field_of_view), HEADLEN );

	fprintf( stderr, "  BNR Header                : '%s' %ld\n", title, strlen(title) );

   fstartw     = SPPB.FXV;
   fstopw      = SPPB.LXV;
   fnum_points = SPPB.NPT;
   spacing = (fstopw-fstartw) / (fnum_points-1);
   if( spacing < 0.0 ) spacing = (fstartw-fstopw) / (fnum_points-1);

   fprintf( stderr, "  Data range                : %12.4lf to %12.4lf %8d points spaced at %12.8lf\n", fstartw, fstopw, fnum_points, spacing );
	fprintf( stderr, "  Reading spectra block     : %Xx\n", datablocktype );

   if( ! opus_searchdir( opus, datablocktype, &length, &pointer, SWAP )) {
      fprintf( stderr," *Cannot find data block    : %s %Xx\n", blocktype, datablocktype );
      if( ! opus_searchdir( opus, datablocktype + OFFSET, &length, &pointer, SWAP )) {
         fprintf( stderr," *Cannot find data block    : %s %Xx\n", blocktype, datablocktype + OFFSET );
         fclose( opus );
         return(1);
      } else fprintf( stderr,"  Data block Found          : %s 0x%x\n", blocktype, datablocktype + OFFSET );
   } else fprintf( stderr,"  Data block Found          : %s 0x%x\n", blocktype, datablocktype );

	ydata = (float *) malloc ( length*sizeof(float) );

	if( ydata == (float *) 0 ) {
		fprintf( stderr, " *Could not allocate array for y data.\n" );
		return(2);
	}

	if( opus_read_spectrum( opus, &fstartw, &fstopw, length, pointer, SPPB.CSF, SPPB.NPT, ydata, SWAP )) {
		fprintf( stderr," *Cannot read data.\n" );
		return(3);
	}

	fprintf( stderr, "  Opening file             : %s\n", t15name );

   if(( bnr = fopen( t15name, "wa" )) == (FILE *) NULL ) return(4);

   printf("   Writing ascii file\n");

   fprintf( bnr, "%10.5lf %10.5lf %10.5lf %10.5lf %10.5lf\n", res.zen, res.roe, res.lat, res.lon, -999.9 );
   fprintf( bnr, "%7d %7d %7d %7d %7d %7.0lf\n", utc_coll_date.year, utc_coll_date.month, utc_coll_date.day,
                  utc_coll_time.hour,  utc_coll_time.minute, utc_coll_time.second );
   fprintf( bnr, "%s\n", title );
   fprintf( bnr, "%15.6lf %15.6lf %15.6lf %15d\n", fstartw, fstopw, spacing, fnum_points );
   for( i=0; i< fnum_points; i++ ) {
      fprintf( bnr, "%16.6f\n", ydata[i] );
   }

   fclose( bnr );

	fprintf( stderr, "  Closed t15asc file: %s\n", t15name );

	free(ydata);

   if(( txt = fopen( "ckopus.out", "wa" )) == (FILE *) NULL) return(7);

	fprintf( txt, "%09.4lf\n", res.zen );
	fprintf( txt, "%09.4lf\n", 0.9/AQPB.RES );
	fprintf( txt, "%05.3lf\n", field_of_view );
	fprintf( txt, "%s\n", FTPB.APF );
	fprintf( txt, "%0.lf\n", fstartw );
	fprintf( txt, "%0.lf\n", fstopw );
	fprintf( txt, "%04i %02i %02i %02i %02i %4.1lf\n",
	         utc_coll_date.year, utc_coll_date.month,  utc_coll_date.day,
				utc_coll_time.hour, utc_coll_time.minute, utc_coll_time.second );
	fprintf( txt, "%s\n", title );
	fprintf( txt, "%010.4lf\n", res.az );
        fprintf( txt, "%010.1lf\n", INPB.DUR );
	fprintf( txt, "%010.4lf\n", res.roe );

	fclose( txt );

return(0);

}


int write_bnr( char *blocktype, int SWAP, int ftype ) {

   FILE *bnr, *txt;
   char title[80], bnrname[12], cdum[ENUMSZ];
   int fnum_points, length, pointer, rc, i;
   float *ydata;
   double fstartw, fstopw, spacing;

   // time taken care of in initialize
   strncpy( title, bnr_makeheader( utc_coll_date, utc_coll_time, res,
                        INPB.DUR, AQPB.RES, FTPB.APF, field_of_view), HEADLEN );

	fprintf( stderr, "  BNR Header                : '%s' %ld\n", title, strlen(title) );

   fstartw     = SPPB.FXV;
   fstopw      = SPPB.LXV;
   fnum_points = SPPB.NPT;
   spacing = (fstopw-fstartw) / (fnum_points-1);
   if( spacing < 0.0 ) spacing = (fstartw-fstopw) / (fnum_points-1);

   fprintf( stderr, "  Data range                : %12.4lf to %12.4lf %8d points spaced at %12.8lf\n", fstartw, fstopw, fnum_points, spacing );
	fprintf( stderr, "  Reading spectra block     : %Xx\n", datablocktype );

   if( ! opus_searchdir( opus, datablocktype, &length, &pointer, SWAP )) {
      fprintf( stderr," *Cannot find data block    : %s %Xx\n", blocktype, datablocktype );
      if( ! opus_searchdir( opus, datablocktype + OFFSET, &length, &pointer, SWAP )) {
         fprintf( stderr," *Cannot find data block    : %s %Xx\n", blocktype, datablocktype + OFFSET );
         fclose( opus );
         return(1);
      } else fprintf( stderr,"  Data block Found          : %s 0x%x\n", blocktype, datablocktype + OFFSET );
   } else fprintf( stderr,"  Data block Found          : %s 0x%x\n", blocktype, datablocktype );

	ydata = (float *) malloc ( length*sizeof(float) );

	if( ydata == (float *) 0 ) {
		fprintf( stderr, " *Could not allocate array for y data.\n" );
		return(2);
	}

	if( opus_read_spectrum( opus, &fstartw, &fstopw, length, pointer, SPPB.CSF, SPPB.NPT, ydata, SWAP )) {
		fprintf( stderr," *Cannot read data.\n" );
		return(3);
	}

   if( fstartw > fstopw ){
      fprintf( stderr," For .bnr file reversing  to low to high wavenumber order\n" );
      double temp = fstartw;
      fstartw = fstopw;
      fstopw = temp;
      float *ytmp;
      ytmp = (float *) malloc ( length*sizeof(float) );
      for(i=0; i<fnum_points; i++ )ytmp[fnum_points-i] = ydata[i];
      ydata = ytmp;
   }

// 	fprintf( stderr, "\tFirst Data points\n");
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+(0*spacing),ydata[0] ) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+(1*spacing),ydata[1] ) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+(2*spacing),ydata[2] ) ;
// 	fprintf( stderr, "\n\tLast Data points\n");
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+((length-3)*spacing), ydata[length-3]) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+((length-2)*spacing), ydata[length-2]) ;
// 	fprintf( stderr, "\t%11.6f %11.9e\n", fstartw+((length-1)*spacing), ydata[length-1]) ;

	sprintf( bnrname, "%4s.bnr", blocktype );
	fprintf( stderr, "  Opening bnr file          : %s\n", bnrname );

	if( bnr_write_init( &bnr, bnrname )) {
		fprintf( stderr, " *Cannot create %s bnr file.\n", bnrname );
		return(4);
	}

   if( ftype == 'F' ) {
      printf("  Writing FORTRAN file\n");
      if( bnr_write_header_F( bnr, title, fstartw, fstopw, spacing, length )) {
         fprintf( stderr, " *Cannot write header of %s bnr file.\n", bnrname );
         return(5);
      }

      if( bnr_write_data_F( bnr, ydata, length )) {
         fprintf( stderr, " *Cannot write data of %s bnr file.\n", bnrname );
         return(6);
      }

   } else {
      printf("   Writing C file\n");
      if( bnr_write_header( bnr, title, fstartw, fstopw, spacing, length )) {
         fprintf( stderr, " *Cannot write header of %s bnr file.\n", bnrname );
         return(5);
      }

      if( bnr_write_data( bnr, ydata, length )) {
         fprintf( stderr, " *Cannot write data of %s bnr file.\n", bnrname );
         return(6);
      }
   }

	bnr_write_done( bnr );
	fprintf( stderr, "  Closed bnr file: %s\n", bnrname );

	free(ydata);

   if(( txt = fopen( "ckopus.out", "wa" )) == (FILE *) NULL) return(7);

	fprintf( txt, "%09.4lf\n", res.zen );
	fprintf( txt, "%09.4lf\n", 0.9/AQPB.RES );
	fprintf( txt, "%05.3lf\n", field_of_view );
	fprintf( txt, "%s\n", FTPB.APF );
	fprintf( txt, "%0.lf\n", fstartw );
	fprintf( txt, "%0.lf\n", fstopw );
	fprintf( txt, "%04i %02i %02i %02i %02i %4.1lf\n",
	         utc_coll_date.year, utc_coll_date.month,  utc_coll_date.day,
				utc_coll_time.hour, utc_coll_time.minute, utc_coll_time.second );
	fprintf( txt, "%s\n", title );
	fprintf( txt, "%010.4lf\n", res.az );
        fprintf( txt, "%010.1lf\n", INPB.DUR );
	fprintf( txt, "%010.4lf\n", res.roe );

	fclose( txt );

return(0);

}

int showtime( char *filename, int SWAP ) {

   int rc=0;

   if(( rc = opus_init( &opus, filename, SWAP )) != FALSE ) return(rc);

   printf( "OPUS DAT & TIM strings:\n" );
   if( read_SPPAR_block(  opus,IFGMPARBLK, &IFPB, SWAP ))
      printf( " IFGM: %s %s\n", IFPB.DAT, IFPB.TIM );

   if( read_SPPAR_block( opus, SNGCPARBLK, &SCPB, SWAP ))
      printf( " SNGC: %s %s\n", SCPB.DAT, SCPB.TIM );

   if( read_SPPAR_block( opus, EMISPARBLK, &EMPB, SWAP ))
      printf( " EMIS: %s %s\n", EMPB.DAT, EMPB.TIM );

   if( read_SPPAR_block( opus, PHASPARBLK, &PHPB, SWAP ))
      printf( " PHAS: %s %s\n", PHPB.DAT, PHPB.TIM );

   if( read_SPPAR_block( opus, SNG2PARBLK, &S2PB, SWAP ))
      printf( " SNG2: %s %s\n", S2PB.DAT, S2PB.TIM );

   if( read_SPPAR_block( opus, IFG2PARBLK, &I2PB, SWAP ))
      printf( " IFG2: %s %s\n", I2PB.DAT, I2PB.TIM );

   if( read_SPPAR_block( opus, TRANPARBLK, &TRAN, SWAP ))
      printf( " TRAN: %s %s\n", TRAN.DAT, TRAN.TIM );

   printf("\n");
   fclose( opus );

   return(0);
}


int initialize( char *filename, char *blocktype, char *tblocktype, int SWAP ) {

   int    rc=0, flt=0, fl1=0;
   char   filter[]="filter", timblk[6];
   double bruker_time_adjustment = 0;

   fprintf( stderr, " File: <%s>\n", filename );

   if(( rc = opus_init( &opus, filename, SWAP )) != FALSE ) return(rc);
//   printf( "init rc : %i\n", rc );
//   if( 1 ) printf( "init rc : %i\n", 1 );
//   if( ! 0 ) printf( "init rc : %i\n", 0 );
//   if( -1 ) printf( "init rc : %i\n", -1 );
//   if( 10010011 ) printf( "init rc : %i\n", 10010011 );

//   if( rc ) {
//      fprintf( stderr,"Cannot open %s opus file \n",filename );
//      return(1);
//   }

   if( VERBOSE ) {
      fprintf( stderr, "showinfo: file handle value       : %lx\n", (long) opus );
      fprintf( stderr, "showinfo: file handle address     : %lx\n", (long) &opus );
      //fprintf( stderr, "showinfo: file handle pointing to : %lx\n", (int) *opus );
      fprintf( stderr, "showinfo: opus filename           : %s\n", filename );
   }

   if( read_AQPAR_block( opus, &AQPB, SWAP )) {
      fprintf( stderr,"  Acquisition Param Block Read OK.\n" );
      fprintf( stderr,"     NSS, GSW, GSG          :  %16d %16d %16s\n", AQPB.NSS, AQPB.GSW, AQPB.GSG );
      fprintf( stderr,"     RES, LFW, HFW          :  %16.6lf %16.1lf %16.1lf\n", AQPB.RES, AQPB.LFW, AQPB.HFW );
      fprintf( stderr,"     AQM                    :  %16s\n", AQPB.AQM );
   } else {
      fprintf( stderr,"  Acquisition Param Block Read NOT OK!\n" );
      //return(3);
      }

   if( read_FTPAR_block( opus, &FTPB, SWAP )) {
      fprintf( stderr,"  FT Param Block Read OK.\n" );
      fprintf( stderr,"     LFQ, HFQ, APF          :  %16.4lf %16.4lf %16s\n", FTPB.LFQ, FTPB.HFQ, FTPB.APF );
      fprintf( stderr,"     PHR, ZFF, PHZ          :  %16lf %16s %16s\n", FTPB.PHR, FTPB.ZFF, FTPB.PHZ );
   } else {
      fprintf( stderr,"  FT Param Block Read NOT OK!\n" );
      //return(4);
      }

   if( read_OPPAR_block(opus, &OPPB, SWAP )) {
      fprintf( stderr,"  Optic Param Block Read OK.\n" );
      fprintf( stderr,"     APT, PGN, BMS          :  %16s %16s %16s\n", OPPB.APT, OPPB.PGN, OPPB.BMS );

      // Get filter id if possible
      // no date yet... filterid();
      fprintf( stderr,"     OPF, OF1, FltID        :  %16s %16s %16s\n\n", OPPB.OPF, OPPB.OF1, fltr );
      fprintf( stderr,"     DTC                    :  %16s\n", OPPB.DTC );
      fprintf( stderr,"     SRC                    :  %16s\n", OPPB.SRC );
      fprintf( stderr,"     VEL, LPF, HPF          :  %16s %16s %16s\n", OPPB.VEL, OPPB.LPF, OPPB.HPF );
      sscanf(OPPB.APT," %f ",&aperture); //,units); // in mm
      sscanf(OPPB.OPF," %s %i ", filter, &flt);  // "Filter n ...
      sscanf(OPPB.OF1," %s %i ", filter, &fl1);  // "Filter n ...
      //printf( "%s\n", strpbrk( OPPB.APT, "0123456789" ));
      //printf( "filter # %i\n", flt);
   } else {
      fprintf( stderr,"  Optic Param Block Read NOT OK!\n" );
      //return(5);
   }

   field_of_view = 0.0;
   if( read_INPAR_block( opus, &INPB, SWAP )) {
      fprintf( stderr,"  Instrument Param Block Read OK.\n" );
      fprintf( stderr,"     DUR, ASG, LWN          :  %16.4lf %16d %16lf\n", INPB.DUR, INPB.ASG, INPB.LWN );
      fprintf( stderr,"     ASS, INS,              :  %16d %16s\n", INPB.ASS, INPB.INS );
      fprintf( stderr,"     LFL, HFL, SSP          :  %16.4lf %16.4lf %16d\n", INPB.LFL, INPB.HFL, INPB.SSP );
      fprintf( stderr,"     LFF, HFF, DFR          :  %16.4lf %16.4lf %16d\n", INPB.LFF, INPB.HFF, INPB.DFR );
      if( INPB.FOC != 0.0 ) field_of_view = aperture/INPB.FOC * 1000.; //milli-radians
      if( field_of_view < 0.0 )field_of_view = -0.999;
      fprintf( stderr,"     FOV, FOC, ABP          :  %16.4lf %16.4lf %16d\n", field_of_view, INPB.FOC, INPB.ABP );
      if( INPB.GFW > 0 )
      fprintf( stderr,"     GFW, PKL, PKA          :  %16d %16d %16d\n", INPB.GFW, INPB.PKL, INPB.PKA );
      if( INPB.GBW > 0 )
      fprintf( stderr,"     GBW, PRL, PRA          :  %16d %16d %16d\n", INPB.GBW, INPB.PRL, INPB.PRA );
   } else {
      fprintf( stderr,"  Instrument Param Block Read NOT OK!\n" );
      //return(6);
      }

   if( read_SPPAR_block(  opus, IFGMPARBLK, &IFPB, SWAP )) {
      fprintf( stderr,"  Interferogram Param Block Read OK.\n" );
      fprintf( stderr,"     Date and Time String   :  %16s %16s\n", IFPB.DAT, IFPB.TIM );
      fprintf( stderr,"     MNY, MXY, CSF          :  %16.4lf %16.4lf %16.4lf\n", IFPB.MNY, IFPB.MXY, IFPB.CSF );
      fprintf( stderr,"     NPT, DXU               :  %16d %16s\n", IFPB.NPT, IFPB.DXU );
   } else {
      fprintf( stderr,"  Interferogram Param Block Read NOT OK!\n" );
      //return(7);
   }

   // get spectra parameter block for the looksee
   rc = getSPblock( filename, blocktype, SWAP );
   if( rc != 0 ) {
     fclose( opus ) ;
     return(rc);
   }

//    printf(" blocktype        %10s\n", blocktype );
//    printf(" paramblocktype   %10Xx\n", paramblocktype );
//    printf(" datablocktype    %10Xx\n", datablocktype );

   fprintf( stderr,"     Date and Time String   :  %16s %16s\n", SPPB.DAT, SPPB.TIM );
   fprintf( stderr,"     FXV, LXV, NPT          :  %16.4lf %16.4lf %16d\n", SPPB.FXV, SPPB.LXV, SPPB.NPT );
   fprintf( stderr,"     MNY, MXY, CSF          :  %16.4lf %16.6lf %16.1lf\n", SPPB.MNY, SPPB.MXY, SPPB.CSF );
   fprintf( stderr,"     NF1, NF2, SN1          :  %16.4lf %16.4lf %16.1lf\n", SPPB.NF1, SPPB.NF2, SPPB.SN1 );

   // Get spectra parameter block for the time
   rc = getSPblock( filename, tblocktype, SWAP);
   if( rc != 0 ) {
     fclose( opus ) ;
     return(rc);
   }

   fprintf( stderr,"  Using DATE & TIME from Param Block : %s\n", specblock );
   fprintf( stderr,"     Date and Time String   :  %16s %16s\n", SPPB.DAT, SPPB.TIM );

   // Get date and time into structures, from strings
   // Assume dd/mm/yyyy
   sscanf( SPPB.DAT, "%d/%d/%d", &collection_date.day, &collection_date.month, &collection_date.year );
   // Check for yyyy/mm/dd
   if( collection_date.day > 31 && collection_date.year < 32 ){
      sscanf( SPPB.DAT, "%d/%d/%d", &collection_date.year, &collection_date.month, &collection_date.day );
      //printf("%d  %d  %d\n", collection_date.day, collection_date.month, collection_date.year );
   }
   sscanf( SPPB.TIM, "%d:%d:%lf", &collection_time.hour, &collection_time.minute, &collection_time.second );
   strcpy( timblk, specblock );

   // Apply correction to Bruker time + clock (or empirical Bruker correction from site.offs_secs )
   starttime( &bruker_time_adjustment );
   adjust_date_time( &collection_date, &collection_time, bruker_time_adjustment );
   offs = bruker_time_adjustment;         // seconds

   //printf(" brk offset %lf\n", bruker_time_adjustment );

   utc_coll_date = collection_date;
   utc_coll_time = collection_time;

   // Apply UTC offset correction
   if( applyUTC == 1 ) {
      adjust_date_time( &utc_coll_date, &utc_coll_time, site.utc_offs*3600 );
      offs = (offs + site.utc_offs*3600)/3600.;
    } else {
      offs = offs/3600.;                     // hours, sum of 3 possible time offsets
    }

   // Calculate zenith angle
   az_and_zen( &utc_coll_date, &utc_coll_time, site.n_lat, site.w_lon, &res );

	// Calculate the radius of the earth ellipsoid at observation lat & solar azimuth
	rad_of_earth( site.n_lat, site.w_lon, &res );

   fprintf( stderr,"\n  UTC Date Time from %s block: %04d%02d%02d  %02d:%02d:%02.0lf\n", timblk,
      utc_coll_date.year,   utc_coll_date.month,    utc_coll_date.day,
      utc_coll_time.hour,  utc_coll_time.minute, utc_coll_time.second );

   fprintf( stderr, "      Site, UTC Offset, JD  :  %16s %16d %16.7f\n", site.tag, site.utc_offs, res.jd );
   fprintf( stderr, "     N Lat, W Lon, Alt      :  %16.2lf %16.2lf %16.1f\n", site.n_lat, site.w_lon, site.altud );
   fprintf( stderr, "     S Azi, S Zen, RoE      :  %16.4lf %16.4lf %16.4lf\n\n", res.az, res.zen, res.roe );

   // Get filter id if possible
   filterid();
   fprintf( stderr,"       OPF, OF1, FltID      :  %16s %16s %16s\n\n", OPPB.OPF, OPPB.OF1, fltr );

   // Get spectra parameter block for the spectrum
   rc = getSPblock( filename, blocktype, SWAP );
   if( rc != 0 ) {
     fclose( opus ) ;
     return(rc);
   }

   return(0);

}


int showdb( char *filename ) {

   float tstamp=0.0;

   tstamp =  utc_coll_time.hour*10000 + utc_coll_time.minute*100 + (float)utc_coll_time.second;

   printf( "%s %3s %4s %7.4f %06.0f %04d%02d%02d  %02d:%02d:%02.0lf  ", filename, site.tag, specblock, offs, tstamp,
      utc_coll_date.year, utc_coll_date.month,    utc_coll_date.day,
      utc_coll_time.hour,   utc_coll_time.minute, utc_coll_time.second );

   //printf( "%.2e\t%.2lf\t%.2lf\t%.1f\t",
   printf( "%7.1lf %7.2lf  %8.2lf  %6.1f  ", SPPB.SN1, site.n_lat, site.w_lon, site.altud );

   //printf( "%.2lf\t%.2lf\t%.4lf\t%6.2lf\t%.4lf\t%s\t%.4lf\t%.3lf\t%.3lf\t%i\t%.3le\t%.3le\n",
   printf( "%8.2lf  %6.2lf  %9.4lf  %6.2lf  %6.4lf  %2s  %6.4lf  %9.3lf  %9.3lf  %1s  %.3le  %.3le  %3i %3i %3i %3i\n",
      res.az, res.zen, res.roe, INPB.DUR, AQPB.RES, FTPB.APF, field_of_view,
      SPPB.FXV, SPPB.LXV, fltr, SPPB.MXY, SPPB.MNY, INPB.ASS, AQPB.NSS, INPB.GFW, INPB.GBW );

   return(0);

}

int showdbhead() {

   printf("Filename Site SBlock TOffs TStamp Date Time SNR N_Lat W_Lon Alt SAzm SZen ROE Dur Reso Apd FOV LWN HWN Flt MaxY MinY FLSCN EXSCN GFW GBW\n");

   return(0);

}


int opusmws( char *filename, int SWAP ) {

// this filename is not actual opus filename

	FILE   *inp, *out, *spc;

	char   path[80], mwfn[80][40], buf[80];
	char   blocktype[] = "NONE\0", tblocktype[] = "NONE\0";

	int    j, n1, n2, i, length, pointer, fnum_points, ftype, nmws, rc;

	float  *ydata, temp;
	float  aperature;

	double fstartw, fstopw, spacing;
	double w, scan_time, resolution, focal_length;
	double sum, mws[2][40];
	double field_of_view;

	if(( out = fopen( "cutmws.out", "w" )) == (FILE *) NULL){
	   fprintf( stdout, "Cannot open %s output file.\n", "cutmws.out" );
	   exit(1);
	}

	if(( inp = fopen( "cutmws.inp", "r" )) == (FILE *) NULL){
	   fprintf( stdout, "Cannot open %s input file.\n", "cutmws.inp" );
	   exit(1);
	}

	skipComment( inp );

	/*  Read input file name and type */
	fscanf( inp, "%s", filename );
	fscanf( inp, "%d", &ftype );
	fprintf( out,"Opus file and type %s : %d\n", filename, ftype );
	switch( ftype )
	{
	case 0:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	 case 1:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	 case 2:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	 case 3:
		  fprintf( out, "Opening OPUS file...\n" );
		  break;
	 default:
		  fprintf( out, "Only support OPUS files exiting...\n" );
		  exit(0);
		  break;
	}

	skipComment( inp );

	/* Read number of micro windows */
	fscanf( inp, "%d", &nmws );
	fprintf( out,"# mws : %d\n", nmws );

	skipComment( inp );

	/* Read micro windows */
	/*	for( i=0; i<nmws; i++ ) fscanf( input, "%lf, %lf %s", &mws[0][i], &mws[1][i], rem );*/
	for( i=0; i<nmws; i++ ) fscanf( inp, "%lf,%lf", &mws[0][i], &mws[1][i] );
	for( i=0; i<nmws; i++ ) fprintf( out, "%12.6lf  %12.6lf\n", mws[0][i], mws[1][i] );
   fprintf( out,"End of mws\n" );

	skipComment( inp );

	/* Read output path */
	fscanf( inp, "%s", path );
	fprintf( out, "%s\n", path );

	skipComment( inp );

	/* Read output file names */
	for( i=0; i<nmws; i++ ) fscanf( inp, "%s", mwfn[i] );
	for( i=0; i<nmws; i++ ) fprintf( out, "%s\n", mwfn[i] );
	for( i=0; i<nmws; i++ ) {
		strcpy( buf, path );
		strcat( buf, mwfn[i] );
		strcpy( mwfn[i], buf );
	 }
   fprintf( out,"End of mws filenames\n" );

	for( i=0; i<nmws; i++ ) fprintf( out, "%s\n", mwfn[i] );
   fprintf( out,"End of mws path/filenames\n" );

	fclose( inp );

	if(( rc = initialize( filename, blocktype, tblocktype, SWAP )) != 0 ) {
		fclose( opus );
	   fprintf( stderr, " Initialize error rc = %i \n", rc );
	   return(0);
	}

   fstartw     = SPPB.FXV;
   fstopw      = SPPB.LXV;
   fnum_points = SPPB.NPT;
   spacing = (fstopw-fstartw) / (fnum_points-1);
   if( spacing < 0.0 ) spacing = (fstartw-fstopw) / (fnum_points-1);

   fprintf( stderr, "  %12.4lf to %12.4lf %8d points spaced at %12.8lf\n", fstartw, fstopw, fnum_points, spacing );
	fprintf( stderr, "  Reading Spectra data block : %Xx\n", datablocktype );

   if( ! opus_searchdir( opus, datablocktype, &length, &pointer, SWAP )) {
      fprintf( stderr," *Cannot find data block    : %s %Xx\n", blocktype, datablocktype );
      if( ! opus_searchdir( opus, datablocktype + OFFSET, &length, &pointer, SWAP )) {
         fprintf( stderr," *Cannot find data block    : %s %Xx\n", blocktype, datablocktype + OFFSET );
         fclose( opus );
         return(1);
      } else fprintf( stderr,"  Data block Found          : %s 0x%x\n", blocktype, datablocktype + OFFSET );
   } else fprintf( stderr,"  Data block Found          : %s 0x%x\n", blocktype, datablocktype );

	ydata = (float *) malloc ( length*sizeof(float) );

	if( ydata == (float *) 0 ) {
		fprintf( stderr, " *Could not allocate array for y data.\n" );
		fclose( opus );
		return(2);
	}

	if( opus_read_spectrum( opus, &fstartw, &fstopw, length, pointer, SPPB.CSF, SPPB.NPT, ydata, SWAP )) {
		fprintf( stderr," *Cannot read data.\n" );
		fclose( opus );
		return(3);
	}
	fclose( opus );

	/* linefit microwindow output: */
	for( i=0; i<nmws; i++ ) {
	  n1 = (int) (( mws[0][i] - fstartw ) / spacing );
	  n2 = (int) (( mws[1][i] - fstartw ) / spacing );
	  fprintf( out, "%3i:  %i %i %lf %lf\n", i, n1, n2, fstartw+n1*spacing, fstartw+n2*spacing );
	  if(( spc = fopen( mwfn[i], "w" )) == (FILE *) NULL) {
		  fprintf( stderr, "Cannot open %s output file.\n", mwfn[i] );
		  exit(1);
	  }
	  // Normalize by mean value... SKIP
	  //sum =0.0;
	  //for( j=n1; j<=n2; j++ ) {
		//	sum = sum + ydata[j] ;
	  //}
	  //sum = sum / (n2-n1+1);
	  //fprintf( out, "Normalize MW by mean value:   : %10.4lf\n", sum );
	  sum = 1.0;
	  for( j=n1; j<=n2; j++ ) {
			fprintf( spc, "%14.6f,%14.6f\n", fstartw + j*spacing, ydata[j]/sum );
	  }
	  fclose( spc );

	}

	free(ydata);

   sscanf(OPPB.APT," %f ",&aperture); //,units); // in mm
   if( INPB.FOC != 0.0 ) field_of_view = aperture/INPB.FOC * 1000.;
   if( field_of_view < 0.0 )field_of_view = -0.999;

	fprintf( out, "Resolution[cm-1]   : %10.4lf\n", AQPB.RES );
	fprintf( out, "Field of View [mr] : %10.4lf\n", field_of_view );
	fprintf( out, "Low Wavenumber     : %10.4lf\n", fstartw );
	fprintf( out, "High Wavenumber    : %10.4lf\n", fstopw );

   return(0);

}



int main( int argc, char *argv[] ) {

   char  *optarg;
   // 4 char string EMIS |SNGC | IFGM | PHAS
   char  blocktype[] = "NONE\0", tblocktype[] = "NONE\0";;
   // 3 char string TAB | MLO | SGP | PKF | KPK | FL0
   char  sitetag[] = "UKN\0";
   char  filename[255];

   int   rc, c, type='\0';
   int   optind=0, SWAP=0, utc=99, sec=0, bflag=0, bbflag=0;
   float wlon=0.0, nlat=0.0, alt=0.0;

   site = setsite( DEFx );

   while( --argc > 0 && (*++argv)[0] == '-' ) {
      while(( c = *++argv[0]) != FALSE )
         switch( c ) {
            case 'w' :
               //printf("%s\n", ++argv[0]);
               if(( wlon = atof( ++argv[0] )) == FALSE ) wlon = 0.0;
               //printf("%f\n", site.w_lon);
               break;
            case 'n' :
               if(( nlat = atof( ++argv[0] )) == FALSE ) nlat = 0.0;
               printf("%f\n", nlat);
               break;
            case 'a' :
               if(( alt  = atof( ++argv[0] )) == FALSE ) alt = 0.0;
               break;
            case 't' :
               if(( sec = atoi( ++argv[0] )) == 0 ) sec = 0;
               //printf( "len argv %d %s\n", strlen( argv[0] ), argv[0] );
               memset( argv[0], '\0', strlen( argv[0] ) );
               break;
            case 'b' :
               bbflag = 1;
               if(( bflag = atoi( ++argv[0] )) == 0 ) bflag = 0;
               //printf( "len argv %d %s\n", strlen( argv[0] ), argv[0] );
               memset( argv[0], '\0', strlen( argv[0] ) );
               break;
            case 'u' :
               if(( utc = atoi( ++argv[0] )) == 0 ) utc = 0;
               break;
            case 's' :
               SWAP = 1;
               break;
            case 'S' :
               strncpy( sitetag, ++argv[0], 3 );
               memset( argv[0], '\0', 3 );
               //printf(" char site : '%s'\n", csite );
               break;
            case 'U' :
               applyUTC=1;
               break;
            case 'C' :
               type = 'C';
               break;
            case 'v' :
               type = 'v';
               break;
            case 'L' :
               type = 'L';
               break;
            case 'B' :
               type = 'B';
               break;
            case 'p' :
               strncpy( tblocktype, ++argv[0], 4 );
               memset( argv[0], '\0', 4 );
               //printf(" block : '%s' %c\n", tblocktype, type );
               break;
            case 'P' :
               type = 'P';
               break;
            case 'D' :
               type = 'D';
               strncpy( blocktype, ++argv[0], 4 );
               memset( argv[0], '\0', 4 );
               //printf(" block : '%s' %c\n", blocktype, type );
               break;
            case 'H' :
               type = 'H';
               break;
            case 'M' :
               type = 'M';
               break;
            case 'R' :
               type = 'R';
               strncpy( blocktype, ++argv[0], 4 );
               memset( argv[0], '\0', 4 );
               //printf(" block : '%s' %c\n", blocktype, type );
               break;
            case 'F' :
               type = 'F';
               strncpy( blocktype, ++argv[0], 4 );
               memset( argv[0], '\0', 4 );
               //printf(" block : '%s' %c\n", blocktype, type );
               break;
            case 'T' :
               type = 'T';
               strncpy( blocktype, ++argv[0], 4 );
               memset( argv[0], '\0', 4 );
               //printf(" block : '%s' %c\n", blocktype, type );
               break;
            case '?' :
            case 'h' :
               printf( "usage: ckopus -wXXX -nXXX -aXXX -tNNN -s -Ssss -[S|L|B|P|R] -[R|F]ssss file1 file2 ...\n" );
               printf( "where:\n" );
               printf( "   -v print the current version\n" );
               printf( "   -w is west longitude defaults to %.2lf [decimal degrees]\n",site.w_lon );
               printf( "   -n is north latitude defaults to %.2lf [decimal degrees]\n", site.n_lat );
               printf( "   -a is altitude defaults to %.2lf [meters]\n", site.altud );
               printf( "   -t is time offset defaults to %i [seconds]\n", site.sec_offs );
               printf( "   -p use this block for time calculations [TRAN | SGN2 | IFG2 | EMIS | IFGM | PHAS | SNGC] ex. -pSNGC \n" );
               printf( "   -b Bruker time calculation flag\n" );
               printf( "      0 - default no time correction\n" );
               printf( "      1 - empirical value given in -t time to block time\n" );
               printf( "      2 - for 2 file forward / backward scheme (OPUS v3.04 OS/2 ~1996) plus -t time\n" );
               printf( "      3 - forward only fast return plus -t time (use ifgm points)\n" );
               printf( "      4 - forward / backward one file (IFS125) plus -t time\n" );
               printf( "      5 - forward only fast return plus -t time (use scantime duration value)\n" );
               printf( "      6 - forward only fast return plus -t time (use scantime duration, file time is end of scan)\n" );
               printf( "   -u is UTC time offset defaults to %i [hours]\n", site.utc_offs );
               printf( "   -U apply UTC time offset\n" );
               printf( "   -s swap bytes on opus read defaults to No Swap\n" );
               printf( "   -S set lat lon & alt for [TAB | FL0 | MLO | KPK | PKF | MSA | SGP | TMK]\n" );
               printf( "   -C short listing of file contents\n" );
               printf( "   -L long listing of file contents\n" );
               printf( "   -B list file blocks\n" );
               printf( "   -P list time stamps\n" );
               printf( "   -M make Linefit microwindow files\n" );
               printf( "   -R write 'C' bnr from type     [TRAN | SGN2 | IFG2 | EMIS | IFGM | PHAS | SNGC] ex. -RSNGC \n" );
               printf( "   -F write FORTRAN bnr from type [TRAN | SGN2 | IFG2 | EMIS | IFGM | PHAS | SNGC] ex. -FEMIS \n" );
               printf( "   -T write T15asc file from type [TRAN | SGN2 | IFG2 | EMIS | IFGM | PHAS | SNGC] ex. -TEMIS \n" );
               printf( "   -D one line param list for database to stdout ex. -DEMIS (use NONE for default block)\n" );
               printf( "   -H print database header to stdout\n" );
               return(0);
          }
   }

   site = setmysite( sitetag );
   if( nlat != 0.0 )site.n_lat    = nlat;
   if( wlon != 0.0 )site.w_lon    = wlon;
   if( alt  != 0.0 )site.altud    = alt;
   if( bbflag == 1 )site.strt_flg = bflag;
   if( utc  != 99  ){
      site.utc_offs = utc;
      applyUTC = 1;
   }
   site.sec_offs = sec;

   //printf(" site strt %d  %d  %d\n", site.strt_flg, bbflag, bflag );
   //printf(" site strt %d\n", site.strt_flg );
   //printf(" site offs %d\n", site.utc_offs );
   //printf(" block : '%s' %c\n", blocktype, type );

   if( type == 'H' ) {
      rc = showdbhead();
   } else if( type == 'v' ) {
       printf("Version : %s\n", version );
   } else if( argc <= 0 ) {
      fprintf( stderr, "You must also enter an OPUS data file name(s).\n" );
      fprintf( stderr, "STOP.\n" );
      return( 0 );
   } else {
      while( argc-- > 0 ) {

         strcpy( &filename[0], (argv++)[0] );
         if( VERBOSE ) fprintf( stderr, "main: filename: %s\n", filename );

         switch( type ) {
            case 'L' :
               rc = opus_listParams( filename, SWAP );
            break;
            case 'B' :
               rc = opus_listBlocks( filename, SWAP );
            break;
            case 'P' :
               rc = showtime( filename, SWAP );
            break;
            case 'M' :
               rc = opusmws( filename, SWAP );
            break;
            case 'C' :
                if(( rc = initialize( filename, blocktype, tblocktype, SWAP )) == 0 ) {
                  fclose( opus );
               } else printf( " Initialize error rc = %i \n", rc );
            break;
            case 'D' :
               if(( rc = initialize( filename, blocktype, tblocktype, SWAP )) == 0 ) {
                  fclose( opus );
                  rc = showdb( filename );
               }  else printf( " Initialize error rc = %i \n", rc );
            break;
            case 'R' :
               if(( rc = initialize( filename, blocktype, tblocktype, SWAP )) == 0 ) {
                  rc = write_bnr( blocktype, SWAP, 'C' );
                  fclose( opus );
               } else printf( " Initialize error rc = %i \n", rc );
            break;
            case 'F' :
               if(( rc = initialize( filename, blocktype, tblocktype, SWAP )) == 0 ) {
                  rc = write_bnr( blocktype, SWAP, 'F' );
                  fclose( opus );
               } else printf( " Initialize error rc = %i \n", rc );
            break;
            case 'T' :
               if(( rc = initialize( filename, blocktype, tblocktype, SWAP )) == 0 ) {
                  rc = write_t15( blocktype, SWAP );
                  fclose( opus );
               } else printf( " Initialize error rc = %i \n", rc );
            break;
         }
      }
   }
   //printf("end %d\n", rc);
   return(rc);
}
