// mesuse.c
/*
   sub-routines coded directly from:
   "Astronomical Algorithms" by Jean Meeus, 1st English Ed.
   @1991 Willmann-Bell ISBN 0-943396-35-2
   by jwh July 2002

*/

#include"constant.h"
#include"meeus.h"

#include<stdlib.h>
#include<stdio.h>
#include<math.h>
#include<string.h>

#define FALSE 0
#define TRUE 1
#define VERBOSE FALSE

// Jan 2011
// added date_jul() to convert a julian date back to a date & time


// convert a julian day to a date
// store in our structs
// from Meeus by way of http://www.christophedavid.org

int date_jul( double jd, struct utc_date *d, struct utc_time *t ) {

   int rc = 0;

   double do1     = (double) 0;
   double do2     = (double) 0;
   double do3     = (double) 0;
   double doZ     = floor(jd + (double) 0.5);
   double doF     = jd + (double) 0.5 - doZ;
   double doA     = (double) 0;
   double doAlpha = (double) 0;
   double doB     = (double) 0;
   double doC     = (double) 0;
   double doD     = (double) 0;
   double doE     = (double) 0;

   if (doZ < (double) 2299161) {
      doA = doZ;
      }
   else {
      doAlpha = floor((doZ - (double) 1867216.25) / (double) 36524.25);
      doA = doZ + (double) 1 + doAlpha - floor(doAlpha / (double) 4);
      }

   doB = doA + (double) 1524;
   doC = floor((doB - (double) 122.1) / (double) 365.25);
   doD = floor(doC * (double) 365.25);
   doE = floor((doB - doD) / (double) 30.6001);

   // date
   do1   = ((doB - doD) - floor(doE * (double) 30.6001)) + doF;
   do2 = (double) (doE < (double) 14 ? doE - (double)  1
                                     : doE - (double) 13);
   do3  = (double) (do2 > (double) 2 ? doC - (double) 4716
                                    : doC - (double) 4715);

   // drop fractional parts
   d->year  = (int) floor(do3);
   d->month = (int) floor(do2);
   d->day   = (int) floor(do1);

   // time
   do2        = (double) 0.0;
   do1        = modf(jd - (double) 0.5, &do2) * (double) 86400;
   t->hour    = (int) floor(do1 / (double) 3600);
   do1       -= (t->hour * (double) 3600);
   t->minute  = (int) floor(do1 / (double) 60);
   t->second  = do1 - ( (double) t->minute * (double) 60);

   return rc;

}


//  jul_date returns the instanteanous julian date by including the time of day
//  it also returns the day of year in date structure

double jul_date( struct utc_date *d, struct utc_time *t ) {

   // given the UT date and time retrn the julian day

   int      A, B, C, D, y, m;
   double jd, dptr;

   if( d->month <= 2 ) {
      y = d->year - 1;
      m = d->month + 12;
   } else {
      y = d->year;
      m = d->month;
   }

   modf( y/100, &dptr );
   A = (int) dptr;

   modf( A/4, &dptr );
   B = 2 - A + (int) dptr;

   if( VERBOSE ) printf( "A, B     : %i,   %i \n", A, B );

   modf( 365.25*(y+4716), &dptr );
   C = (int) dptr;

   modf( 30.601*(m+1), &dptr );
   D = (int) dptr;

   if( t->frac < 1 ) t->frac = (double) (t->hour + (t->minute + t->second/60.0)/60.0)/24.0;

   jd = (double) ( C + D + d->day + t->frac + B - 1524.5 );

   if( jd <= 2299160.5 ) {
      B = 0;
      jd = (double) ( C + D + d->day + t->frac + B - 1524.5 );
      }

   /* calculate the day of the year by subtracting jd( year, 1,1 ) from the jd */

   y = d->year - 1;
   m = 13;

   modf( y/100, &dptr );
   A = (int) dptr;

   modf( A/4, &dptr );
   B = 2 - A + (int) dptr;

   if( VERBOSE ) printf( "A, B     : %i,   %i \n", A, B );

   modf( 365.25*(y+4716), &dptr );
   C = (int) dptr;

   modf( 30.601*(m+1), &dptr );
   D = (int) dptr;

   /* gregorian or julian calender date given - after 15 October 1582 the gregorian */
   if( jd <= 2299160.5 ) {
      B = 0;
   }

   d->doy = (int) ( jd - (double) ( C + D + 1 + 0 + B - 1524.5 )); // +1 maybe?

   /*printf( "Julian day  2   : %20.9lf\n", jd ); */

   return jd;

   }


/*
* mod(tmp, modulus) computes the value of tmp modulo the integer modulo.
* For example,
*
*  mod(27.3789, 23) = 24.3789
*/

double mod(double tmp, int modulus) {

   double tmp_mod;
   double tmp_frac;

   if(tmp >= 0)
   {
      tmp_frac = tmp - (int)tmp;
      tmp_mod = (((int)tmp)%modulus) + tmp_frac;
      return tmp_mod;
   }
   else
   {
      tmp_frac = (int)tmp - tmp;
      tmp_mod = modulus - ((-(int)tmp)%modulus) - tmp_frac;
      if(tmp_mod >= (double)modulus)
         tmp_mod -= modulus;
      return tmp_mod;
   }
}

void reduce( double * angle ) {
   *angle = fmod( *angle, 360.0 );
   if( *angle < 0.0 ) *angle = *angle + 360.0;
   }

/*
* az_and_zen() computes the azimuth and zenith angles given the date, the
* UTC, and the location.
* Azimuth is +'ve W of S
*/

void az_and_zen( struct utc_date *d, struct utc_time *t,
             double n_lat, double w_lon, struct coord *r) {

   double d2r, tmp, T, L0, M, Mr, e, C, theta, v, eps0, eps, R, omega, lamda;
   double jd, ra, dec, H, h, A, el, az, ST;

   r->lat = n_lat;
   r->lon = w_lon;

   d2r = PI/180.0;

   t->frac = (double) (t->hour + (t->minute + t->second/60.0)/60.0)/24.0;

   jd = jul_date(d, t);
   r->jd = jd;
   if( VERBOSE ) printf( "Julian day              : %20.9lf\n", jd );

   T = (jd-2451545.0)/36525.0;
   if( VERBOSE ) printf( "Julian Century (J2000.0)   : %20.9lf\n", T );

   L0 = 280.46645 + ( 36000.76983 + 0.0003032*T )*T ;
   reduce( &L0 );
   if( VERBOSE ) printf( "Mean solar longitude       : %20.9lf\n", L0 );

   M = 357.52910 + ( 35999.05030 - ( 0.0001559 + 0.00000048*T )*T )*T;
   reduce( &M );
   if( VERBOSE ) printf( "Mean anomoly            : %20.9lf\n", M);

   e = 0.016708617 - ( 0.000042037 + 0.0000001236*T )*T;
   if( VERBOSE ) printf( "Earth's orbit eccentricity : %20.9lf\n", e );

   C = ( 1.914600 - ( 0.004817 + 0.000014*T )*T )*sin( M*d2r ) +
      ( 0.019993 - 0.000101*T )*sin( 2.0*M*d2r ) +
      ( 0.000290 )*sin( 3.0*M*d2r );
   if( VERBOSE ) printf( "Solar equator of center    : %20.9lf\n", C );

   theta = L0 + C;
   reduce( &theta );
   if( VERBOSE ) printf( "True solar longitude       : %20.9lf\n", theta );

   v = M + C;
   reduce( &v );
   if( VERBOSE ) printf( "True solar anomaly      : %20.9lf\n", v );

   R = ( 1.000001018 * ( 1.0 - e*e )) / ( 1 + e*cos( v*d2r ));
   if( VERBOSE ) printf( "Earth-Solar radius (AU)    : %20.9lf\n", R );

   omega = 125.04 - 1934.136*T;
   reduce( &omega );
   if( VERBOSE ) printf( "longitude correction angle : %20.9lf\n", omega );

   lamda = theta - 0.00569 - 0.00478*sin ( omega*d2r );
   reduce( &lamda );
   if( VERBOSE ) printf( "Appearant solar longitude  : %20.9lf\n", lamda );

   eps0 = 23.439291111 - ( 0.013004167 + ( 0.000000164    - 0.000000504*T )*T )*T;
   if( VERBOSE ) printf( "Ecliptic obliquity      : %20.9lf\n", eps0 );

   eps = eps0 + 0.00256*cos( omega*d2r );
   if( VERBOSE ) printf( "Corrected ecliptic obliquity : %18.9lf\n", eps );


/* apparant angles... */
   ra = atan2( cos( eps*d2r )*sin( lamda*d2r ), cos( lamda*d2r ))/d2r;
   reduce( &ra );
   if( VERBOSE ) printf( "Apparant right ascension  : %20.9lf\n", ra );

   dec = asin( sin( eps*d2r )*sin( lamda*d2r ))/d2r;

   if( VERBOSE ) printf( "Apparant declination      : %20.9lf\n", dec );


/* true geometric angles */
   ra = atan2( cos( eps0*d2r )*sin( theta*d2r ), cos( theta*d2r ))/d2r;
   reduce( &ra );
   if( VERBOSE ) printf( "Geometric right ascension  : %20.9lf\n", ra );
   r->ra = ra;

   dec = asin( sin( eps0*d2r )*sin( theta*d2r ))/d2r;
   if( VERBOSE ) printf( "Geometric right ascension  : %20.9lf\n", dec );
   r->dec = dec;

/* mean siderial time in degrees at Greenwich */
   ST = 280.46061837 + 360.98564736629*( jd-2451545.0 ) +
       0.000387933*T*T - T*T*T/38710000.0;
   if( VERBOSE ) printf( "Sidereal time (Greenwich)  : %20.9lf\n", ST );
   reduce( &ST );
   if( VERBOSE ) printf( "Sidereal time (Greenwich)  : %20.9lf\n", ST );

   if( VERBOSE ) printf( "West longitude          : %20.9lf\n", w_lon );
   /*   if( VERBOSE ) printf( "ra          : %20.9lf\n", ra ); */

/* hour angle in degrees positive W of S */
   H = ST - w_lon - ra;
   reduce( &H );
   if( VERBOSE ) printf( "Hour angle (W of S)        : %20.9lf\n", H );

   az = atan2( sin( H*d2r ), ( cos( H*d2r )*sin( n_lat*d2r ) -
                        tan( dec*d2r ) * cos( n_lat*d2r )))/d2r;
   reduce( &az );
/* put az in ± 180° W + or E - of S */
//    if( az > 180.0) az = az - 360.0;
   r->az = az;

   if( VERBOSE ) printf( "Azimuth                 : %20.9lf\n", az );

   el = asin( sin( n_lat*d2r ) * sin( dec*d2r ) +
           cos( n_lat*d2r ) * cos( dec*d2r ) * cos( H*d2r ))/d2r;
   r->zen = 90.0 - el;

   if( VERBOSE ) printf( "Elevation            : %20.9lf\n", el );
   if( VERBOSE ) printf( "Zenith               : %20.9lf\n", r->zen );
   if( VERBOSE ) printf( "time        :    %lf  %i  %02i:%02i02%lf\n", t->frac, t->mday, t->hour, t->minute, t->second );

}

/* calculate the radius of the earth given a latitude, longitude and a bearing */

void rad_of_earth( double n_lat, double w_lon, struct coord *r ) {

   double a;            // km equater
   double b;            // km N Pole
   double M;            // N-S radius at given latitude
   double N;            // E-W radius at given latitude
   double q, p, d2r;

   a = 6378.135;
   b = 6356.766;
   d2r = PI/180.0;

   q = pow( a*cos(n_lat*d2r), 2.0 );
   p = pow( b*sin(n_lat*d2r), 2.0 );

   //printf( "%lf %lf %lf %lf %lf\n", n_lat, a, b, p, q );

   M = pow( a*b, 2.0 ) / pow( q + p, 1.5 );

   N = pow( a, 2.0 ) / pow( q + p, 0.5 );

   // Azimuth required to be East of North
   q = r->az + 180.0;
   if( q > 360.0 ) q = q -360.0;

   r->roe = 1.0 / ( pow( cos(q*d2r), 2.0 ) /M + pow( sin(q*d2r), 2.0) /N );

   //printf( "%lf %lf %lf %lf %lf\n", M, N, r->az, q, r->roe );

}


void adjust_date(struct utc_date *c_date)
{
   int days_this_month;

   days_this_month = days_in_month(c_date->month,c_date->year);

   if (c_date->day < 1) {
      /* decrement month and possibly year */
      if (c_date->month == 1) {
         c_date->month=12;
         c_date->year--;
         c_date->day = 31;
      } else {
         c_date->month--;
         c_date->day = days_in_month(c_date->month,c_date->year);
      }
   } else if (c_date->day > days_this_month) {
      /* increment month and possible year */
      if (c_date->month == 12) {
         c_date->month=1;
         c_date->year++;
      } else {
         c_date->month++;
      }
      c_date->day = 1;
   }
}

void adjust_date_time( struct utc_date *c_date, struct utc_time *c_time,
                       long int adjustment) {

   long int second_of_day;

   second_of_day = c_time->hour*3600 +
         c_time->minute*60 +
         c_time->second;
   second_of_day = second_of_day + adjustment;
   if (second_of_day < 0) {
      /* subtract a day */
      c_date->day--;
      adjust_date(c_date);
      second_of_day = second_of_day + 86400;
   } else if (second_of_day > 86399) {
      /* add a day */
      c_date->day++;
      adjust_date(c_date);
      second_of_day = second_of_day - 86400;
   }
   c_time->hour = second_of_day/3600;
   second_of_day = second_of_day - (c_time->hour * 3600);
   c_time->minute = second_of_day/60;
   c_time->second = second_of_day - (c_time->minute * 60);
}


int days_in_month(int month,int year) {

   int leap_year;
   int days;

   switch(month)
   {
      case 1:
      case 3:
      case 5:
      case 7:
      case 8:
      case 10:
      case 12:
         days = 31;
         break;
      case 4:
      case 6:
      case 9:
      case 11:
         days = 30;
         break;
      case 2:
         leap_year = FALSE;
         if ((year % 4) == 0) {
            if ((year % 100)== 0) {
               if ((year % 400)==0) {
                  /* div by 4 and by 100  but
                     is div by 400 */
                  leap_year = TRUE;
               } else {
               }
            } else {
               /* div by 4 and not div by 100 */
               leap_year = TRUE;
            }
         }
         if (leap_year) {
            days = 29;
         } else {
            days = 28;
         }
         break;
      default:
         fprintf(stderr,
            "Error:   Bad month number given:  %d\n",
            month);
         days = -1;
         break;
   }
   return(days);

}

int months( char *mon ) {

   int mm = 0;

   if( ! strncmp( mon, "Jan", 3 )) mm = 1 ;
   if( ! strncmp( mon, "Feb", 3 )) mm = 2 ;
   if( ! strncmp( mon, "Mar", 3 )) mm = 3 ;
   if( ! strncmp( mon, "Apr", 3 )) mm = 4 ;
   if( ! strncmp( mon, "May", 3 )) mm = 5 ;
   if( ! strncmp( mon, "Jun", 3 )) mm = 6 ;
   if( ! strncmp( mon, "Jul", 3 )) mm = 7 ;
   if( ! strncmp( mon, "Aug", 3 )) mm = 8 ;
   if( ! strncmp( mon, "Sep", 3 )) mm = 9 ;
   if( ! strncmp( mon, "Oct", 3 )) mm = 10 ;
   if( ! strncmp( mon, "Nov", 3 )) mm = 11 ;
   if( ! strncmp( mon, "Dec", 3 )) mm = 12 ;


   return( mm );
}
