// opus.c
// read opus files

#include<ctype.h>
#include<stdlib.h>
#include<stdio.h>
#include<math.h>
#include<string.h>
#include<errno.h>

#include"opus.h"
#include"bnr.h"


void report(char *message,char *message2,char *filename) {
     fprintf(stderr,"%s\n\t%s\n\tIn %s\n",message,message2,filename);
}


int timestamp( struct opus_param_rcd *pblock_ptr, char *datestr, char *timestr ) {

   int i;
   struct opus_param_rcd rcd;

   if( !param_block_find( pblock_ptr, "DAT", &rcd )) {
      sprintf( opus_error, "*Could not find DAT in block" );
      return (0);
   } else {
      if( rcd.item_type != STRING ) {
         sprintf( opus_error, "DAT not string type" );
         return (0);
      } else {
         strncpy( datestr, rcd.value.string, DATE_LEN );
         // force end of string, in case string longer
         datestr[DATE_LEN-1] = '\0';
         for( i=0; i<DATE_LEN; i++) if( datestr[i] == ' ' ) datestr[i] = '0';
      }
   }

   if( !param_block_find( pblock_ptr, "TIM", &rcd )) {
      sprintf( opus_error, "*Could not find TIM in block" );
      return (0);
   } else {
      if( rcd.item_type != STRING ) {
         sprintf( opus_error, "TIM not string type" );
         return (0);
      } else {
         strncpy( timestr, rcd.value.string, TIME_LEN );
         // force end of string, in case string longer
         timestr[TIME_LEN-1] = '\0';
         for( i=0; i<TIME_LEN; i++) if( timestr[i] == ' ' ) timestr[i] = '0';
      }
   }

   return(1);
}

void block_info( int blocktype, int *norecords ) {

   char line[100];
   int print;

   *norecords = 0;

   //printf("Info Block Type = %X hex (%d) \n",blocktype,blocktype);

   //Section 1
   line[0]='\0';
   print = 1;
   switch (blocktype & 3) {
   case 1:
       strcpy(line,"Real part of Complex data");
       break;
   case 2:
       strcpy(line,"Imaginary part of complex data");
       break;
   case 3:
       strcpy(line,"Amplitude data");
       break;
   default:
       print = 0;
   }
   if( print ) printf("      %s\n",line);

   //Section 2
   line[0]='\0';
   print = 1;
   switch ((blocktype & 0xC) >>2) {
   case 1:
       strcpy(line,"Sample data");
       *norecords=1;
       break;
   case 2:
       strcpy(line,"Reference data");
       *norecords=1;
       break;
   case 3:
       strcpy(line,"Ratioed data");
       *norecords=1;
       break;
   default:
       print = 0;
   }
   if( print) printf("      %s\n",line);

   //Section 3
   line[0]='\0';
   print = 1;
   switch ((blocktype & 0x3f0) >>4) {
   case 1:
       strcpy(line,"Data status parameters");
       *norecords=0;
       break;
   case 2:
       strcpy(line,"Instrument status parameters");
       break;
   case 3:
       strcpy(line,"Acquisition parameters");
       break;
   case 4:
       strcpy(line,"FT-parameters");
       break;
   case 5:
       strcpy(line,"Plot and display parameters");
       break;
   case 6:
       strcpy(line,"Optic parameters");
       break;
   case 7:
       strcpy(line,"GC-parameters");
       break;
   case 8:
       strcpy(line,"Library search parameters");
       break;
   case 9:
       strcpy(line,"Communication parameters");
       break;
   case 10:
       strcpy(line,"Sample origin parameters");
       break;
   default:
       print = 0;
   }
   if( print ) printf("      %s\n",line);

   //Section 4
   line[0]='\0';
   print = 1;
   switch ((blocktype & 0x1FC00) >>10) {
   case 1:
       strcpy(line,"Spectrum, undefined y-units");
       break;
   case 2:
       strcpy(line,"Interferogram");
       break;
   case 3:
       strcpy(line,"Phase spectrum");
       break;
   case 4:
       strcpy(line,"absorbance spectrum(peaks upwards)");
       break;
   case 5:
       strcpy(line,"transmittance spectrum (peaks downward)");
       break;
   case 6:
       strcpy(line,"kubelka-munck spectrum (peaks upwards)");
       break;
   case 7:
       strcpy(line,"trace (intensity over time)(peaks upwards)");
       break;
   case 8:
       strcpy(line,"gc file, series of interferograms");
       break;
   case 9:
       strcpy(line,"gc file, series of spectra");
       break;
   case 10:
       strcpy(line,"raman spectrum (peaks upwards)");
       break;
   case 11:
       strcpy(line,"emission spectrum (peaks upward)");
       break;
   case 12:
       strcpy(line,"reflectance spectrum (peaks downwards)");
       break;
   case 13:
       strcpy(line,"Directory block");
       *norecords=1;
       break;
   case 14:
       strcpy(line,"power spectrum (form phase calculation)");
       break;
   case 15:
       strcpy(line,"log reflectance (like absorbance) (peaks upwards)");
       break;
   case 16:
       strcpy(line,"ATR-spectrum (peaks downwards)");
       break;
   case 17:
       strcpy(line,"photoacoustic spectrum (peaks upwards)");
       break;
   case 18:
       strcpy(line,"result of arithmetics, looks like TR (peaks downward)");
       break;
   case 19:
       strcpy(line,"results of arithmetics, look like AB (peaks upward) ");
       break;
   default:
       print = 0;
   }
   if( print) printf("      %s\n",line);

   //Section 5
   line[0]='\0';
   print = 1;
   switch ((blocktype & 0x60000) >>17) {
   case 1:
       strcpy(line,"first derivative");
       break;
   case 2:
       strcpy(line,"second derivative");
       break;
   case 3:
       strcpy(line,"n-th derivative");
       break;
   default:
       print = 0;
   }
   if( print ) printf("      %s\n",line);

   //Section 6
   line[0]='\0';
   print = 1;
   switch ((blocktype & 0x380000) >>19) {
   case 1:
       strcpy(line,"compound information");
       break;
   case 2:
       strcpy(line,"peak table");
       break;
   case 3:
       strcpy(line,"molecular structure");
       break;
   case 4:
       strcpy(line,"macro");
       *norecords=1;
       break;
   case 5:
       strcpy(line,"log of all actions which change data");
       *norecords=1;
       break;
   default:
       print = 0;
   }

   if( print ) printf("      %s\n",line);

}


int opus_init(FILE **opus, char *filename, int SWAP) {

   int magic = 0;

   //printf("filename= %s \n",filename);
   //return((*opus = fopen( filename, "rb" )) != (FILE *) NULL);

   *opus = fopen( filename, "rb" );
   //printf( "opus pointer : %x\n", *opus );
   if( *opus == (FILE *) NULL) return(1);

   if( !fread( &magic, sizeof( int ), 1, *opus )) return(3);

   if( SWAP ) swapem( (char *) &magic, 4 );

   //printf( "opus pointer : %i\n", magic );
   if( magic != -16905718 ) return(3);

   return(0);

}


int get_dirblk(FILE *opus,int *blocktype,int *length, int *pointer) {
     int i;
     int firstblock;

     fseek( opus, 12L, SEEK_SET ) ;
     if (!fread( &firstblock, sizeof( int ), 1, opus )) {return(0); }
     fseek( opus, firstblock, SEEK_SET ) ;
     if (!fread(blocktype, sizeof( int ), 1, opus)) {return(0);}
     if (!fread(length, sizeof( int ), 1, opus)) {return(0);}
     if (!fread(pointer, sizeof( int ), 1, opus)) {return(0);}
     return(1);
}


int get_firstblk(FILE *opus,int *blocktype,int *length,
                  int *pointer, int *num_blocks)
{
     int i;
     int firstblock;

     fseek( opus, 12L, SEEK_SET ) ;
     if (!fread( &firstblock, sizeof( int ), 1, opus )) {return(0); }
     fseek( opus, firstblock-4, SEEK_SET ) ;
     if (!fread( num_blocks, sizeof( int ), 1, opus )) {return(0); }
     fseek( opus, firstblock+12, SEEK_SET ) ;
     if (!fread(blocktype, sizeof( int ), 1, opus)) {return(0);}
     if (!fread(length, sizeof( int ), 1, opus)) {return(0);}
     if (!fread(pointer, sizeof( int ), 1, opus)) {return(0);}
     dir_index = 0;
     return(1);
}

int get_nextblk( FILE *opus, int *blocktype, int *length, int *pointer ) {

     int num_blocks, i, firstblock;

     fseek( opus, 12L, SEEK_SET ) ;
     if (!fread( &firstblock, sizeof( int ), 1, opus )) {return(0); }
     fseek( opus, firstblock-4, SEEK_SET ) ;
     if (!fread( &num_blocks, sizeof( int ), 1, opus )) {return(0); }
     if (dir_index < 0 || dir_index >= num_blocks-2) {
          return(0);
     }
     fseek( opus, firstblock+12, SEEK_SET ) ;

     for (i=0; i<num_blocks && i < (dir_index+2) ; i++) {
          if (!fread(blocktype, sizeof( int ), 1, opus)) {return(0);}
          if (!fread(length, sizeof( int ), 1, opus)) {return(0);}
          if (!fread(pointer, sizeof( int ), 1, opus)) {return(0);}

     }
     dir_index = i-1;
     //  fprintf(stderr,"----using %d %d %d %d\r\n",dir_index,*blocktype,*length,*pointer);
     return(1);
}


int opus_searchdir( FILE *opus, int blocktype, int *length, int *pointer, int SWAP )
{
     int num_blocks,i,blocktype_read;
     int firstblock;

     fseek( opus, 12L, SEEK_SET ) ;
     if (!fread( &firstblock, sizeof( int ), 1, opus )) {return(0); }
     if( SWAP ) swapem( (char *) &firstblock, 4 );

 //fprintf(stderr, "firstblock 0x%x\n", firstblock);
 //fprintf(stderr, "blocktype 0x%x\n", blocktype);

     fseek( opus, firstblock-4, SEEK_SET ) ;
     if (!fread( &num_blocks, sizeof( int ), 1, opus )) {return(0); }
     if( SWAP ) swapem( (char *) &num_blocks, 4 );

 //fprintf( stderr,  "num_blocks %i\n", num_blocks);

     fseek( opus, firstblock+12, SEEK_SET ) ;
     blocktype_read = -1000;
     for (i=0; i<num_blocks && blocktype_read != blocktype ; i++) {
          if (!fread(&blocktype_read, sizeof( int ), 1, opus)) {return(0);}
          if( SWAP ) swapem( (char *) &blocktype_read, 4 );
 //fprintf( stderr, "blocktype_read 0x%x\n", blocktype_read);
          if (!fread(length, sizeof( int ), 1, opus)) {return(0);}
          if (!fread(pointer, sizeof( int ), 1, opus)) {return(0);}
     }

     if( SWAP ) swapem( (char *) length, 4 );
     if( SWAP ) swapem( (char *) pointer, 4 );

 //fprintf( stderr, "length, pointer %i  0x%x\n", *length, *pointer);
 //fprintf( stderr, "block type, found 0x%x 0x%x\n", blocktype, blocktype_read);

     return(blocktype_read==blocktype);
}


// returns a spectrum located at pointer with increasing wavenumber
int opus_read_spectrum( FILE *opus, double *fstartw, double *fstopw, int length, int pointer,
                        double scalefactor, int fnum_points, float ydata[], int SWAP ) {

   long lpointer;
   int i;
   float *yptr, *top, *bot, temp, w;

   lpointer = (long int) pointer;
   fseek( opus, lpointer, SEEK_SET );
   yptr = &ydata[0];
   fread( yptr, sizeof(float), length, opus );

   yptr = &ydata[0];
   if( SWAP )
      for (i=0; i<length; i++, yptr++) {
         //fread( yptr, sizeof( float ), 1, opus );
         swapem( (char *) yptr, 4 );
         *yptr = *yptr*scalefactor;
      }
   else
      for (i=0; i<length; i++, yptr++) {
         *yptr = *yptr*scalefactor;
      }

   //printf(" start stop = %f %f\n",*fstartw, *fstopw);

   // reverse points so forward is opus has reverse scan
   if( *fstopw < *fstartw ) {
      fprintf( stderr, "Reversing order of data.\n" );
      top = ydata;
      bot = &ydata[fnum_points-1];
      //printf("top,bot = %f %f\n",*ydata, ydata[fnum_points-1]);
      //printf("top,bot = %x %x\n",ydata, &ydata[fnum_points-1]);
      do {
         temp = *top;
         *top = *bot;
         *bot = temp;
         top++;
         bot--;
      // repeat until top & bot meet or pass
      } while( bot > top );

      // reverse wavenumbers too
      w = *fstartw;
      *fstartw = *fstopw;
      *fstopw = w;
   }

   return 0;
}


int read_next_item( FILE *opus, char *item_label, int *item_type, int *item_length, int SWAP ) {

   short int i_type, i_length;

   if( fread( item_label, sizeof( char ), ILABELSZ, opus )!= ILABELSZ) return(0);
   if( !fread( &i_type, sizeof( short int ), 1, opus )) return(0);
   if( SWAP ) swapem( (char *) &i_type, 2 );
   if( !fread( &i_length, sizeof( short int ), 1, opus )) return(0);
   if( SWAP ) swapem( (char *) &i_length, 2 );
   *item_type = i_type;
   *item_length = i_length;

   return(1);

}


int read_first_item( FILE *opus, int pointer, char *item_label, int *item_type,int *item_length, int SWAP) {

   long lpointer = pointer;

   fseek( opus, lpointer, SEEK_SET );

   return(read_next_item(opus,item_label,item_type,item_length,SWAP));

}


int read_string( FILE *opus, int length, char *string ) {

   return(fread( string, sizeof( char ), length, opus )==length);

}


int read_int32( FILE *opus, int *num, int SWAP ) {

   int rc;

   rc = fread( num, sizeof( int ), 1, opus );
   if( SWAP ) swapem( (char *) num, 4 );
   return(rc);

}


int read_real64( FILE *opus, double *num, int SWAP ) {

    int rc;
    rc = fread( num, sizeof( double ), 1, opus );
	 if( SWAP ) swapem( (char *) num, 8 );
	 return(rc);

}


int param_block_init(struct opus_param_rcd **pb, FILE *opus, int blocktype, int SWAP) {

     int length, pointer, i, itype, i_type,i_length, readok, done;
     char i_label[ILABELSZ];
     struct opus_param_rcd *lpb, *prev;

     *pb = PARAM_RCD_NULL;
     if (!opus_searchdir( opus, blocktype, &length, &pointer, SWAP )) {
          sprintf(opus_error,"*Could not find blocktype %X",blocktype);
          return(0);
     }

     // length is in 4 byte words
     length = length*4;
     done = FALSE;
     for (i=length; i>0 && !done; i=i-(i_length*2)-8) {
          /* i_length is 2 byte word count.
             label, type & length takes 8 bytes */
          if (i==length) {
               readok = read_first_item( opus, pointer, i_label, &i_type, &i_length, SWAP);
          } else {
               readok = read_next_item( opus, i_label, &i_type, &i_length, SWAP);
          }
          if (!readok) {
               sprintf(opus_error,"*Could not read item in block %X", blocktype);
               return(0);
          }
/*
  printf("i = %d length %d\n",i,length);
  printf("label %s  type %d  length %d\n",i_label,i_type,i_length);
*/
          lpb = (struct opus_param_rcd *) malloc ( sizeof(struct opus_param_rcd ));
          if (lpb == (struct opus_param_rcd *) 0) {
               sprintf(opus_error, "*Could not allocate memory to store block %X", blocktype);
               return(0);
          }

          strncpy( lpb->item_label, i_label, ILABELSZ );
          lpb->item_type = i_type;
          lpb->item_length = i_length;

          if( *pb == PARAM_RCD_NULL ) *pb = lpb;
          else prev->next = lpb;
          prev = lpb;
          readok = TRUE;

          switch (i_type) {

          case INT32:
               readok=read_int32(opus,&(lpb->value.int32), SWAP);
               //printf("int32 = %ld\n",lpb->value.int32);
               break;

          case REAL64:
               readok=read_real64(opus,&(lpb->value.real64), SWAP);
               //printf("real64 = %lf\n",lpb->value.real64);
               break;

          case STRING:
          case ENUM_CHAR:

               // force ENUM_CHAR to STRING
               lpb->item_type = STRING;

               lpb->value.string = malloc (i_length*2);
               if (lpb->value.string == 0) {
                    sprintf(opus_error,"*Could not allocate string");
                    return(0);
               }
               readok=read_string( opus, i_length*2, lpb->value.string );
               //printf("string = %s\n",lpb->value.string);
               break;

          case ENUM:
               lpb->value.enumeration = malloc (i_length*2);
               if (lpb->value.enumeration == 0) {
                    sprintf(opus_error, "*Could not allocate enumeration");
                    return(0);
               }
               readok=read_string( opus, i_length*2, lpb->value.enumeration );
               //printf("enum = %s\n",lpb->value.enumeration);
               break;

          case END_BLOCK:
               //printf("Ignored end block type\n");
               break;

          default:
               fprintf(stderr,"Unexpected item type %d ignored\n", i_type);
               fprintf(stderr,"in blocktype %x HEX at i=%d\n", blocktype, i);
               break;
          }

          if (!readok && strcmp(i_label,"END")!=0) {
               sprintf(opus_error,"*Could not read value of %s",i_label);
               return(0);
          }

          lpb->next = PARAM_RCD_NULL;
          if (strcmp(lpb->item_label,"END")==0) done = TRUE;

     }
     return(1);
}

void param_block_done(struct opus_param_rcd **pb)
{
     struct opus_param_rcd *ptr, *next;
     if (*pb == PARAM_RCD_NULL) { return;}
     for (next = *pb,ptr = *pb; next != PARAM_RCD_NULL ; )
     {
/*		 printf("label %s  type %d  length %d\n",ptr->item_label,
       ptr->item_type,ptr->item_length);  */
          if (ptr->item_type == STRING) {
               free(ptr->value.string);
/*			printf("released value string\n"); */
          }
          if (ptr->item_type == ENUM) {
               free(ptr->value.enumeration);
/*			printf("released value enumeration\n"); */
          }
          next = ptr->next;
          free(ptr);
          ptr=next;
     }
     *pb = ptr;
}

int param_block_find(struct opus_param_rcd *pb, char *item_label, struct opus_param_rcd *item_record )
{
     struct opus_param_rcd *ptr;
     int found;

/*
	printf("first item %s\n",pb->item_label);
	printf("looking for %s \n",item_label);
*/
     for (ptr=pb,found=FALSE; ptr != PARAM_RCD_NULL && !found; ) {
/*		printf("looking at %s \n",ptr->item_label); */
          if (strncmp(ptr->item_label,item_label,strlen(item_label))==0) {
               *item_record = *ptr;
               found = TRUE;
          } else {
               ptr = ptr->next;
          }
     }
     return(found);
}



int param_block_dump( FILE *opus, int blocktype, int length, int pointer, int SWAP) {

   char i_label[ILABELSZ];
   char line[100];

   int i, itype, i_type,i_length, readok, done;
   int norecords=0;

   struct opus_param_rcd *lpb, *prev;

   done = FALSE;
   printf("=========================================================\n");
   printf("Block Type = %X hex (%d) \n", blocktype, blocktype);

   block_info( blocktype, &norecords );

   printf("   Length: %d 4-byte-words\n",length/4);
   printf("   Pointer: %X hex\n",pointer);
   printf("---------------------------------------------------------\n");
   if (norecords) return(1);
   for (i=length; i>0 && !done; i=i-(i_length*2)-8) {

       // i_length is 2 byte word count. label, type & length take 8 bytes
       if (i==length) {
            readok = read_first_item( opus, pointer, i_label, &i_type, &i_length, SWAP );
       } else {
            readok = read_next_item( opus, i_label, &i_type, &i_length, SWAP );
       }
       if (!readok) {
            sprintf(opus_error,"*Could not read item in block %X", blocktype);
            return(0);
       }
       printf("     %s",i_label);
/*
  printf("i = %d length %d\n",i,length);
  printf("label %s  type %d  length %d\n",i_label,i_type,i_length);
*/
          lpb = (struct opus_param_rcd *) malloc (
               sizeof(struct opus_param_rcd));
          if (lpb == (struct opus_param_rcd *) 0) {
               sprintf(opus_error,
                       "*Could not allocate memory to store block %X",
                       blocktype);
               return(0);
          }
          strncpy(lpb->item_label,i_label,ILABELSZ);
          lpb->item_type = i_type;
          lpb->item_length = i_length;
          readok = TRUE;
          switch (i_type) {
          case INT32:
               readok=read_int32(opus,&(lpb->value.int32),SWAP);
               printf("(INT32): %d\n",lpb->value.int32);
               break;
          case REAL64:
               readok=read_real64(opus,&(lpb->value.real64),SWAP);
               printf("(REAL64): %lf\n",lpb->value.real64);
               break;
          case STRING:
               lpb->value.string = malloc (i_length*2);
               if (lpb->value.string == 0) {
                    sprintf(opus_error,"*Could not allocate string");
                    return(0);
               }
               readok=read_string(opus,i_length*2,lpb->value.string);
               printf("(STRING): %s\n",lpb->value.string);
               break;
          case ENUM:
               lpb->item_type = ENUM; /* force to ENUM */
               lpb->value.enumeration = malloc (i_length*2);
               if (lpb->value.enumeration == 0) {
                    sprintf(opus_error,
                            "*Could not allocate enumeration");
                    return(0);
               }
               readok=read_string(opus,i_length*2,
                                       lpb->value.enumeration);
               printf("(ENUM): %s\n",lpb->value.enumeration);
               break;
          case ENUM_CHAR:
               lpb->item_type = STRING;
               lpb->value.enumeration = malloc (i_length*2);
               if (lpb->value.enumeration == 0) {
                    sprintf(opus_error,
                            "*Could not allocate enumeration");
                    return(0);
               }
               readok=read_string(opus,i_length*2,
                                       lpb->value.enumeration);
               printf("(ENUM_CHAR): %s\n",lpb->value.enumeration);
               break;
          case END_BLOCK:
/*			printf("Ignored end block type\n"); */
               printf("(END)\n");
               break;
          default:
               fprintf(stderr,"Unexpected item type %d ignored\n",i_type);
               fprintf(stderr,"in blocktype %x HEX at i=%d\n",blocktype,
                       i);
               break;
          }
          if (!readok && strcmp(i_label,"END")!=0) {
               sprintf(opus_error,"*Could not read value of %s",i_label);
               return(0);
          }
          lpb->next = PARAM_RCD_NULL;
          if (strcmp(lpb->item_label,"END")==0) {
               done = TRUE;
          }
          free(lpb);
     }
     return(1);
}


int read_AQPAR_block(FILE *opus, struct AQPB_rcd *AQPB, int SWAP) {

   struct opus_param_rcd *pblock_ptr, rcd;

   //	printf("Reading Acquisition Parameter block\n");

   if (!param_block_init(&pblock_ptr,opus, AQPARBLK, SWAP)) {
      if (!param_block_init(&pblock_ptr,opus, AQPARBLK + OFFSET, SWAP)) return(0);
   }

   AQPB->RES = 0.0;
   if( param_block_find( pblock_ptr, "RES", &rcd ))
      if( rcd.item_type == REAL64 )
         AQPB->RES = rcd.value.real64;

   AQPB->LFW = 0.0;
   if( param_block_find( pblock_ptr, "LFW", &rcd ))
      if( rcd.item_type == REAL64 )
         AQPB->LFW = rcd.value.real64;

   AQPB->HFW = 0.0;
   if( param_block_find( pblock_ptr, "HFW", &rcd ))
      if( rcd.item_type == REAL64 )
         AQPB->HFW = rcd.value.real64;

   AQPB->NSS = 0;
   if( param_block_find( pblock_ptr, "NSS", &rcd ))
      if( rcd.item_type == INT32 )
         AQPB->NSS = rcd.value.int32;

   memset( AQPB->SGN, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "SGN", &rcd ))
      if( rcd.item_type == ENUM )
         strncpy( AQPB->SGN, rcd.value.enumeration, ENUMSZ );

   memset( AQPB->RGN, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "RGN", &rcd ))
      if( rcd.item_type == ENUM )
         strncpy( AQPB->RGN, rcd.value.enumeration, ENUMSZ );

   AQPB->GSW = 0;
   if( param_block_find( pblock_ptr, "GSW", &rcd ))
      if( rcd.item_type == INT32 )
         AQPB->GSW = rcd.value.int32;

   memset( AQPB->GSG, '\0', ENUM_CHARSZ );
   if( param_block_find( pblock_ptr, "GSG", &rcd ))
      if( rcd.item_type == STRING )
         strncpy( AQPB->GSG, rcd.value.string, 2*rcd.item_length );
//printf("%d, %s, %s\n", rcd.item_length , AQPB->GSG, rcd.value.string );
   param_block_done( &pblock_ptr );

   return(1);
}


int read_FTPAR_block(FILE *opus, struct FTPB_rcd *FTPB, int SWAP) {

   struct opus_param_rcd *pblock_ptr, rcd;

   // printf("Reading FT Parameter block\n");

   if( !param_block_init(&pblock_ptr,opus, FTPARBLK, SWAP )) {
      if( !param_block_init(&pblock_ptr,opus, FTPARBLK + OFFSET, SWAP )) return(0);
   }
   memset( FTPB->APF, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "APF", &rcd ))
      if( rcd.item_type == ENUM )
         strncpy( FTPB->APF, rcd.value.enumeration, ENUMSZ );

   FTPB->HFQ = 0.0;
   if( param_block_find( pblock_ptr, "HFQ", &rcd ))
      if( rcd.item_type == REAL64 )
         FTPB->HFQ = rcd.value.real64;

   FTPB->LFQ = 0.0;
   if( param_block_find( pblock_ptr, "LFQ", &rcd ))
      if( rcd.item_type == REAL64 )
         FTPB->LFQ = rcd.value.real64;

   memset( FTPB->PHZ, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "PHZ", &rcd ))
      if( rcd.item_type == ENUM)
         strncpy( FTPB->PHZ, rcd.value.enumeration, ENUMSZ );

   FTPB->PHR = 0.0;
   if( param_block_find( pblock_ptr, "PHR", &rcd ))
      if( rcd.item_type == REAL64 )
         FTPB->PHR = rcd.value.real64;

   memset( FTPB->ZFF, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "ZFF", &rcd ))
      if( rcd.item_type == ENUM )
         strncpy( FTPB->ZFF, rcd.value.enumeration, ENUMSZ );

   param_block_done( &pblock_ptr );

   return(1);
}


int read_OPPAR_block( FILE *opus, struct OPPB_rcd *OPPB, int SWAP ) {

   struct opus_param_rcd *pblock_ptr, rcd;
   char aperaturestr[APERATSZ];
   char units[5];

   //printf("Reading Optic Parameter block\n");

   if ( !param_block_init( &pblock_ptr, opus, OPPARBLK, SWAP )) {
      if ( !param_block_init( &pblock_ptr, opus, OPPARBLK + OFFSET, SWAP )) return(0);
   }

   memset( OPPB->APT, '\0', ENUM_CHARSZ );
   if( param_block_find( pblock_ptr, "APT", &rcd ))
      if( rcd.item_type == STRING)
         strncpy( OPPB->APT, rcd.value.string, 2*rcd.item_length);

   memset( OPPB->BMS, '\0', ENUM_CHARSZ );
   if( param_block_find( pblock_ptr, "BMS", &rcd ))
      if( rcd.item_type == STRING)
         strncpy( OPPB->BMS, rcd.value.string, 2*rcd.item_length);

   memset( OPPB->DTC, '\0', ENUM_CHARSZ );
   if( param_block_find( pblock_ptr, "DTC", &rcd ))
      if( rcd.item_type == STRING)
         strncpy( OPPB->DTC, rcd.value.string, 2*rcd.item_length);

   memset( OPPB->OPF, '\0', ENUM_CHARSZ );
   if( param_block_find( pblock_ptr, "OPF", &rcd )) {
      if( rcd.item_type == STRING)
         strncpy( OPPB->OPF, rcd.value.string, 2*rcd.item_length);
      else
         strncpy( OPPB->OPF, "N/A", 3 );
   }

   memset( OPPB->OF1, '\0', ENUM_CHARSZ );
   if( param_block_find( pblock_ptr, "OF1", &rcd )) {
      if( rcd.item_type == STRING)
         strncpy( OPPB->OF1, rcd.value.string, 2*rcd.item_length);
      else
        strncpy( OPPB->OF1, "N/A", 3 );
   }

   memset( OPPB->PGN, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "PGN", &rcd ))
      if( rcd.item_type == ENUM)
         strncpy( OPPB->PGN, rcd.value.string, ENUMSZ);

   memset( OPPB->SRC, '\0', ENUM_CHARSZ );
   if( param_block_find( pblock_ptr, "SRC", &rcd ))
      if( rcd.item_type == STRING)
         strncpy( OPPB->SRC, rcd.value.string, 2*rcd.item_length);

   memset( OPPB->VEL, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "VEL", &rcd ))
      if( rcd.item_type == ENUM)
         strncpy( OPPB->VEL, rcd.value.string, ENUMSZ);

   memset( OPPB->HPF, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "HPF", &rcd ))
      if( rcd.item_type == ENUM)
         strncpy( OPPB->HPF, rcd.value.string, ENUMSZ);

   memset( OPPB->LPF, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "LPF", &rcd ))
      if( rcd.item_type == ENUM)
         strncpy( OPPB->LPF, rcd.value.string, ENUMSZ);

   param_block_done(&pblock_ptr);

   return(1);
}


int read_INPAR_block(FILE *opus, struct INPB_rcd *INPB, int SWAP) {

   struct opus_param_rcd *pblock_ptr, rcd;

   //printf("Reading INSTR block\n");

   if (!param_block_init(&pblock_ptr,opus, INPARBLK, SWAP)) {
      if (!param_block_init(&pblock_ptr,opus, INPARBLK + OFFSET, SWAP)) return(0);
   }

   INPB->HFL = 0.0;
   if( param_block_find( pblock_ptr, "HFL", &rcd ))
      if( rcd.item_type == REAL64 )
         INPB->HFL = rcd.value.real64;

   INPB->LFL = 0.0;
   if( param_block_find( pblock_ptr, "LFL", &rcd ))
      if( rcd.item_type == REAL64 )
         INPB->LFL = rcd.value.real64;

   INPB->ASS = 0;
   if( param_block_find( pblock_ptr, "ASS", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->ASS = rcd.value.int32;

   INPB->DUR = 0.0;
   if( param_block_find( pblock_ptr, "DUR", &rcd ))
      if( rcd.item_type == REAL64 )
         INPB->DUR = rcd.value.real64;

   INPB->RSN = 0;
   if( param_block_find( pblock_ptr, "RSN", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->RSN = rcd.value.int32;

   INPB->PKA = 0;
   if( param_block_find( pblock_ptr, "PKA", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->PKA = rcd.value.int32;

   INPB->PKL = 0;
   if( param_block_find( pblock_ptr, "PKL", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->PKL = rcd.value.int32;

   INPB->SSP = 0;
   if( param_block_find( pblock_ptr, "SSP", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->SSP = rcd.value.int32;

   memset( INPB->INS, '\0', STRSZ );
   if( param_block_find( pblock_ptr, "INS", &rcd ))
      if( rcd.item_type == STRING)
         strncpy( INPB->INS, rcd.value.string, 2*rcd.item_length );

   INPB->FOC = 0.0;
   if( param_block_find( pblock_ptr, "FOC", &rcd ))
      if( rcd.item_type == REAL64 )
         INPB->FOC = rcd.value.real64;

   INPB->ABP = 0;
   if( param_block_find( pblock_ptr, "ABP", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->ABP = rcd.value.int32;

   INPB->LWN = 0.0;
   if( param_block_find( pblock_ptr, "LWN", &rcd ))
      if( rcd.item_type == REAL64 )
         INPB->LWN = rcd.value.real64;

   memset( INPB->RDY, '\0', ENUMSZ );
   if( param_block_find( pblock_ptr, "RDY", &rcd ))
      if( rcd.item_type == ENUM)
         strncpy( INPB->RDY, rcd.value.enumeration, ENUMSZ );

   INPB->SSM = 0;
   if( param_block_find( pblock_ptr, "SSM", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->DFC = rcd.value.int32;

   INPB->SGP = 0;
   if( param_block_find( pblock_ptr, "SGP", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->SGP = rcd.value.int32;

   INPB->SGW = 0;
   if( param_block_find( pblock_ptr, "SGW", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->SGW = rcd.value.int32;

   INPB->DFC = 0;
   if( param_block_find( pblock_ptr, "DFC", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->DFC = rcd.value.int32;

   INPB->DFR = 0;
   if( param_block_find( pblock_ptr, "DFR", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->DFR = rcd.value.int32;

   INPB->HFF = 0.0;
   if( param_block_find( pblock_ptr, "HFF", &rcd ))
      if( rcd.item_type == REAL64 )
         INPB->HFF = rcd.value.real64;

   INPB->LFF = 0.0;
   if( param_block_find( pblock_ptr, "LFF", &rcd ))
      if( rcd.item_type == REAL64 )
         INPB->LFF = rcd.value.real64;

   INPB->GFW = 0;
   if( param_block_find( pblock_ptr, "GFW", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->GFW = rcd.value.int32;

   INPB->GBW = 0;
   if( param_block_find( pblock_ptr, "GBW", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->GBW = rcd.value.int32;

   INPB->PRA = 0;
   if( param_block_find( pblock_ptr, "PRA", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->PRA = rcd.value.int32;

   INPB->PRL = 0;
   if( param_block_find( pblock_ptr, "PRL", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->PRL = rcd.value.int32;

   INPB->ASG = 0;
   if( param_block_find( pblock_ptr, "ASG", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->ASG = rcd.value.int32;

   INPB->ARS = 0;
   if( param_block_find( pblock_ptr, "ARS", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->ARS = rcd.value.int32;

   INPB->END = 0;
   if( param_block_find( pblock_ptr, "END", &rcd ))
      if( rcd.item_type == INT32 )
         INPB->END = rcd.value.int32;

   param_block_done( &pblock_ptr );

   return(1);
}



int read_SPPAR_block( FILE *opus, int paramblocktype, struct SPPB_rcd *SPPB, int SWAP ) {

   int rc;
   struct opus_param_rcd *pblock_ptr, rcd;

   //fprintf( stderr,"Reading SP Parameter block\n");

   if( !param_block_init( &pblock_ptr, opus, paramblocktype, SWAP )) {
      //fprintf( stderr, "*Could not read SP PAR block... %4Xx\n", paramblocktype  );
      if( !param_block_init( &pblock_ptr, opus, paramblocktype + OFFSET, SWAP )) return(0);
   }

   SPPB->FXV = 0.0;
   if( param_block_find( pblock_ptr, "FXV", &rcd ))
      if( rcd.item_type == REAL64 )
        SPPB->FXV = rcd.value.real64;

   SPPB->LXV = 0.0;
   if( param_block_find( pblock_ptr, "LXV", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->LXV = rcd.value.real64;

   SPPB->MXY = 0.0;
   if( param_block_find( pblock_ptr, "MXY", &rcd ))
      if( rcd.item_type == REAL64 )
        SPPB->MXY = rcd.value.real64;

   SPPB->MNY = 0.0;
   if( param_block_find( pblock_ptr, "MNY", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->MNY = rcd.value.real64;

   SPPB->CSF = 0.0;
   if( param_block_find( pblock_ptr, "CSF", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->CSF = rcd.value.real64;

   SPPB->DPF = 0;
   if( param_block_find( pblock_ptr, "DPF", &rcd ))
      if( rcd.item_type == INT32 )
         SPPB->DPF = rcd.value.int32;

   SPPB->NPT = 0;
   if( param_block_find( pblock_ptr, "NPT", &rcd ))
      if( rcd.item_type == INT32 )
         SPPB->NPT = rcd.value.int32;

   strncpy( SPPB->DXU, "    ", ENUMSZ );
   if( param_block_find( pblock_ptr, "DXU", &rcd ))
      if( rcd.item_type == ENUM )
         strncpy( SPPB->DXU, rcd.value.enumeration, ENUMSZ );

   SPPB->SN1 = 0.0;
   if( param_block_find( pblock_ptr, "SN1", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->SN1 = rcd.value.real64;

   SPPB->SN2 = 0.0;
   if( param_block_find( pblock_ptr, "SN2", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->SN2 = rcd.value.real64;

   SPPB->SN3 = 0.0;
   if( param_block_find( pblock_ptr, "SN3", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->SN3 = rcd.value.real64;

   SPPB->SN4 = 0.0;
   if( param_block_find( pblock_ptr, "SN4", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->SN4 = rcd.value.real64;

   SPPB->NF1 = 0.0;
   if( param_block_find( pblock_ptr, "NF1", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->NF1 = rcd.value.real64;

   SPPB->NF2 = 0.0;
   if( param_block_find( pblock_ptr, "NF2", &rcd ))
      if( rcd.item_type == REAL64 )
         SPPB->NF2 = rcd.value.real64;

   rc = timestamp( pblock_ptr, SPPB->DAT, SPPB->TIM );

   param_block_done( &pblock_ptr );

   return(1);
}



int opus_listParams(char *rname, int SWAP) {

   FILE *opus;
   int blocktype, length=0, pointer=0, result=0, numblks=0, inp, rc=0;

   if(( rc = opus_init( &opus, rname, SWAP )) != FALSE ) return(rc);

   if( ! get_dirblk( opus, &blocktype, &length, &pointer )) {
      sprintf( opus_error, "*Could not get OPUS Dir block\n" );
      fclose( opus );
      return(0);
   }

   if( ! fwrite( &result, 1, sizeof(result), stdout )) {
      fprintf( stderr, "Cannot write good result code to pipe\n" );
      exit(1);
   }

   // length is in 4 byte words
   length = length*4;
   param_block_dump( opus, blocktype, length, pointer, SWAP );

   if( ! get_firstblk( opus, &blocktype, &length, &pointer, &numblks )) {
      sprintf( opus_error, "*Could not get first block of OPUS directory\n" );
      fclose( opus );
      return(0);
   }
   fprintf( stderr, "number of blocks : %d\n", numblks );

   do {
      length = length*4;
      param_block_dump( opus, blocktype, length, pointer, SWAP );

      //printf("Blocktype=%d Length=%d Pointer=%d\n",blocktype,length,pointer);
   }
   while (get_nextblk( opus, &blocktype, &length, &pointer ));

   printf( "&&&&\n" );
   fclose( opus );

   return(0);
}


int opus_listBlocks(char *rname, int SWAP) {

   FILE *opus;

   int blocktype,length, pointer, result, numblks, rc=0;
   int block_num=0, norecords;

   if(( rc = opus_init( &opus, rname, SWAP )) != FALSE ) return(rc);

   if( !get_dirblk( opus, &blocktype, &length, &pointer )) {
      sprintf( opus_error, "*Could not get OPUS Dir block\n" );
      fclose( opus );
      return(0);
   }

   result=0;
   if( !fwrite( &result, 1, sizeof(result), stdout )) {
      fprintf( stderr, "Cannot write good result code to pipe \n" );
      exit(1);
   }

   fprintf( stderr, "\rThe blocks in the file '%s' are:\n", rname );

   // length is in 4 byte words
   length = length * 4;
   ++block_num;
   //printf("Blocktype=%d Length=%d Pointer=%d\n",blocktype,length,pointer);
   //fprintf(stderr,"@%10d%10d\n", ++block_num, blocktype);

   fprintf(stderr,"\r%2d. Blocktype = %6d %4Xx Length = %d\n", block_num, blocktype, blocktype, length);
   block_info( blocktype, &norecords );
   //fprintf(stderr,"\n");

   if (!get_firstblk( opus, &blocktype, &length, &pointer, &numblks )) {
      sprintf( opus_error, "*Could not get first block of OPUS directory\n" );
      fclose( opus );
      return(0);
   }

   do {
      length = length*4; /* length is in 4 byte words */
      ++block_num;
      //printf("Blocktype=%d Length=%d Pointer=%d\n",blocktype,length,pointer);
      //fprintf(stderr,"@%10d%10d\n", ++block_num, blocktype);
      fprintf( stderr," \r%2d. Blocktype = %6d %4Xx Length = %d\n", block_num, blocktype, blocktype, length);
      block_info( blocktype, &norecords );
      //fprintf(stderr,"\n");
   }
   while( get_nextblk( opus, &blocktype, &length, &pointer ));

   printf( "&&&&\n" );

   //fprintf(stderr,"\rEOL\n\r");
   fclose( opus );

   return(0);

}

